/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.internal.type;

import java.io.Serializable;

import org.jbpm.pvm.PvmException;
import org.jbpm.pvm.internal.model.ExecutionImpl;

/**
 * is a jbpm-internal class that serves as a base class for classes 
 * that store variable values in the database.
 */
public abstract class Variable implements Serializable {

  private static final long serialVersionUID = 1L;
  
  protected long dbid = -1;
  protected int dbversion = 0;
  
  protected String key = null;
  protected Converter converter = null;
  protected String queryText = null;
  protected ExecutionImpl processInstance;
  
  // constructors /////////////////////////////////////////////////////////////
  
  // abstract methods /////////////////////////////////////////////////////////

  /**
   * is true if this variable-instance supports the given value, false otherwise.
   */
  public abstract boolean isStorable(Object value);
  /**
   * is the value, stored by this variable instance.
   */
  protected abstract Object getObject();
  /**
   * stores the value in this variable instance.
   */
  protected abstract void setObject(Object value);

  // variable management //////////////////////////////////////////////////////

  public boolean supports(Object value) {
    if (converter!=null) {
      return converter.supports(value);
    }
    return isStorable(value);
  }

  public void setValue(Object value) {
    if (converter!=null) {
      if (! converter.supports(value)) {
        throw new PvmException("the converter '"+converter.getClass().getName()+"' in variable instance '"+this.getClass().getName()+"' does not support values of type '"+value.getClass().getName()+"'.  to change the type of a variable, you have to delete it first");
      }
      value = converter.convert(value);
    }
    if ( (value!=null)
         && (! this.isStorable(value)) ) {
      throw new PvmException("variable instance '"+this.getClass().getName()+"' does not support values of type '"+value.getClass().getName()+"'.  to change the type of a variable, you have to delete it first");
    }
    setObject(value);
  }

  public Object getValue() {
    Object value = getObject();
    if ( (value!=null)
         && (converter!=null) ) {
      value = converter.revert(value);
    }
    return value;
  }
  
  // utility methods /////////////////////////////////////////////////////////

  public String toString() {
    return "${"+key+"}";
  }
  
  public Type getType() {
    Type type = new Type();
    type.setConverter(converter);
    type.setVariableClass(getClass());
    return type;
  }
  
  // getters and setters //////////////////////////////////////////////////////

  public String getKey() {
    return key;
  }
  public Converter getConverter() {
    return converter;
  }
  public void setConverter(Converter converter) {
    this.converter = converter;
  }
  public void setKey(String key) {
    this.key = key;
  }
  public ExecutionImpl getProcessInstance() {
    return processInstance;
  }
  public void setProcessInstance(ExecutionImpl processInstance) {
    this.processInstance = processInstance;
  }
}
