/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.listener;

import org.jbpm.pvm.Execution;
import org.jbpm.pvm.activity.Activity;
import org.jbpm.pvm.model.Event;
import org.jbpm.pvm.model.ObservableElement;
import org.jbpm.pvm.model.OpenExecution;
import org.jbpm.pvm.model.Transition;
import org.jbpm.pvm.processlog.ProcessLog;


/** view upon an {@link Execution path of execution} exposed to 
 * {@link EventListener} implementations.
 * 
 * @author Tom Baeyens
 */
public interface EventListenerExecution extends OpenExecution {
  
  // logs /////////////////////////////////////////////////////////////////////

  /** adds a <a href="package-summary.html#logs">log</a> to this execution. */
  void addLog(ProcessLog processLog);

  // extra state information methods //////////////////////////////////////////
  
  /** the current transition indicating the position in the process definition graph.
   * Can be null in case this execution is not taking a transition. */
  Transition getTransition();
  
  /** fires the event on the given eventSource and then propagates the event 
   * up to the eventSource's parent chain.  All the actions will see the given 
   * eventSource in {@link #getEventSource()}, event if the events are 
   * registered to parent's of the given eventSource. */
  void fire(String eventName, ObservableElement eventSource);

  /** the exception in case an exception handler is handling an exception. */
  Exception getException();

  /** the original object that fired the event, part of the current position in the 
   * process graph.  Can be null in case no event is being fired. This is mostly the 
   * object that is listened to with an {@link Activity}, but the eventSource can also 
   * be a child of the object to which is listened in case of event propagation. */
  ObservableElement getEventSource();
  
  /** the event that is being fired, part of the current position in the process 
   * graph.  Can be null in case no event is being fired. */
  Event getEvent();
  
  // extensions //////////////////////////////////////////////////////////////

  /** way to access process language extensions in the execution without 
   * having to cast.  Casting can be problematic for persistence. */
  <T> T getExtension(Class<T> extensionClass);
  
  
  /** setter for the priority.  The default priority is 0, which means 
   * NORMAL. Other recognized named priorities are HIGHEST (2), HIGH (1), 
   * LOW (-1) and LOWEST (-2). For the rest, the user can set any other 
   * priority integer value, but then, the UI will have to display it as 
   * an integer and not the named value.*/
  void setPriority(int priority);

}
