/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.processlog;

import java.text.SimpleDateFormat;
import java.util.List;

import org.jbpm.pvm.Execution;
import org.w3c.dom.Document;
import org.w3c.dom.Element;


/**
 * @author Tom Baeyens
 */
public class ProcessLogXmlSerializer {
  
  public static final SimpleDateFormat dateFormatter = new SimpleDateFormat("dd/MM/yyyy HH:mm:ss,SSS");

  protected List<ProcessLogProperty> properties;

  public void appendLog(Document document, Element parent, ProcessLog processLog) {
    String logType = getEventType(processLog);
    String time = getTime(processLog);
    String executionDbid = getExecutionDbid(processLog);
    String processInstanceDbid = getProcessInstanceDbid(processLog);
    String processInstanceKey = getProcessInstanceKey(processLog);
    
    Element logElement = document.createElement("log");
    parent.appendChild(logElement);
    
    logElement.setAttribute("type", logType);
    logElement.setAttribute("time", time);
    if (executionDbid!=null) {
      logElement.setAttribute("execution", executionDbid);
    }
    if (processInstanceDbid!=null) {
      logElement.setAttribute("instance", processInstanceDbid);
    }
    if (processInstanceKey!=null) {
      logElement.setAttribute("key", processInstanceKey);
    }
    
    appendProperties(document, logElement, processLog.getProperties());
  }

  protected void appendProperties(Document document, Element parent, List<ProcessLogProperty> properties) {
    if (properties!=null) {
      for (ProcessLogProperty property: properties) {
        Element propertyElement = document.createElement("property");
        parent.appendChild(propertyElement);
        
        if (property.name!=null) {
          propertyElement.setAttribute("name", property.name);
        }
        if (property.value!=null) {
          propertyElement.setAttribute("value", property.value);
        }
        appendProperties(document, propertyElement, property.getProperties());
      }
    }
  }

  protected String getEventType(ProcessLog processLog) {
    return processLog.getType();
  }

  protected String getTime(ProcessLog processLog) {
    return dateFormatter.format(processLog.getTime());
  }

  protected String getExecutionDbid(ProcessLog processLog) {
    Execution execution = processLog.getExecution();
    return (execution!=null ? Long.toString(execution.getDbid()) : null);
  }

  protected String getProcessInstanceDbid(ProcessLog processLog) {
    Execution processInstance = processLog.getProcessInstance();
    return (processInstance!=null ? Long.toString(processInstance.getDbid()) : null);
  }

  protected String getProcessInstanceKey(ProcessLog processLog) {
    Execution processInstance = processLog.getProcessInstance();
    return (processInstance!=null ? processInstance.getKey() : null);
  }
}
