/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.pvm.test.base;

import java.lang.reflect.Field;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.jbpm.pvm.Execution;
import org.jbpm.pvm.internal.model.ExecutionImpl;
import org.jbpm.pvm.internal.model.ProcessDefinitionImpl;
import org.jbpm.pvm.internal.util.ReflectUtil;
import org.jbpm.pvm.model.OpenProcessDefinition;
import org.jbpm.pvm.session.DbSession;
import org.jbpm.pvm.session.PvmDbSession;


/** for tests that use persistence inside environment blocks.
 * 
 * An environment is opened in the setUp and closed in the tearDown.
 * DB is cleaned inbetween tests. Extra convenience methods for usage 
 * inside an environment block are provided. 
 * 
 * @author Tom Baeyens
 */
public class EnvironmentDbTestCase extends EnvironmentTestCase {
  
  Transaction transaction = null;
  
  public EnvironmentDbTestCase() {
  }

  public EnvironmentDbTestCase(String configResource) {
    super(configResource);
  }

  public void setUp() throws Exception {
    if (isEnvironmentFactoryCached()) {
      Db.clean(getEnvironmentFactory());
    }
    super.setUp();
    beginTransaction();
  }
  
  public void tearDown() throws Exception {
    commitTransaction();
    super.tearDown();
  }

  void beginTransaction() {
    Session session = environment.get(Session.class);
    transaction = session.beginTransaction();
  }

  void commitTransaction() {
    transaction.commit();
    transaction = null;
  }
  
  void rollbackTransaction() {
    transaction.rollback();
    transaction = null;
  }
  
  public DbSession getDbSession() {
    return environment.get(DbSession.class);
  }
  
  public void rollbackAndBeginNewTransaction() {
    rollbackTransaction();
    closeEnvironment();
    openEnvironment();
    beginTransaction();
  }
  
  public void newTransaction() {
    try {
      commitTransaction();
      closeEnvironment();
    } finally {
      openEnvironment();
      beginTransaction();
    }
  }

  public void beginCacheTest() {
    SessionFactory sessionFactory = environment.get(SessionFactory.class);
    if (sessionFactory != null) {
      log.debug("=================================================================");
      log.debug("Beginning of the cache test, no more sql query should be performed before the end of the test");
      log.debug("=================================================================");

      sessionFactory.getStatistics().clear();
      sessionFactory.getStatistics().setStatisticsEnabled(true);
    }
  }
  
  public void endCacheTest() {
    SessionFactory sessionFactory = environment.get(SessionFactory.class);
    if (sessionFactory != null) {
      assertEquals(0, sessionFactory.getStatistics().getEntityLoadCount());
      assertEquals(0, sessionFactory.getStatistics().getCollectionLoadCount());
    }
  }
  
  public ProcessDefinitionImpl reload(OpenProcessDefinition processDefinition) {
    environment.get(PvmDbSession.class).save(processDefinition);
    newTransaction();
    return environment.get(PvmDbSession.class).get(ProcessDefinitionImpl.class, processDefinition.getDbid());
  }

  public <T> T reload(T object, Class<T> persistentClass) {
    environment.get(DbSession.class).save(object);
    newTransaction();
    
    Long dbid = null;
    try {
      Field dbidField = ReflectUtil.getField(persistentClass, "dbid");
      dbidField.setAccessible(true);
      dbid = (Long) dbidField.get(object);
    } catch (Exception e) {
      e.printStackTrace();
    }
    
    return environment.get(DbSession.class).get(persistentClass, dbid);
  }

  public ExecutionImpl reload(Execution execution) {
    environment.get(PvmDbSession.class).save(execution);
    newTransaction();
    return environment.get(PvmDbSession.class).get(ExecutionImpl.class, execution.getDbid());
  }

}
