/*
 * Decompiled with CFR 0.152.
 */
package org.jclouds.blobstore;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Supplier;
import com.google.common.base.Throwables;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.io.ByteStreams;
import com.google.common.io.InputSupplier;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import javax.annotation.Resource;
import javax.inject.Inject;
import javax.inject.Named;
import org.jclouds.blobstore.BlobStoreContext;
import org.jclouds.blobstore.ContainerNotFoundException;
import org.jclouds.blobstore.KeyNotFoundException;
import org.jclouds.blobstore.LocalStorageStrategy;
import org.jclouds.blobstore.domain.Blob;
import org.jclouds.blobstore.domain.BlobMetadata;
import org.jclouds.blobstore.domain.MutableBlobMetadata;
import org.jclouds.blobstore.domain.MutableStorageMetadata;
import org.jclouds.blobstore.domain.PageSet;
import org.jclouds.blobstore.domain.StorageMetadata;
import org.jclouds.blobstore.domain.StorageType;
import org.jclouds.blobstore.domain.internal.MutableStorageMetadataImpl;
import org.jclouds.blobstore.domain.internal.PageSetImpl;
import org.jclouds.blobstore.internal.BaseAsyncBlobStore;
import org.jclouds.blobstore.options.CreateContainerOptions;
import org.jclouds.blobstore.options.GetOptions;
import org.jclouds.blobstore.options.ListContainerOptions;
import org.jclouds.blobstore.options.PutOptions;
import org.jclouds.blobstore.strategy.IfDirectoryReturnNameStrategy;
import org.jclouds.blobstore.util.BlobStoreUtils;
import org.jclouds.blobstore.util.BlobUtils;
import org.jclouds.collect.Memoized;
import org.jclouds.domain.Location;
import org.jclouds.http.HttpCommand;
import org.jclouds.http.HttpRequest;
import org.jclouds.http.HttpResponse;
import org.jclouds.http.HttpResponseException;
import org.jclouds.http.HttpUtils;
import org.jclouds.io.ContentMetadata;
import org.jclouds.io.ContentMetadataCodec;
import org.jclouds.io.MutableContentMetadata;
import org.jclouds.io.Payload;
import org.jclouds.logging.Logger;

public class LocalAsyncBlobStore
extends BaseAsyncBlobStore {
    @Resource
    protected Logger logger = Logger.NULL;
    protected final ContentMetadataCodec contentMetadataCodec;
    protected final IfDirectoryReturnNameStrategy ifDirectoryReturnName;
    protected final Blob.Factory blobFactory;
    protected final LocalStorageStrategy storageStrategy;

    @Inject
    protected LocalAsyncBlobStore(BlobStoreContext context, BlobUtils blobUtils, @Named(value="jclouds.user-threads") ListeningExecutorService userExecutor, Supplier<Location> defaultLocation, @Memoized Supplier<Set<? extends Location>> locations, ContentMetadataCodec contentMetadataCodec, IfDirectoryReturnNameStrategy ifDirectoryReturnName, Blob.Factory blobFactory, LocalStorageStrategy storageStrategy) {
        super(context, blobUtils, userExecutor, defaultLocation, locations);
        this.blobFactory = blobFactory;
        this.contentMetadataCodec = contentMetadataCodec;
        this.ifDirectoryReturnName = ifDirectoryReturnName;
        this.storageStrategy = storageStrategy;
    }

    @Override
    public ListenableFuture<PageSet<? extends StorageMetadata>> list(final String container, ListContainerOptions options) {
        if (!this.storageStrategy.containerExists(container)) {
            return Futures.immediateFailedFuture((Throwable)((Object)this.cnfe(container)));
        }
        Iterable<String> blobBelongingToContainer = null;
        try {
            blobBelongingToContainer = this.storageStrategy.getBlobKeysInsideContainer(container);
        }
        catch (IOException e) {
            this.logger.error((Throwable)e, "An error occurred loading blobs contained into container %s", new Object[]{container});
            Throwables.propagate((Throwable)e);
        }
        SortedSet<Object> contents = Sets.newTreeSet((Iterable)Iterables.transform(blobBelongingToContainer, (Function)new Function<String, StorageMetadata>(){

            public StorageMetadata apply(String key) {
                Blob oldBlob = LocalAsyncBlobStore.this.loadBlob(container, key);
                Preconditions.checkState((oldBlob != null ? 1 : 0) != 0, (Object)("blob " + key + " is not present although it was in the list of " + container));
                Preconditions.checkState((oldBlob.getMetadata() != null ? 1 : 0) != 0, (Object)("blob " + container + "/" + key + " has no metadata"));
                MutableBlobMetadata md = BlobStoreUtils.copy(oldBlob.getMetadata());
                String directoryName = LocalAsyncBlobStore.this.ifDirectoryReturnName.execute(md);
                if (directoryName != null) {
                    md.setName(directoryName);
                    md.setType(StorageType.RELATIVE_PATH);
                }
                return md;
            }
        }));
        String marker = null;
        if (options != null) {
            int maxResults;
            String prefix;
            if (options.getMarker() != null) {
                final String finalMarker = options.getMarker();
                StorageMetadata lastMarkerMetadata = (StorageMetadata)Iterables.find((Iterable)contents, (Predicate)new Predicate<StorageMetadata>(){

                    public boolean apply(StorageMetadata metadata) {
                        return metadata.getName().compareTo(finalMarker) > 0;
                    }
                });
                contents = contents.tailSet(lastMarkerMetadata);
            }
            if ((prefix = options.getDir()) != null) {
                contents = Sets.newTreeSet((Iterable)Sets.filter((SortedSet)contents, (Predicate)new Predicate<StorageMetadata>(){

                    public boolean apply(StorageMetadata o) {
                        return o != null && o.getName().startsWith(prefix) && !o.getName().equals(prefix);
                    }
                }));
            }
            int n = maxResults = options.getMaxResults() != null ? options.getMaxResults() : 1000;
            if (!contents.isEmpty()) {
                StorageMetadata lastElement = (StorageMetadata)contents.last();
                if (!(contents = Sets.newTreeSet((Iterable)Iterables.limit((Iterable)contents, (int)maxResults))).contains(lastElement)) {
                    marker = ((StorageMetadata)contents.last()).getName();
                }
            }
            if (!options.isRecursive()) {
                String delimiter = this.storageStrategy.getSeparator();
                TreeSet treeSet = Sets.newTreeSet((Iterable)Iterables.transform(contents, (Function)new CommonPrefixes(prefix, delimiter)));
                treeSet.remove("NO_PREFIX");
                contents = Sets.newTreeSet((Iterable)Sets.filter(contents, (Predicate)new DelimiterFilter(prefix, delimiter)));
                for (String o : treeSet) {
                    MutableStorageMetadataImpl md = new MutableStorageMetadataImpl();
                    md.setType(StorageType.RELATIVE_PATH);
                    md.setName(o);
                    contents.add(md);
                }
            }
            if (!options.isDetailed()) {
                for (StorageMetadata storageMetadata : contents) {
                    storageMetadata.getUserMetadata().clear();
                }
            }
        }
        return Futures.immediateFuture(new PageSetImpl(contents, marker));
    }

    private ContainerNotFoundException cnfe(String name) {
        return new ContainerNotFoundException(name, String.format("container %s not in %s", name, this.storageStrategy.getAllContainerNames()));
    }

    @Override
    public ListenableFuture<Void> removeBlob(String container, String key) {
        this.storageStrategy.removeBlob(container, key);
        return Futures.immediateFuture(null);
    }

    @Override
    public ListenableFuture<Void> clearContainer(String container) {
        this.storageStrategy.clearContainer(container);
        return Futures.immediateFuture(null);
    }

    @Override
    public ListenableFuture<Void> deleteContainer(String container) {
        this.deleteAndVerifyContainerGone(container);
        return Futures.immediateFuture(null);
    }

    public ListenableFuture<Boolean> deleteContainerIfEmpty(String container) {
        Boolean returnVal = true;
        if (this.storageStrategy.containerExists(container)) {
            try {
                if (Iterables.isEmpty(this.storageStrategy.getBlobKeysInsideContainer(container))) {
                    this.storageStrategy.deleteContainer(container);
                } else {
                    returnVal = false;
                }
            }
            catch (IOException e) {
                this.logger.error((Throwable)e, "An error occurred loading blobs contained into container %s", new Object[]{container});
                Throwables.propagate((Throwable)e);
            }
        }
        return Futures.immediateFuture((Object)returnVal);
    }

    @Override
    public ListenableFuture<Boolean> containerExists(String containerName) {
        return Futures.immediateFuture((Object)this.storageStrategy.containerExists(containerName));
    }

    @Override
    public ListenableFuture<PageSet<? extends StorageMetadata>> list() {
        Iterable<String> containers = this.storageStrategy.getAllContainerNames();
        return Futures.immediateFuture(new PageSetImpl(Iterables.transform(containers, (Function)new Function<String, StorageMetadata>(){

            public StorageMetadata apply(String name) {
                MutableStorageMetadata cmd = LocalAsyncBlobStore.this.create();
                cmd.setName(name);
                cmd.setType(StorageType.CONTAINER);
                cmd.setLocation(LocalAsyncBlobStore.this.storageStrategy.getLocation(name));
                return cmd;
            }
        }), null));
    }

    protected MutableStorageMetadata create() {
        return new MutableStorageMetadataImpl();
    }

    @Override
    public ListenableFuture<Boolean> createContainerInLocation(Location location, String name) {
        boolean result = this.storageStrategy.createContainerInLocation(name, location);
        return Futures.immediateFuture((Object)result);
    }

    private Blob loadBlob(String container, String key) {
        this.logger.debug("Opening blob in container: %s - %s", new Object[]{container, key});
        return this.storageStrategy.getBlob(container, key);
    }

    public static HttpResponseException returnResponseException(int code) {
        HttpResponse response = HttpResponse.builder().statusCode(code).build();
        return new HttpResponseException(new HttpCommand(HttpRequest.builder().method("GET").endpoint("http://stub").build()), response);
    }

    @Override
    public ListenableFuture<String> putBlob(String containerName, Blob blob) {
        Preconditions.checkNotNull((Object)containerName, (Object)"containerName must be set");
        Preconditions.checkNotNull((Object)blob, (Object)"blob must be set");
        String blobKey = blob.getMetadata().getName();
        this.logger.debug("Put blob with key [%s] to container [%s]", new Object[]{blobKey, containerName});
        if (!this.storageStrategy.containerExists(containerName)) {
            return Futures.immediateFailedFuture((Throwable)new IllegalStateException("containerName not found: " + containerName));
        }
        try {
            return Futures.immediateFuture((Object)this.storageStrategy.putBlob(containerName, blob));
        }
        catch (IOException e) {
            this.logger.error((Throwable)e, "An error occurred storing the new blob with name [%s] to container [%s].", new Object[]{blobKey, containerName});
            throw Throwables.propagate((Throwable)e);
        }
    }

    private void copyPayloadHeadersToBlob(Payload payload, Blob blob) {
        blob.getAllHeaders().putAll(this.contentMetadataCodec.toHeaders((ContentMetadata)payload.getContentMetadata()));
    }

    @Override
    public ListenableFuture<Boolean> blobExists(String containerName, String key) {
        if (!this.storageStrategy.containerExists(containerName)) {
            return Futures.immediateFailedFuture((Throwable)((Object)this.cnfe(containerName)));
        }
        return Futures.immediateFuture((Object)this.storageStrategy.blobExists(containerName, key));
    }

    @Override
    public ListenableFuture<Blob> getBlob(String containerName, String key, GetOptions options) {
        this.logger.debug("Retrieving blob with key %s from container %s", new Object[]{key, containerName});
        if (!this.storageStrategy.containerExists(containerName)) {
            this.logger.debug("Container %s does not exist", new Object[]{containerName});
            return Futures.immediateFailedFuture((Throwable)((Object)this.cnfe(containerName)));
        }
        if (!this.storageStrategy.blobExists(containerName, key)) {
            this.logger.debug("Item %s does not exist in container %s", new Object[]{key, containerName});
            return Futures.immediateFuture(null);
        }
        Blob blob = this.loadBlob(containerName, key);
        if (options != null) {
            if (options.getIfMatch() != null && !blob.getMetadata().getETag().equals(options.getIfMatch())) {
                return Futures.immediateFailedFuture((Throwable)LocalAsyncBlobStore.returnResponseException(412));
            }
            if (options.getIfNoneMatch() != null && blob.getMetadata().getETag().equals(options.getIfNoneMatch())) {
                return Futures.immediateFailedFuture((Throwable)LocalAsyncBlobStore.returnResponseException(304));
            }
            if (options.getIfModifiedSince() != null) {
                Date modifiedSince = options.getIfModifiedSince();
                if (blob.getMetadata().getLastModified().before(modifiedSince)) {
                    HttpResponse response = HttpResponse.builder().statusCode(304).build();
                    return Futures.immediateFailedFuture((Throwable)new HttpResponseException(String.format("%1$s is before %2$s", blob.getMetadata().getLastModified(), modifiedSince), null, response));
                }
            }
            if (options.getIfUnmodifiedSince() != null) {
                Date unmodifiedSince = options.getIfUnmodifiedSince();
                if (blob.getMetadata().getLastModified().after(unmodifiedSince)) {
                    HttpResponse response = HttpResponse.builder().statusCode(412).build();
                    return Futures.immediateFailedFuture((Throwable)new HttpResponseException(String.format("%1$s is after %2$s", blob.getMetadata().getLastModified(), unmodifiedSince), null, response));
                }
            }
            blob = this.copyBlob(blob);
            if (options.getRanges() != null && options.getRanges().size() > 0) {
                byte[] data;
                try {
                    data = ByteStreams.toByteArray((InputSupplier)blob.getPayload());
                }
                catch (IOException e) {
                    return Futures.immediateFailedFuture((Throwable)new RuntimeException(e));
                }
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                for (String s : options.getRanges()) {
                    int offset = 0;
                    int last = data.length - 1;
                    if (s.startsWith("-")) {
                        offset = last - Integer.parseInt(s.substring(1)) + 1;
                    } else if (s.endsWith("-")) {
                        offset = Integer.parseInt(s.substring(0, s.length() - 1));
                    } else if (s.contains("-")) {
                        String[] firstLast = s.split("\\-");
                        offset = Integer.parseInt(firstLast[0]);
                        last = Integer.parseInt(firstLast[1]);
                    } else {
                        return Futures.immediateFailedFuture((Throwable)new IllegalArgumentException("illegal range: " + s));
                    }
                    if (offset > last) {
                        return Futures.immediateFailedFuture((Throwable)new IllegalArgumentException("illegal range: " + s));
                    }
                    if (last + 1 > data.length) {
                        last = data.length - 1;
                    }
                    out.write(data, offset, last - offset + 1);
                }
                MutableContentMetadata cmd = blob.getPayload().getContentMetadata();
                byte[] byteArray = out.toByteArray();
                blob.setPayload(byteArray);
                HttpUtils.copy((ContentMetadata)cmd, (MutableContentMetadata)blob.getPayload().getContentMetadata());
                blob.getPayload().getContentMetadata().setContentLength(Long.valueOf(byteArray.length));
            }
        }
        Preconditions.checkNotNull((Object)blob.getPayload(), (Object)("payload " + blob));
        return Futures.immediateFuture((Object)blob);
    }

    @Override
    public ListenableFuture<BlobMetadata> blobMetadata(String container, String key) {
        try {
            Blob blob = (Blob)this.getBlob(container, key).get();
            return Futures.immediateFuture(blob != null ? BlobStoreUtils.copy(blob.getMetadata()) : null);
        }
        catch (Exception e) {
            if (Iterables.size((Iterable)Iterables.filter((Iterable)Throwables.getCausalChain((Throwable)e), KeyNotFoundException.class)) >= 1) {
                return Futures.immediateFuture(null);
            }
            return Futures.immediateFailedFuture((Throwable)e);
        }
    }

    private Blob copyBlob(Blob blob) {
        Blob returnVal = this.blobFactory.create(BlobStoreUtils.copy(blob.getMetadata()));
        returnVal.setPayload(blob.getPayload());
        this.copyPayloadHeadersToBlob(blob.getPayload(), returnVal);
        return returnVal;
    }

    @Override
    protected boolean deleteAndVerifyContainerGone(String container) {
        this.storageStrategy.deleteContainer(container);
        return this.storageStrategy.containerExists(container);
    }

    @Override
    public ListenableFuture<String> putBlob(String container, Blob blob, PutOptions options) {
        return this.putBlob(container, blob);
    }

    @Override
    public ListenableFuture<Boolean> createContainerInLocation(Location location, String container, CreateContainerOptions options) {
        if (options.isPublicRead()) {
            throw new UnsupportedOperationException("publicRead");
        }
        return this.createContainerInLocation(location, container);
    }

    protected static class CommonPrefixes
    implements Function<StorageMetadata, String> {
        private final String prefix;
        private final String delimiter;
        public static final String NO_PREFIX = "NO_PREFIX";

        public CommonPrefixes(String prefix, String delimiter) {
            this.prefix = prefix;
            this.delimiter = delimiter;
        }

        public String apply(StorageMetadata metadata) {
            String working = metadata.getName();
            if (this.prefix != null) {
                String toMatch;
                String string = toMatch = this.prefix.endsWith("/") ? this.prefix : this.prefix + this.delimiter;
                if (working.startsWith(toMatch)) {
                    working = working.replaceFirst(toMatch, "");
                }
            }
            if (working.contains(this.delimiter)) {
                return working.substring(0, working.indexOf(this.delimiter));
            }
            return NO_PREFIX;
        }
    }

    protected static class DelimiterFilter
    implements Predicate<StorageMetadata> {
        private final String prefix;
        private final String delimiter;

        public DelimiterFilter(String prefix, String delimiter) {
            this.prefix = prefix;
            this.delimiter = delimiter;
        }

        public boolean apply(StorageMetadata metadata) {
            String toMatch;
            if (this.prefix == null) {
                return metadata.getName().indexOf(this.delimiter) == -1;
            }
            String string = toMatch = this.prefix.endsWith("/") ? this.prefix : this.prefix + this.delimiter;
            if (metadata.getName().startsWith(toMatch)) {
                String unprefixedName = metadata.getName().replaceFirst(toMatch, "");
                if (unprefixedName.equals("")) {
                    return false;
                }
                return unprefixedName.indexOf(this.delimiter) == -1;
            }
            return false;
        }
    }
}

