/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jdbi.v3.sqlobject.customizer.internal;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.time.Clock;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.util.function.Function;

import org.jdbi.v3.meta.Beta;
import org.jdbi.v3.sqlobject.customizer.SqlStatementCustomizer;
import org.jdbi.v3.sqlobject.customizer.SqlStatementCustomizerFactory;
import org.jdbi.v3.sqlobject.customizer.Timestamped;
import org.jdbi.v3.sqlobject.customizer.TimestampedConfig;

public class TimestampedFactory implements SqlStatementCustomizerFactory {
    private static Function<ZoneId, Clock> timeSource = Clock::system;

    @Override
    public SqlStatementCustomizer createForMethod(Annotation annotation, Class<?> sqlObjectType, Method method) {
        final String parameterName = ((Timestamped) annotation).value();

        return stmt -> {
            ZoneId zone = stmt.getConfig(TimestampedConfig.class).getTimezone();
            stmt.bind(parameterName, OffsetDateTime.now(timeSource.apply(zone)));
        };
    }

    /**
     * for testing purposes only
     */
    @Beta
    static void setTimeSource(Function<ZoneId, Clock> timeSource) {
        TimestampedFactory.timeSource = timeSource;
    }
}
