/*
 * The MIT License
 *
 * Copyright (c) 2004-2009, Sun Microsystems, Inc., Kohsuke Kawaguchi
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

package hudson.model;

import hudson.search.Search;
import hudson.search.SearchFactory;
import hudson.search.SearchIndex;
import hudson.search.SearchIndexBuilder;
import hudson.search.SearchableModelObject;
import java.io.IOException;
import javax.servlet.ServletException;
import org.kohsuke.stapler.Stapler;
import org.kohsuke.stapler.StaplerRequest;
import org.kohsuke.stapler.StaplerResponse;
import org.kohsuke.stapler.interceptor.RequirePOST;

/**
 * {@link ModelObject} with some convenience methods.
 *
 * @author Kohsuke Kawaguchi
 */
public abstract class AbstractModelObject implements SearchableModelObject {
    /**
     * Displays the error in a page.
     */
    protected final void sendError(Exception e, StaplerRequest req, StaplerResponse rsp) throws ServletException, IOException {
        req.setAttribute("exception", e);
        sendError(e.getMessage(), req, rsp);
    }

    protected final void sendError(Exception e) throws ServletException, IOException {
        sendError(e, Stapler.getCurrentRequest(), Stapler.getCurrentResponse());
    }

    protected final void sendError(String message, StaplerRequest req, StaplerResponse rsp) throws ServletException, IOException {
        req.setAttribute("message", message);
        rsp.forward(this, "error", req);
    }

    /**
     * @param pre
     *      If true, the message is put in a PRE tag.
     */
    protected final void sendError(String message, StaplerRequest req, StaplerResponse rsp, boolean pre) throws ServletException, IOException {
        req.setAttribute("message", message);
        if (pre)
            req.setAttribute("pre", true);
        rsp.forward(this, "error", req);
    }

    protected final void sendError(String message) throws ServletException, IOException {
        sendError(message, Stapler.getCurrentRequest(), Stapler.getCurrentResponse());
    }

    /**
     * Convenience method to verify that the current request is a POST request.
     *
     * @deprecated
     *      Use {@link RequirePOST} on your method.
     */
    @Deprecated
    protected final void requirePOST() throws ServletException {
        StaplerRequest req = Stapler.getCurrentRequest();
        if (req == null)  return; // invoked outside the context of servlet
        String method = req.getMethod();
        if (!method.equalsIgnoreCase("POST"))
            throw new ServletException("Must be POST, Can't be " + method);
    }

    /**
     * Default implementation that returns empty index.
     */
    protected SearchIndexBuilder makeSearchIndex() {
        return new SearchIndexBuilder().addAllAnnotations(this);
    }

    @Override
    public final SearchIndex getSearchIndex() {
        return makeSearchIndex().make();
    }

    @Override
    public Search getSearch() {
        for (SearchFactory sf : SearchFactory.all()) {
            Search s = sf.createFor(this);
            if (s != null)
                return s;
        }
        return new Search();
    }

    /**
     * Default implementation that returns the display name.
     */
    @Override
    public String getSearchName() {
        return getDisplayName();
    }
}
