/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.backend.wasm.utils

import org.jetbrains.kotlin.backend.wasm.WasmBackendContext
import org.jetbrains.kotlin.descriptors.Modality
import org.jetbrains.kotlin.ir.declarations.IrClass
import org.jetbrains.kotlin.ir.expressions.IrTry
import org.jetbrains.kotlin.ir.util.IdSignature
import org.jetbrains.kotlin.ir.util.defaultType
import org.jetbrains.kotlin.ir.util.isFunction
import org.jetbrains.kotlin.ir.util.isFunctionMarker
import org.jetbrains.kotlin.wasm.ir.WasmExpressionBuilder
import org.jetbrains.kotlin.wasm.ir.source.location.SourceLocation

// Backed codegen can only handle try/catch in the canonical form.
// The defined for Wasm backend canonical form of try/catch:
// try {
//   ...exprs
// }
// catch (e: Throwable) {
//   ...exprs
// }
// no-finally
internal fun IrTry.isCanonical(context: WasmBackendContext) =
    catches.size <= 1 &&
            catches.all { it.catchParameter.type == context.irBuiltIns.throwableType } &&
            finallyExpression == null

internal val IrClass.isAbstractOrSealed
    get() = modality == Modality.ABSTRACT || modality == Modality.SEALED

internal fun getFunctionalInterfaceSlot(iFace: IrClass): Int {
    check(iFace.symbol.isFunction())
    return if (iFace.defaultType.isFunctionMarker()) return 0 else iFace.typeParameters.size
}

internal val String.fitsLatin1
    get() = this.all { it.code in 0..255 }

internal fun WasmExpressionBuilder.buildUnreachableForVerifier() {
    buildUnreachable(SourceLocation.NoLocation("This instruction should never be reached, but required for wasm verifier"))
}

internal fun WasmExpressionBuilder.buildUnreachableAfterNothingType() {
    buildUnreachable(
        SourceLocation.NoLocation(
            "The unreachable instruction after an expression with Nothing type to make sure that " +
                    "execution doesn't come here (or it fails fast if so). It also might be required for wasm verifier."
        )
    )
}

internal fun redefinitionError(signature: IdSignature, info: String): Nothing {
    error("Symbol $signature cannot be redefined for $info")
}

internal val String.hasUnpairedSurrogates: Boolean
    get() = this.withIndex().any { (i, c) ->
        (c.isLowSurrogate() && (i == 0 || !this[i - 1].isHighSurrogate())) ||
        (c.isHighSurrogate() && (i == this.lastIndex || !this[i + 1].isLowSurrogate()))
    }