/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package org.jetbrains.kotlin.fir.declarations

import kotlinx.collections.immutable.PersistentList
import kotlinx.collections.immutable.persistentListOf
import kotlinx.collections.immutable.toPersistentList
import org.jetbrains.kotlin.fir.FirSession
import org.jetbrains.kotlin.fir.SessionAndScopeSessionHolder
import org.jetbrains.kotlin.fir.resolve.*
import org.jetbrains.kotlin.fir.resolve.calls.*
import org.jetbrains.kotlin.fir.scopes.FirContainingNamesAwareScope
import org.jetbrains.kotlin.fir.scopes.FirScope
import org.jetbrains.kotlin.fir.scopes.FirTypeScope
import org.jetbrains.kotlin.fir.scopes.impl.FirLocalScope
import org.jetbrains.kotlin.fir.scopes.impl.wrapNestedClassifierScopeWithSubstitutionForSuperType
import org.jetbrains.kotlin.fir.symbols.FirBasedSymbol
import org.jetbrains.kotlin.fir.symbols.impl.FirClassSymbol
import org.jetbrains.kotlin.fir.symbols.impl.FirRegularClassSymbol
import org.jetbrains.kotlin.fir.types.ConeErrorType
import org.jetbrains.kotlin.fir.types.ConeKotlinType
import org.jetbrains.kotlin.fir.types.ConeStubType
import org.jetbrains.kotlin.name.Name
import org.jetbrains.kotlin.utils.addIfNotNull

fun SessionAndScopeSessionHolder.collectTowerDataElementsForClass(owner: FirClass, defaultType: ConeKotlinType): TowerElementsForClass {
    val allImplicitCompanionValues = mutableListOf<ImplicitReceiverValue<*>>()

    val companionObject = (owner as? FirRegularClass)?.companionObjectSymbol?.fir
    val companionReceiver = companionObject?.let { companion ->
        ImplicitDispatchReceiverValue(
            companion.symbol, useSiteSession = session, scopeSession = scopeSession
        )
    }
    allImplicitCompanionValues.addIfNotNull(companionReceiver)

    val superClassesStaticsAndCompanionReceivers = mutableListOf<FirTowerDataElement>()
    for (superType in lookupSuperTypes(owner, lookupInterfaces = false, deep = true, useSiteSession = session, substituteTypes = true)) {
        val expandedType = superType.fullyExpandedType()
        val superClass = expandedType.lookupTag.toRegularClassSymbol()?.fir ?: continue

        superClass.staticScope(this)
            ?.wrapNestedClassifierScopeWithSubstitutionForSuperType(expandedType, session)
            ?.asTowerDataElementForStaticScope(staticScopeOwnerSymbol = superClass.symbol)
            ?.let(superClassesStaticsAndCompanionReceivers::add)

        superClass.companionObjectSymbol?.let {
            val superCompanionReceiver = ImplicitDispatchReceiverValue(
                it, useSiteSession = session, scopeSession = scopeSession
            )

            superClassesStaticsAndCompanionReceivers += superCompanionReceiver.asTowerDataElement()
            allImplicitCompanionValues += superCompanionReceiver
        }
    }

    val thisReceiver = ImplicitDispatchReceiverValue(owner.symbol, defaultType, session, scopeSession)

    return TowerElementsForClass(
        thisReceiver,
        owner.staticScope(this),
        companionReceiver,
        companionObject?.staticScope(this),
        superClassesStaticsAndCompanionReceivers.asReversed(),
    )
}

class TowerElementsForClass(
    val thisReceiver: ImplicitReceiverValue<*>,
    val staticScope: FirScope?,
    val companionReceiver: ImplicitReceiverValue<*>?,
    val companionStaticScope: FirScope?,
    // Ordered from inner scopes to outer scopes.
    val superClassesStaticsAndCompanionReceivers: List<FirTowerDataElement>,
)

@ConsistentCopyVisibility
data class FirTowerDataContext private constructor(
    val towerDataElements: PersistentList<FirTowerDataElement>,
    // These properties are effectively redundant, their content should be consistent with `towerDataElements`,
    // i.e. implicitReceiverStack == towerDataElements.mapNotNull { it.receiver }
    // i.e. localScopes == towerDataElements.mapNotNull { it.scope?.takeIf { it.isLocal } }
    val implicitValueStorage: ImplicitValueStorage,
    val classesUnderInitialization: PersistentList<FirClassSymbol<*>>,
    val localScopes: FirLocalScopes,
    val nonLocalTowerDataElements: PersistentList<FirTowerDataElement>,
    val localVariableScopeStorage: LocalVariableScopeStorage,
) {

    constructor() : this(
        persistentListOf(),
        ImplicitValueStorage(),
        persistentListOf(),
        persistentListOf(),
        persistentListOf(),
        LocalVariableScopeStorage(),
    )

    fun addLocalVariable(variable: FirVariable, session: FirSession): FirTowerDataContext {
        val oldLastScope = localScopes.lastOrNull() ?: return this
        val indexOfLastLocalScope = towerDataElements.indexOfLast { it.scope === oldLastScope }
        val newLastScope = oldLastScope.storeVariable(variable, session)

        return copy(
            towerDataElements = towerDataElements.set(indexOfLastLocalScope, newLastScope.asTowerDataElement(isLocal = true)),
            localScopes = localScopes.set(localScopes.lastIndex, newLastScope),
            localVariableScopeStorage = localVariableScopeStorage.addLocalVariable(variable.symbol)
        )
    }

    fun setLastLocalScope(newLastScope: FirLocalScope): FirTowerDataContext {
        val oldLastScope = localScopes.last()
        val indexOfLastLocalScope = towerDataElements.indexOfLast { it.scope === oldLastScope }

        return copy(
            towerDataElements = towerDataElements.set(indexOfLastLocalScope, newLastScope.asTowerDataElement(isLocal = true)),
            localScopes = localScopes.set(localScopes.lastIndex, newLastScope),
        )
    }

    fun addNonLocalTowerDataElements(newElements: List<FirTowerDataElement>): FirTowerDataContext {
        return copy(
            towerDataElements = towerDataElements.addAll(newElements),
            implicitValueStorage = implicitValueStorage
                .addAllImplicitReceivers(newElements.mapNotNull { it.implicitReceiver })
                .addAllContexts(
                    newElements.flatMap { it.contextParameterGroup.orEmpty() }
                ),
            nonLocalTowerDataElements = nonLocalTowerDataElements.addAll(newElements)
        )
    }

    fun addLocalScope(localScope: FirLocalScope): FirTowerDataContext {
        return copy(
            towerDataElements = towerDataElements.add(localScope.asTowerDataElement(isLocal = true)),
            localScopes = localScopes.add(localScope),
        )
    }

    fun addReceiver(name: Name?, implicitReceiverValue: ImplicitReceiverValue<*>): FirTowerDataContext {
        val element = implicitReceiverValue.asTowerDataElement()
        return copy(
            towerDataElements = towerDataElements.add(element),
            implicitValueStorage = implicitValueStorage.addImplicitReceiver(name, implicitReceiverValue),
            nonLocalTowerDataElements = nonLocalTowerDataElements.add(element)
        )
    }

    fun addReceiverIfNotNull(name: Name?, implicitReceiverValue: ImplicitReceiverValue<*>?): FirTowerDataContext {
        if (implicitReceiverValue == null) return this
        return addReceiver(name, implicitReceiverValue)
    }

    fun addContextGroups(
        contextParameterGroup: ContextParameterGroup,
    ): FirTowerDataContext {
        if (contextParameterGroup.isEmpty()) return this
        val element = FirTowerDataElement(
            scope = null,
            implicitReceiver = null,
            contextParameterGroup = contextParameterGroup,
            isLocal = false
        )

        return copy(
            towerDataElements = towerDataElements.add(element),
            implicitValueStorage = implicitValueStorage.addAllContexts(contextParameterGroup),
            nonLocalTowerDataElements = nonLocalTowerDataElements.add(element)
        )
    }

    fun addAnonymousInitializer(anonymousInitializer: FirAnonymousInitializer): FirTowerDataContext {
        val correspondingClass = anonymousInitializer.containingDeclarationSymbol as? FirClassSymbol<*> ?: return this
        return copy(
            classesUnderInitialization = classesUnderInitialization.add(correspondingClass),
        )
    }

    fun addNonLocalScopeIfNotNull(scope: FirScope?): FirTowerDataContext {
        if (scope == null) return this
        return addNonLocalScope(scope)
    }

    // Optimized version for two parameters
    fun addNonLocalScopesIfNotNull(scope1: FirScope?, scope2: FirScope?): FirTowerDataContext {
        return if (scope1 != null) {
            if (scope2 != null) {
                addNonLocalScopeElements(listOf(scope1.asTowerDataElement(isLocal = false), scope2.asTowerDataElement(isLocal = false)))
            } else {
                addNonLocalScope(scope1)
            }
        } else if (scope2 != null) {
            addNonLocalScope(scope2)
        } else {
            this
        }
    }

    fun addNonLocalScope(scope: FirScope): FirTowerDataContext {
        val element = scope.asTowerDataElement(isLocal = false)
        return copy(
            towerDataElements = towerDataElements.add(element),
            nonLocalTowerDataElements = nonLocalTowerDataElements.add(element)
        )
    }

    private fun addNonLocalScopeElements(elements: List<FirTowerDataElement>): FirTowerDataContext {
        return copy(
            towerDataElements = towerDataElements.addAll(elements),
            nonLocalTowerDataElements = nonLocalTowerDataElements.addAll(elements)
        )
    }

    fun createSnapshot(keepMutable: Boolean): FirTowerDataContext {
        val implicitValueMapper = object : ImplicitValueMapper {
            val implicitValueCache = HashMap<ImplicitValue<*>, ImplicitValue<*>>()

            override fun <S : FirBasedSymbol<*>, T : ImplicitValue<S>> invoke(value: T): T {
                @Suppress("UNCHECKED_CAST")
                return implicitValueCache.getOrPut(value) { value.createSnapshot(keepMutable) } as T
            }
        }

        return copy(
            towerDataElements = towerDataElements.map { it.createSnapshot(keepMutable, implicitValueMapper) }.toPersistentList(),
            implicitValueStorage = implicitValueStorage.createSnapshot(implicitValueMapper),
            localScopes = localScopes.toPersistentList(),
            nonLocalTowerDataElements = nonLocalTowerDataElements.map { it.createSnapshot(keepMutable, implicitValueMapper) }.toPersistentList()
        )
    }

    fun replaceTowerDataElements(
        towerDataElements: PersistentList<FirTowerDataElement>,
        nonLocalTowerDataElements: PersistentList<FirTowerDataElement>,
    ): FirTowerDataContext {
        return copy(
            towerDataElements = towerDataElements,
            nonLocalTowerDataElements = nonLocalTowerDataElements
        )
    }
}

/**
 * Each FirTowerDataElement has exactly one non-null value among [scope] or [implicitReceiver].
 */
class FirTowerDataElement(
    val scope: FirScope?,
    val implicitReceiver: ImplicitReceiverValue<*>?,
    val contextParameterGroup: ContextParameterGroup? = null,
    val isLocal: Boolean,
    val staticScopeOwnerSymbol: FirRegularClassSymbol? = null,
) {
    internal fun createSnapshot(keepMutable: Boolean, mapper: ImplicitValueMapper): FirTowerDataElement =
        FirTowerDataElement(
            scope,
            implicitReceiver?.let { mapper(it) },
            contextParameterGroup?.map { it.createSnapshot(keepMutable) },
            isLocal,
            staticScopeOwnerSymbol
        )

    /**
     * Returns [scope] if it is not null. Otherwise, returns scopes of implicit receiver.
     *
     * Note that a scope for a companion object is an implicit scope.
     */
    fun getAvailableScopes(
        processTypeScope: FirTypeScope.(ConeKotlinType) -> FirTypeScope = { this },
    ): List<FirScope> = when {
        scope != null -> listOf(scope)
        implicitReceiver != null -> listOf(implicitReceiver.getImplicitScope(processTypeScope))
        contextParameterGroup != null -> emptyList()
        else -> error("Tower data element is expected to have either scope or implicit receivers.")
    }

    private fun ImplicitReceiverValue<*>.getImplicitScope(
        processTypeScope: FirTypeScope.(ConeKotlinType) -> FirTypeScope,
    ): FirScope {
        // N.B.: implicitScope == null when the type sits in a user-defined 'kotlin' package,
        // but there is no '-Xallow-kotlin-package' compiler argument provided
        val implicitScope = implicitScope ?: return FirTypeScope.Empty

        val type = type.fullyExpandedType()
        if (type is ConeErrorType || type is ConeStubType) return FirTypeScope.Empty

        return implicitScope.processTypeScope(type)
    }
}

fun ImplicitReceiverValue<*>.asTowerDataElement(): FirTowerDataElement =
    FirTowerDataElement(scope = null, implicitReceiver = this, isLocal = false)

fun FirScope.asTowerDataElement(isLocal: Boolean): FirTowerDataElement =
    FirTowerDataElement(scope = this, implicitReceiver = null, isLocal = isLocal)

fun FirScope.asTowerDataElementForStaticScope(staticScopeOwnerSymbol: FirRegularClassSymbol?): FirTowerDataElement =
    FirTowerDataElement(scope = this, implicitReceiver = null, isLocal = false, staticScopeOwnerSymbol = staticScopeOwnerSymbol)

fun FirClassSymbol<*>.staticScope(sessionHolder: SessionAndScopeSessionHolder): FirContainingNamesAwareScope? =
    fir.staticScope(sessionHolder)

fun FirClassSymbol<*>.staticScope(session: FirSession, scopeSession: ScopeSession): FirContainingNamesAwareScope? =
    fir.staticScope(session, scopeSession)

fun FirClass.staticScope(sessionHolder: SessionAndScopeSessionHolder): FirContainingNamesAwareScope? =
    staticScope(sessionHolder.session, sessionHolder.scopeSession)

fun FirClass.staticScope(session: FirSession, scopeSession: ScopeSession): FirContainingNamesAwareScope? =
    scopeProvider.getStaticScope(this, session, scopeSession)

typealias ContextParameterGroup = List<ImplicitContextParameterValue>
typealias FirLocalScopes = PersistentList<FirLocalScope>
