// Copyright 2000-2021 JetBrains s.r.o. and contributors. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package org.jetbrains.kotlin.arguments

import org.jetbrains.kotlin.cli.common.arguments.Argument
import org.jetbrains.kotlin.cli.common.arguments.CommonToolArguments
import org.jetbrains.kotlin.cli.common.arguments.ManualLanguageFeatureSetting
import org.jetbrains.kotlin.utils.addToStdlib.safeAs
import kotlin.reflect.KClass
import kotlin.reflect.KProperty1
import kotlin.reflect.KType
import kotlin.reflect.full.memberProperties
import kotlin.reflect.jvm.javaField

object CompilerArgumentsContentProspector {
    private val argumentPropertiesCache: MutableMap<KClass<out CommonToolArguments>, Collection<KProperty1<out CommonToolArguments, *>>> =
        mutableMapOf()

    private val flagArgumentPropertiesCache: MutableMap<KClass<out CommonToolArguments>, Collection<KProperty1<out CommonToolArguments, Boolean>>> =
        mutableMapOf()

    private val stringArgumentPropertiesCache: MutableMap<KClass<out CommonToolArguments>, Collection<KProperty1<out CommonToolArguments, String?>>> =
        mutableMapOf()

    private val arrayArgumentPropertiesCache: MutableMap<KClass<out CommonToolArguments>, Collection<KProperty1<out CommonToolArguments, Array<String>?>>> =
        mutableMapOf()

    private fun getCompilerArgumentsProperties(kClass: KClass<out CommonToolArguments>) = argumentPropertiesCache.getOrPut(kClass) {
        kClass.memberProperties.filter { prop -> prop.javaField?.getAnnotation(Argument::class.java) != null }
    }

    private inline fun <reified R : Any?> Collection<KProperty1<out CommonToolArguments, *>>.filterByReturnType(predicate: (KType?) -> Boolean) =
        filter { predicate(it.returnType) }.mapNotNull { it.safeAs<KProperty1<CommonToolArguments, R>>() }

    fun getFlagCompilerArgumentProperties(kClass: KClass<out CommonToolArguments>): Collection<KProperty1<out CommonToolArguments, Boolean>> =
        flagArgumentPropertiesCache.getOrPut(kClass) { getCompilerArgumentsProperties(kClass).filterByReturnType { it?.classifier == Boolean::class } }

    fun getStringCompilerArgumentProperties(kClass: KClass<out CommonToolArguments>): Collection<KProperty1<out CommonToolArguments, String?>> =
        stringArgumentPropertiesCache.getOrPut(kClass) { getCompilerArgumentsProperties(kClass).filterByReturnType { it?.classifier == String::class } }

    fun getArrayCompilerArgumentProperties(kClass: KClass<out CommonToolArguments>): Collection<KProperty1<out CommonToolArguments, Array<String>?>> =
        arrayArgumentPropertiesCache.getOrPut(kClass) { getCompilerArgumentsProperties(kClass).filterByReturnType { (it?.classifier as? KClass<*>)?.java?.isArray == true } }

    val freeArgsProperty: KProperty1<in CommonToolArguments, List<String>>
        get() = CommonToolArguments::freeArgs
    val internalArgumentsProperty: KProperty1<in CommonToolArguments, List<ManualLanguageFeatureSetting>>
        get() = CommonToolArguments::internalArguments
}
