package org.jfrog.jade.plugins.common.naming;

/*
 * Copyright 2005-2006 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import org.apache.maven.artifact.Artifact;
import org.apache.maven.model.Dependency;
import org.apache.maven.project.MavenProject;

import java.util.HashSet;
import java.util.Set;

/**
 * @author Fred Simon
 */
public class ProjectNameProviderImpl implements ProjectNameProvider {
    private boolean initialized = false;

    private Set<GroupDefinition> groupDefinitions = new HashSet<GroupDefinition>();

    public ProjectNameProviderImpl() {
    }

    public boolean isInitialized() {
        return initialized;
    }

    public void setInitialized(boolean initialized) {
        this.initialized = initialized;
    }

    public Set<GroupDefinition> getGroupDefinitions() {
        return groupDefinitions;
    }

    public void setGroupDefinitions(Set<GroupDefinition> groupDefinitions) {
        this.groupDefinitions = groupDefinitions;
    }

    public String getProjectName(MavenProject project) {
        String artifactId = project.getArtifactId();
        String groupId = project.getGroupId();

        return getProjectName(groupId, artifactId);
    }

    public String getProjectName(Artifact artifact) {
        String artifactId = artifact.getArtifactId();
        String groupId = artifact.getGroupId();

        return getProjectName(groupId, artifactId);
    }

    public String getProjectName(Dependency dependency) {
        String artifactId = dependency.getArtifactId();
        String groupId = dependency.getGroupId();

        return getProjectName(groupId, artifactId);
    }

    private String getProjectName(String groupId, String artifactId) {
        if (!hasGroupDefinitions()) {
            // No group change, just use artifactId
            return artifactId;
        }

        StringBuffer result = new StringBuffer(artifactId);

        String groupName = getGroupName(groupId);
        if (groupName != null) {
            result.insert(0, '-');
            result.insert(0, groupName);
        }

        return result.toString();
    }

    public String getGroupName(String groupId) {
        String bestMatch = null;
        int priority = 0;
        for (GroupDefinition groupDefinition : groupDefinitions) {
            String groupName = groupDefinition.getName(groupId);
            if (groupName != null) {
                int groupDefPriority = groupDefinition.getPriority();
                if (groupDefPriority > priority) {
                    bestMatch = groupName;
                    priority = groupDefPriority;
                }
            }
        }
        return bestMatch;
    }

    public boolean hasGroupDefinitions() {
        return groupDefinitions != null && !groupDefinitions.isEmpty();
    }
}
