/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.protocol.oidc;

import jakarta.ws.rs.core.HttpHeaders;
import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.UriInfo;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.BiConsumer;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.jboss.logging.Logger;
import org.keycloak.OAuthErrorException;
import org.keycloak.Token;
import org.keycloak.TokenCategory;
import org.keycloak.TokenVerifier;
import org.keycloak.authentication.authenticators.util.AcrStore;
import org.keycloak.broker.oidc.OIDCIdentityProvider;
import org.keycloak.broker.oidc.OIDCIdentityProviderConfig;
import org.keycloak.broker.provider.IdentityBrokerException;
import org.keycloak.common.ClientConnection;
import org.keycloak.common.Profile;
import org.keycloak.common.VerificationException;
import org.keycloak.common.util.Time;
import org.keycloak.crypto.HashProvider;
import org.keycloak.crypto.SignatureProvider;
import org.keycloak.events.EventBuilder;
import org.keycloak.http.HttpRequest;
import org.keycloak.jose.jws.JWSInput;
import org.keycloak.jose.jws.JWSInputException;
import org.keycloak.jose.jws.crypto.HashUtils;
import org.keycloak.migration.migrators.MigrationUtils;
import org.keycloak.models.AuthenticatedClientSessionModel;
import org.keycloak.models.ClientModel;
import org.keycloak.models.ClientScopeModel;
import org.keycloak.models.ClientSessionContext;
import org.keycloak.models.ImpersonationSessionNote;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.ProtocolMapperModel;
import org.keycloak.models.RealmModel;
import org.keycloak.models.RoleModel;
import org.keycloak.models.SingleUseObjectProvider;
import org.keycloak.models.UserConsentModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.models.UserSessionProvider;
import org.keycloak.models.light.LightweightUserAdapter;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.models.utils.RoleUtils;
import org.keycloak.models.utils.SessionExpirationUtils;
import org.keycloak.protocol.ProtocolMapper;
import org.keycloak.protocol.ProtocolMapperUtils;
import org.keycloak.protocol.oidc.LogoutTokenValidationCode;
import org.keycloak.protocol.oidc.OIDCAdvancedConfigWrapper;
import org.keycloak.protocol.oidc.mappers.OIDCAccessTokenMapper;
import org.keycloak.protocol.oidc.mappers.OIDCAccessTokenResponseMapper;
import org.keycloak.protocol.oidc.mappers.OIDCIDTokenMapper;
import org.keycloak.protocol.oidc.mappers.TokenIntrospectionTokenMapper;
import org.keycloak.protocol.oidc.mappers.UserInfoTokenMapper;
import org.keycloak.protocol.oidc.utils.OIDCResponseType;
import org.keycloak.rar.AuthorizationDetails;
import org.keycloak.rar.AuthorizationRequestContext;
import org.keycloak.representations.AccessToken;
import org.keycloak.representations.AccessTokenResponse;
import org.keycloak.representations.AuthorizationDetailsJSONRepresentation;
import org.keycloak.representations.IDToken;
import org.keycloak.representations.JsonWebToken;
import org.keycloak.representations.LogoutToken;
import org.keycloak.representations.RefreshToken;
import org.keycloak.representations.dpop.DPoP;
import org.keycloak.services.ErrorResponseException;
import org.keycloak.services.Urls;
import org.keycloak.services.managers.AuthenticationManager;
import org.keycloak.services.managers.AuthenticationSessionManager;
import org.keycloak.services.managers.UserConsentManager;
import org.keycloak.services.managers.UserSessionCrossDCManager;
import org.keycloak.services.managers.UserSessionManager;
import org.keycloak.services.resources.IdentityBrokerService;
import org.keycloak.services.util.AuthorizationContextUtil;
import org.keycloak.services.util.DPoPUtil;
import org.keycloak.services.util.DefaultClientSessionContext;
import org.keycloak.services.util.MtlsHoKTokenUtil;
import org.keycloak.sessions.AuthenticationSessionModel;
import org.keycloak.util.TokenUtil;

public class TokenManager {
    private static final Logger logger = Logger.getLogger(TokenManager.class);
    private static final String JWT = "JWT";

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public TokenValidation validateToken(KeycloakSession session, UriInfo uriInfo, ClientConnection connection, RealmModel realm, RefreshToken oldToken, HttpHeaders headers, String oldTokenScope) throws OAuthErrorException {
        DefaultClientSessionContext clientSessionCtx;
        UserModel user;
        UserSessionModel userSession = null;
        boolean offline = "Offline".equals(oldToken.getType());
        if (offline) {
            UserSessionManager sessionManager = new UserSessionManager(session);
            userSession = sessionManager.findOfflineUserSession(realm, oldToken.getSessionState());
            if (userSession == null) throw new OAuthErrorException("invalid_grant", "Offline user session not found", "Offline user session not found");
            if (!AuthenticationManager.isOfflineSessionValid(realm, userSession)) {
                sessionManager.revokeOfflineUserSession(userSession);
                throw new OAuthErrorException("invalid_grant", "Offline session not active", "Offline session not active");
            }
        } else {
            userSession = session.sessions().getUserSession(realm, oldToken.getSessionState());
            if (!AuthenticationManager.isSessionValid(realm, userSession)) {
                AuthenticationManager.backchannelLogout(session, realm, userSession, uriInfo, connection, headers, true);
                throw new OAuthErrorException("invalid_grant", "Session not active", "Session not active");
            }
        }
        if ((user = userSession.getUser()) == null) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token", "Unknown user");
        }
        if (!user.isEnabled()) {
            throw new OAuthErrorException("invalid_grant", "User disabled", "User disabled");
        }
        if (oldToken.isIssuedBeforeSessionStart((long)userSession.getStarted())) {
            logger.debug((Object)"Refresh token issued before the user session started");
            throw new OAuthErrorException("invalid_grant", "Refresh token issued before the user session started");
        }
        ClientModel client = session.getContext().getClient();
        AuthenticatedClientSessionModel clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        if (clientSession == null) {
            if ((userSession = new UserSessionCrossDCManager(session).getUserSessionWithClient(realm, userSession.getId(), offline, client.getId())) == null) throw new OAuthErrorException("invalid_grant", "Session doesn't have required client", "Session doesn't have required client");
            clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        }
        if (oldToken.isIssuedBeforeSessionStart((long)clientSession.getStarted())) {
            logger.debug((Object)"refresh token issued before the client session started");
            throw new OAuthErrorException("invalid_grant", "refresh token issued before the client session started");
        }
        if (!client.getClientId().equals(oldToken.getIssuedFor())) {
            throw new OAuthErrorException("invalid_grant", "Unmatching clients", "Unmatching clients");
        }
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)oldToken).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(client), NotBeforeCheck.forModel(session, realm, user)}).verify();
        }
        catch (VerificationException e) {
            throw new OAuthErrorException("invalid_grant", "Stale token");
        }
        if (oldTokenScope == null && userSession.isOffline()) {
            logger.debugf("Migrating offline token of user '%s' for client '%s' of realm '%s'", (Object)user.getUsername(), (Object)client.getClientId(), (Object)realm.getName());
            MigrationUtils.migrateOldOfflineToken((KeycloakSession)session, (RealmModel)realm, (ClientModel)client, (UserModel)user);
            oldTokenScope = "offline_access";
        }
        if (!TokenManager.verifyConsentStillAvailable(session, user, client, (clientSessionCtx = DefaultClientSessionContext.fromClientSessionAndScopeParameter(clientSession, oldTokenScope, session)).getClientScopesStream())) {
            throw new OAuthErrorException("invalid_scope", "Client no longer has requested consent from user");
        }
        clientSessionCtx.setAttribute("nonce", oldToken.getNonce());
        AccessToken newToken = this.createClientAccessToken(session, realm, client, user, userSession, clientSessionCtx);
        return new TokenValidation(user, userSession, clientSessionCtx, newToken);
    }

    public boolean checkTokenValidForIntrospection(KeycloakSession session, RealmModel realm, AccessToken token, boolean updateTimestamps, EventBuilder eventBuilder) {
        ClientModel client = realm.getClientByClientId(token.getIssuedFor());
        if (client == null) {
            logger.debugf("Introspection access token : client with clientId %s does not exist", (Object)token.getIssuedFor());
            eventBuilder.detail("reason", String.format("Could not find client for %s", token.getIssuedFor()));
            return false;
        }
        if (!client.isEnabled()) {
            logger.debugf("Introspection access token : client with clientId %s is disabled", (Object)token.getIssuedFor());
            eventBuilder.detail("reason", String.format("Client with clientId %s is disabled", token.getIssuedFor()));
            return false;
        }
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)token).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(client), TokenVerifier.IS_ACTIVE, new TokenRevocationCheck(session)}).verify();
        }
        catch (VerificationException e) {
            logger.debugf("Introspection access token for %s client: JWT check failed: %s", (Object)token.getIssuedFor(), (Object)e.getMessage());
            eventBuilder.detail("reason", "Introspection access token for " + token.getIssuedFor() + " client: JWT check failed");
            return false;
        }
        boolean valid = false;
        if (token.getSessionState() == null) {
            UserModel user = TokenManager.lookupUserFromStatelessToken(session, realm, token);
            valid = this.isUserValid(session, realm, token, user);
            if (!valid) {
                eventBuilder.detail("reason", "Could not find valid transient user session");
            }
        } else {
            AuthenticatedClientSessionModel clientSession;
            UserSessionModel userSession = new UserSessionCrossDCManager(session).getUserSessionWithClient(realm, token.getSessionState(), false, client.getId());
            if (userSession == null) {
                userSession = session.sessions().getUserSessionWithPredicate(realm, token.getSessionState(), false, model -> client.getId().equals(model.getNote(ImpersonationSessionNote.IMPERSONATOR_CLIENT.toString())));
            }
            if (AuthenticationManager.isSessionValid(realm, userSession)) {
                valid = this.isUserValid(session, realm, token, userSession.getUser());
            } else {
                userSession = new UserSessionCrossDCManager(session).getUserSessionWithClient(realm, token.getSessionState(), true, client.getId());
                if (AuthenticationManager.isOfflineSessionValid(realm, userSession)) {
                    valid = this.isUserValid(session, realm, token, userSession.getUser());
                }
            }
            if (!valid) {
                logger.debugf("Could not find valid user session for session_state = %s", (Object)token.getSessionState());
                eventBuilder.detail("reason", String.format("Could not find valid user session for session_state = %s", token.getSessionState()));
            }
            if (valid && token.isIssuedBeforeSessionStart((long)userSession.getStarted())) {
                valid = false;
                logger.debugf("Token is issued (%s) before session () has started", (Object)String.valueOf(token.getIat()), (Object)String.valueOf(userSession.getStarted()));
                eventBuilder.detail("reason", String.format("Token is issued (%s) before user session () has started", String.valueOf(token.getIat()), String.valueOf(userSession.getStarted())));
            }
            AuthenticatedClientSessionModel authenticatedClientSessionModel = clientSession = userSession == null ? null : userSession.getAuthenticatedClientSessionByClient(client.getId());
            if (clientSession != null && valid && token.isIssuedBeforeSessionStart((long)clientSession.getStarted())) {
                valid = false;
                logger.debugf("Token is issued (%s) before session () has started", (Object)String.valueOf(token.getIat()), (Object)String.valueOf(clientSession.getStarted()));
                eventBuilder.detail("reason", String.format("Token is issued (%s) before client session () has started", String.valueOf(token.getIat()), String.valueOf(clientSession.getStarted())));
            }
            String tokenType = token.getType();
            if (realm.isRevokeRefreshToken() && (tokenType.equals("Refresh") || tokenType.equals("Offline")) && !this.validateTokenReuseForIntrospection(session, realm, token)) {
                logger.debug((Object)("Introspection access token for " + token.getIssuedFor() + " client: failed to validate Token reuse for introspection"));
                eventBuilder.detail("reason", "Realm revoke refresh token, token type is " + tokenType + " and token is not eligible for introspection");
                return false;
            }
            if (updateTimestamps && valid) {
                int currentTime = Time.currentTime();
                userSession.setLastSessionRefresh(currentTime);
                if (clientSession != null) {
                    clientSession.setTimestamp(currentTime);
                }
            }
        }
        return valid;
    }

    private boolean validateTokenReuseForIntrospection(KeycloakSession session, RealmModel realm, AccessToken token) {
        UserSessionModel userSession = null;
        if (token.getType().equals("Refresh")) {
            userSession = session.sessions().getUserSession(realm, token.getSessionState());
        } else {
            UserSessionManager sessionManager = new UserSessionManager(session);
            userSession = sessionManager.findOfflineUserSession(realm, token.getSessionState());
        }
        ClientModel client = realm.getClientByClientId(token.getIssuedFor());
        AuthenticatedClientSessionModel clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        try {
            this.validateTokenReuse(session, realm, token, clientSession, false);
            return true;
        }
        catch (OAuthErrorException e) {
            logger.debug((Object)"validateTokenReuseForIntrospection is false", (Throwable)e);
            return false;
        }
    }

    private boolean isUserValid(KeycloakSession session, RealmModel realm, AccessToken token, UserModel user) {
        if (user == null) {
            logger.debugf("User does not exist for token introspection", new Object[0]);
            return false;
        }
        if (!user.isEnabled()) {
            logger.debugf("User is disable for token introspection", new Object[0]);
            return false;
        }
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)token).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(session, realm, user)}).verify();
        }
        catch (VerificationException e) {
            logger.debugf("JWT check failed: %s", (Object)e.getMessage());
            return false;
        }
        return true;
    }

    public static UserModel lookupUserFromStatelessToken(KeycloakSession session, RealmModel realm, AccessToken token) {
        UserModel user = session.users().getUserById(realm, token.getSubject());
        if (user != null) {
            return user;
        }
        if (token.getPreferredUsername() != null) {
            return session.users().getUserByUsername(realm, token.getPreferredUsername());
        }
        return null;
    }

    public AccessTokenResponseBuilder refreshAccessToken(KeycloakSession session, UriInfo uriInfo, ClientConnection connection, RealmModel realm, ClientModel authorizedClient, String encodedRefreshToken, EventBuilder event, HttpHeaders headers, HttpRequest request, String scopeParameter) throws OAuthErrorException {
        String scopeParam;
        RefreshToken refreshToken = this.verifyRefreshToken(session, realm, authorizedClient, request, encodedRefreshToken, true);
        event.user(refreshToken.getSubject()).session(refreshToken.getSessionState()).detail("refresh_token_id", refreshToken.getId()).detail("refresh_token_type", refreshToken.getType());
        String oldTokenScope = refreshToken.getScope();
        if (scopeParameter != null && !scopeParameter.isEmpty()) {
            Set scopeParamScopes = Arrays.stream(scopeParameter.split(" ")).collect(Collectors.toSet());
            oldTokenScope = Arrays.stream(oldTokenScope.split(" ")).filter(sc -> scopeParamScopes.contains(sc) || sc.equals("offline_access")).collect(Collectors.joining(" "));
        }
        TokenValidation validation = this.validateToken(session, uriInfo, connection, realm, refreshToken, headers, oldTokenScope);
        AuthenticatedClientSessionModel clientSession = validation.clientSessionCtx.getClientSession();
        OIDCAdvancedConfigWrapper clientConfig = OIDCAdvancedConfigWrapper.fromClientModel(authorizedClient);
        if (!clientSession.getClient().getId().equals(authorizedClient.getId())) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token. Token client and authorized client don't match");
        }
        this.validateTokenReuseForRefresh(session, realm, refreshToken, validation);
        if (refreshToken.getAuthorization() != null) {
            validation.newToken.setAuthorization(refreshToken.getAuthorization());
        }
        AccessTokenResponseBuilder responseBuilder = this.responseBuilder(realm, authorizedClient, event, session, validation.userSession, validation.clientSessionCtx).accessToken(validation.newToken);
        if (clientConfig.isUseRefreshToken()) {
            responseBuilder.generateRefreshToken(refreshToken.getScope());
        }
        if (validation.newToken.getAuthorization() != null && clientConfig.isUseRefreshToken()) {
            responseBuilder.getRefreshToken().setAuthorization(validation.newToken.getAuthorization());
        }
        if (TokenUtil.isOIDCRequest((String)(scopeParam = clientSession.getNote("scope")))) {
            responseBuilder.generateIDToken().generateAccessTokenHash();
        }
        return responseBuilder;
    }

    private void validateTokenReuseForRefresh(KeycloakSession session, RealmModel realm, RefreshToken refreshToken, TokenValidation validation) throws OAuthErrorException {
        if (realm.isRevokeRefreshToken()) {
            AuthenticatedClientSessionModel clientSession = validation.clientSessionCtx.getClientSession();
            try {
                this.validateTokenReuse(session, realm, (AccessToken)refreshToken, clientSession, true);
                int currentCount = clientSession.getCurrentRefreshTokenUseCount();
                clientSession.setCurrentRefreshTokenUseCount(currentCount + 1);
            }
            catch (OAuthErrorException oee) {
                if (logger.isDebugEnabled()) {
                    logger.debugf("Failed validation of refresh token %s due it was used before. Realm: %s, client: %s, user: %s, user session: %s. Will detach client session from user session", new Object[]{refreshToken.getId(), realm.getName(), clientSession.getClient().getClientId(), clientSession.getUserSession().getUser().getUsername(), clientSession.getUserSession().getId()});
                }
                clientSession.detachFromUserSession();
                throw oee;
            }
        }
    }

    private void validateTokenReuse(KeycloakSession session, RealmModel realm, AccessToken refreshToken, AuthenticatedClientSessionModel clientSession, boolean refreshFlag) throws OAuthErrorException {
        int currentCount;
        int startupTime = ((UserSessionProvider)session.getProvider(UserSessionProvider.class)).getStartupTime(realm);
        if (clientSession.getCurrentRefreshToken() != null && !refreshToken.getId().equals(clientSession.getCurrentRefreshToken()) && refreshToken.getIssuedAt() < clientSession.getTimestamp() && startupTime <= clientSession.getTimestamp()) {
            throw new OAuthErrorException("invalid_grant", "Stale token");
        }
        if (!refreshToken.getId().equals(clientSession.getCurrentRefreshToken())) {
            if (refreshFlag) {
                clientSession.setCurrentRefreshToken(refreshToken.getId());
                clientSession.setCurrentRefreshTokenUseCount(0);
            } else {
                return;
            }
        }
        if ((currentCount = clientSession.getCurrentRefreshTokenUseCount()) > realm.getRefreshTokenMaxReuse()) {
            throw new OAuthErrorException("invalid_grant", "Maximum allowed refresh token reuse exceeded", "Maximum allowed refresh token reuse exceeded");
        }
    }

    public RefreshToken verifyRefreshToken(KeycloakSession session, RealmModel realm, ClientModel client, HttpRequest request, String encodedRefreshToken, boolean checkExpiration) throws OAuthErrorException {
        try {
            RefreshToken refreshToken = this.toRefreshToken(session, encodedRefreshToken);
            if (!"Refresh".equals(refreshToken.getType()) && !"Offline".equals(refreshToken.getType())) {
                throw new OAuthErrorException("invalid_grant", "Invalid refresh token");
            }
            TokenVerifier tokenVerifier = TokenVerifier.createWithoutSignature((JsonWebToken)refreshToken).withChecks(new TokenVerifier.Predicate[]{new TokenVerifier.RealmUrlCheck(Urls.realmIssuer(session.getContext().getUri().getBaseUri(), realm.getName()))});
            if (checkExpiration) {
                tokenVerifier.withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(realm), TokenVerifier.IS_ACTIVE});
            }
            try {
                tokenVerifier.verify();
            }
            catch (VerificationException e) {
                throw new OAuthErrorException("invalid_grant", e.getMessage());
            }
            if (!client.getClientId().equals(refreshToken.getIssuedFor())) {
                throw new OAuthErrorException("invalid_grant", "Invalid refresh token. Token client and authorized client don't match");
            }
            if (OIDCAdvancedConfigWrapper.fromClientModel(client).isUseMtlsHokToken() && !MtlsHoKTokenUtil.verifyTokenBindingWithClientCertificate((AccessToken)refreshToken, request, session)) {
                throw new OAuthErrorException("unauthorized_client", "Client certificate missing, or its thumbprint and one in the refresh token did NOT match");
            }
            if (Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DPOP)) {
                DPoP dPoP = (DPoP)session.getAttribute("dpop");
                if (client.isPublicClient() && (OIDCAdvancedConfigWrapper.fromClientModel(client).isUseDPoP() || dPoP != null)) {
                    try {
                        DPoPUtil.validateBinding((AccessToken)refreshToken, dPoP);
                    }
                    catch (VerificationException ex) {
                        throw new OAuthErrorException("invalid_grant", ex.getMessage());
                    }
                }
            }
            return refreshToken;
        }
        catch (JWSInputException e) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token", (Throwable)e);
        }
    }

    public RefreshToken toRefreshToken(KeycloakSession session, String encodedRefreshToken) throws JWSInputException, OAuthErrorException {
        RefreshToken refreshToken = (RefreshToken)session.tokens().decode(encodedRefreshToken, RefreshToken.class);
        if (refreshToken == null) {
            throw new OAuthErrorException("invalid_grant", "Invalid refresh token");
        }
        return refreshToken;
    }

    public IDToken verifyIDToken(KeycloakSession session, RealmModel realm, String encodedIDToken) throws OAuthErrorException {
        IDToken idToken = (IDToken)session.tokens().decode(encodedIDToken, IDToken.class);
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)idToken).withChecks(new TokenVerifier.Predicate[]{NotBeforeCheck.forModel(realm), TokenVerifier.IS_ACTIVE}).verify();
        }
        catch (VerificationException e) {
            throw new OAuthErrorException("invalid_grant", e.getMessage());
        }
        return idToken;
    }

    public IDToken verifyIDTokenSignature(KeycloakSession session, String encodedIDToken) throws OAuthErrorException {
        IDToken idToken = (IDToken)session.tokens().decode(encodedIDToken, IDToken.class);
        if (idToken == null) {
            throw new OAuthErrorException("invalid_grant", "Invalid IDToken");
        }
        return idToken;
    }

    public AccessToken createClientAccessToken(KeycloakSession session, RealmModel realm, ClientModel client, UserModel user, UserSessionModel userSession, ClientSessionContext clientSessionCtx) {
        AccessToken token = this.initToken(realm, client, user, userSession, clientSessionCtx, (UriInfo)session.getContext().getUri());
        token = this.transformAccessToken(session, token, userSession, clientSessionCtx);
        return token;
    }

    public static ClientSessionContext attachAuthenticationSession(KeycloakSession session, UserSessionModel userSession, AuthenticationSessionModel authSession) {
        ClientModel client = authSession.getClient();
        AuthenticatedClientSessionModel clientSession = userSession.getAuthenticatedClientSessionByClient(client.getId());
        if (clientSession == null) {
            clientSession = session.sessions().createClientSession(userSession.getRealm(), client, userSession);
        }
        clientSession.setRedirectUri(authSession.getRedirectUri());
        clientSession.setProtocol(authSession.getProtocol());
        Set<String> clientScopeIds = Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DYNAMIC_SCOPES) ? AuthorizationContextUtil.getClientScopesStreamFromAuthorizationRequestContextWithClient(session, authSession.getClientNote("scope")).map(ClientScopeModel::getId).collect(Collectors.toSet()) : authSession.getClientScopes();
        Map transferredNotes = authSession.getClientNotes();
        for (Map.Entry entry : transferredNotes.entrySet()) {
            clientSession.setNote((String)entry.getKey(), (String)entry.getValue());
        }
        Map transferredUserSessionNotes = authSession.getUserSessionNotes();
        for (Map.Entry entry : transferredUserSessionNotes.entrySet()) {
            userSession.setNote((String)entry.getKey(), (String)entry.getValue());
        }
        clientSession.setNote("level-of-authentication", String.valueOf(new AcrStore(session, authSession).getLevelOfAuthenticationFromCurrentAuthentication()));
        clientSession.setTimestamp(userSession.getLastSessionRefresh());
        new AuthenticationSessionManager(session).updateAuthenticationSessionAfterSuccessfulAuthentication(userSession.getRealm(), authSession);
        DefaultClientSessionContext defaultClientSessionContext = DefaultClientSessionContext.fromClientSessionAndClientScopeIds(clientSession, clientScopeIds, session);
        return defaultClientSessionContext;
    }

    public static void dettachClientSession(AuthenticatedClientSessionModel clientSession) {
        UserSessionModel userSession = clientSession.getUserSession();
        if (userSession == null) {
            return;
        }
        clientSession.detachFromUserSession();
    }

    public static Set<RoleModel> getAccess(UserModel user, ClientModel client, Stream<ClientScopeModel> clientScopes) {
        Set roleMappings = RoleUtils.getDeepUserRoleMappings((UserModel)user);
        if (client.isFullScopeAllowed()) {
            if (logger.isTraceEnabled()) {
                logger.tracef("Using full scope for client %s", (Object)client.getClientId());
            }
            return roleMappings;
        }
        Stream scopeMappings = client.getRolesStream();
        Stream clientScopesMappings = !logger.isTraceEnabled() ? clientScopes.flatMap(clientScope -> clientScope.getScopeMappingsStream()) : clientScopes.flatMap(clientScope -> {
            logger.tracef("Adding client scope role mappings of client scope '%s' to client '%s'", (Object)clientScope.getName(), (Object)client.getClientId());
            return clientScope.getScopeMappingsStream();
        });
        scopeMappings = Stream.concat(scopeMappings, clientScopesMappings);
        scopeMappings = RoleUtils.expandCompositeRolesStream(scopeMappings);
        roleMappings.retainAll(scopeMappings.collect(Collectors.toSet()));
        return roleMappings;
    }

    public static Stream<ClientScopeModel> getRequestedClientScopes(String scopeParam, ClientModel client) {
        Stream<ClientModel> clientScopes = Stream.concat(client.getClientScopes(true).values().stream(), Stream.of(client)).distinct();
        if (scopeParam == null) {
            return clientScopes;
        }
        Map allOptionalScopes = client.getClientScopes(false);
        return Stream.concat(TokenManager.parseScopeParameter(scopeParam).map(allOptionalScopes::get).filter(Objects::nonNull), clientScopes).distinct();
    }

    public static boolean isValidScope(String scopes, AuthorizationRequestContext authorizationRequestContext, ClientModel client) {
        if (scopes == null) {
            return true;
        }
        Collection requestedScopes = TokenManager.parseScopeParameter(scopes).collect(Collectors.toSet());
        Set rarScopes = authorizationRequestContext.getAuthorizationDetailEntries().stream().map(AuthorizationDetails::getAuthorizationDetails).map(AuthorizationDetailsJSONRepresentation::getScopeNameFromCustomData).collect(Collectors.toSet());
        if (TokenUtil.isOIDCRequest((String)scopes)) {
            requestedScopes.remove("openid");
        }
        if ((authorizationRequestContext.getAuthorizationDetailEntries() == null || authorizationRequestContext.getAuthorizationDetailEntries().isEmpty()) && requestedScopes.size() > 0) {
            return false;
        }
        if (logger.isTraceEnabled()) {
            logger.tracef("Rar scopes to validate requested scopes against: %1s", (Object)String.join((CharSequence)" ", rarScopes));
            logger.tracef("Requested scopes: %1s", (Object)String.join((CharSequence)" ", requestedScopes));
        }
        for (String requestedScope : requestedScopes) {
            if (rarScopes.contains(requestedScope) || client.getDynamicClientScope(requestedScope) != null) continue;
            return false;
        }
        return true;
    }

    public static boolean isValidScope(String scopes, ClientModel client) {
        if (scopes == null) {
            return true;
        }
        Set clientScopes = TokenManager.getRequestedClientScopes(scopes, client).filter(((Predicate<ClientScopeModel>)ClientModel.class::isInstance).negate()).map(ClientScopeModel::getName).collect(Collectors.toSet());
        Collection requestedScopes = TokenManager.parseScopeParameter(scopes).collect(Collectors.toSet());
        if (TokenUtil.isOIDCRequest((String)scopes)) {
            requestedScopes.remove("openid");
        }
        if (!requestedScopes.isEmpty() && clientScopes.isEmpty()) {
            return false;
        }
        for (String requestedScope : requestedScopes) {
            if (clientScopes.contains(requestedScope) || client.getDynamicClientScope(requestedScope) != null) continue;
            return false;
        }
        return true;
    }

    public static Stream<String> parseScopeParameter(String scopeParam) {
        return Arrays.stream(scopeParam.split(" ")).distinct();
    }

    public static boolean verifyConsentStillAvailable(KeycloakSession session, UserModel user, ClientModel client, Stream<ClientScopeModel> requestedClientScopes) {
        if (!client.isConsentRequired()) {
            return true;
        }
        UserConsentModel grantedConsent = UserConsentManager.getConsentByClient(session, client.getRealm(), user, client.getId());
        return requestedClientScopes.filter(ClientScopeModel::isDisplayOnConsentScreen).noneMatch(requestedScope -> {
            if (grantedConsent == null || !grantedConsent.getGrantedClientScopes().contains(requestedScope)) {
                logger.debugf("Client '%s' no longer has requested consent from user '%s' for client scope '%s'", (Object)client.getClientId(), (Object)user.getUsername(), (Object)requestedScope.getName());
                return true;
            }
            return false;
        });
    }

    public AccessToken transformAccessToken(final KeycloakSession session, AccessToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof OIDCAccessTokenMapper).collect(new TokenCollector<AccessToken>(token){

            @Override
            protected AccessToken applyMapper(AccessToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((OIDCAccessTokenMapper)mapper.getValue()).transformAccessToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public AccessTokenResponse transformAccessTokenResponse(final KeycloakSession session, AccessTokenResponse accessTokenResponse, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof OIDCAccessTokenResponseMapper).collect(new TokenCollector<AccessTokenResponse>(accessTokenResponse){

            @Override
            protected AccessTokenResponse applyMapper(AccessTokenResponse token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((OIDCAccessTokenResponseMapper)mapper.getValue()).transformAccessTokenResponse(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public AccessToken transformUserInfoAccessToken(final KeycloakSession session, AccessToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof UserInfoTokenMapper).collect(new TokenCollector<AccessToken>(token){

            @Override
            protected AccessToken applyMapper(AccessToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((UserInfoTokenMapper)mapper.getValue()).transformUserInfoToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public AccessToken transformIntrospectionAccessToken(final KeycloakSession session, AccessToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof TokenIntrospectionTokenMapper).collect(new TokenCollector<AccessToken>(token){

            @Override
            protected AccessToken applyMapper(AccessToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((TokenIntrospectionTokenMapper)mapper.getValue()).transformIntrospectionToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    public Map<String, Object> generateUserInfoClaims(AccessToken userInfo, UserModel userModel) {
        HashMap<String, Object> claims = new HashMap<String, Object>();
        claims.put("sub", userInfo.getSubject() == null ? userModel.getId() : userInfo.getSubject());
        if (userInfo.getIssuer() != null) {
            claims.put("iss", userInfo.getIssuer());
        }
        if (userInfo.getAudience() != null) {
            claims.put("aud", userInfo.getAudience());
        }
        if (userInfo.getName() != null) {
            claims.put("name", userInfo.getName());
        }
        if (userInfo.getGivenName() != null) {
            claims.put("given_name", userInfo.getGivenName());
        }
        if (userInfo.getFamilyName() != null) {
            claims.put("family_name", userInfo.getFamilyName());
        }
        if (userInfo.getMiddleName() != null) {
            claims.put("middle_name", userInfo.getMiddleName());
        }
        if (userInfo.getNickName() != null) {
            claims.put("nickname", userInfo.getNickName());
        }
        if (userInfo.getPreferredUsername() != null) {
            claims.put("preferred_username", userInfo.getPreferredUsername());
        }
        if (userInfo.getProfile() != null) {
            claims.put("profile", userInfo.getProfile());
        }
        if (userInfo.getPicture() != null) {
            claims.put("picture", userInfo.getPicture());
        }
        if (userInfo.getWebsite() != null) {
            claims.put("website", userInfo.getWebsite());
        }
        if (userInfo.getEmail() != null) {
            claims.put("email", userInfo.getEmail());
        }
        if (userInfo.getEmailVerified() != null) {
            claims.put("email_verified", userInfo.getEmailVerified());
        }
        if (userInfo.getGender() != null) {
            claims.put("gender", userInfo.getGender());
        }
        if (userInfo.getBirthdate() != null) {
            claims.put("birthdate", userInfo.getBirthdate());
        }
        if (userInfo.getZoneinfo() != null) {
            claims.put("zoneinfo", userInfo.getZoneinfo());
        }
        if (userInfo.getLocale() != null) {
            claims.put("locale", userInfo.getLocale());
        }
        if (userInfo.getPhoneNumber() != null) {
            claims.put("phone_number", userInfo.getPhoneNumber());
        }
        if (userInfo.getPhoneNumberVerified() != null) {
            claims.put("phone_number_verified", userInfo.getPhoneNumberVerified());
        }
        if (userInfo.getAddress() != null) {
            claims.put("address", userInfo.getAddress());
        }
        if (userInfo.getUpdatedAt() != null) {
            claims.put("updated_at", userInfo.getUpdatedAt());
        }
        if (userInfo.getClaimsLocales() != null) {
            claims.put("claims_locales", userInfo.getClaimsLocales());
        }
        claims.putAll(userInfo.getOtherClaims());
        if (userInfo.getRealmAccess() != null) {
            HashMap<String, Set> realmAccess = new HashMap<String, Set>();
            realmAccess.put("roles", userInfo.getRealmAccess().getRoles());
            claims.put("realm_access", realmAccess);
        }
        if (userInfo.getResourceAccess() != null && !userInfo.getResourceAccess().isEmpty()) {
            HashMap resourceAccessMap = new HashMap();
            for (Map.Entry resourceAccessMapEntry : userInfo.getResourceAccess().entrySet()) {
                HashMap<String, Set> resourceAccess = new HashMap<String, Set>();
                resourceAccess.put("roles", ((AccessToken.Access)resourceAccessMapEntry.getValue()).getRoles());
                resourceAccessMap.put((String)resourceAccessMapEntry.getKey(), resourceAccess);
            }
            claims.put("resource_access", resourceAccessMap);
        }
        return claims;
    }

    public IDToken transformIDToken(final KeycloakSession session, IDToken token, final UserSessionModel userSession, final ClientSessionContext clientSessionCtx) {
        return ProtocolMapperUtils.getSortedProtocolMappers(session, clientSessionCtx, mapper -> mapper.getValue() instanceof OIDCIDTokenMapper).collect(new TokenCollector<IDToken>(token){

            @Override
            protected IDToken applyMapper(IDToken token, Map.Entry<ProtocolMapperModel, ProtocolMapper> mapper) {
                return ((OIDCIDTokenMapper)mapper.getValue()).transformIDToken(token, mapper.getKey(), session, userSession, clientSessionCtx);
            }
        });
    }

    protected AccessToken initToken(RealmModel realm, ClientModel client, UserModel user, UserSessionModel session, ClientSessionContext clientSessionCtx, UriInfo uriInfo) {
        String authTime;
        AccessToken token = new AccessToken();
        token.id(KeycloakModelUtils.generateId());
        token.type("Bearer");
        token.subject(user.getId());
        token.issuedNow();
        token.issuedFor(client.getClientId());
        AuthenticatedClientSessionModel clientSession = clientSessionCtx.getClientSession();
        token.issuer(clientSession.getNote("iss"));
        token.setNonce((String)clientSessionCtx.getAttribute("nonce", String.class));
        token.setScope(clientSessionCtx.getScopeString());
        if (!Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.STEP_UP_AUTHENTICATION)) {
            String acr = AuthenticationManager.isSSOAuthentication(clientSession) ? "0" : "1";
            token.setAcr(acr);
        }
        if ((authTime = session.getNote("AUTH_TIME")) != null) {
            token.setAuthTime(Integer.parseInt(authTime));
        }
        token.setSessionState(session.getId());
        ClientScopeModel offlineAccessScope = KeycloakModelUtils.getClientScopeByName((RealmModel)realm, (String)"offline_access");
        boolean offlineTokenRequested = offlineAccessScope == null ? false : clientSessionCtx.getClientScopeIds().contains(offlineAccessScope.getId());
        token.expiration(this.getTokenExpiration(realm, client, session, clientSession, offlineTokenRequested));
        return token;
    }

    private int getTokenExpiration(RealmModel realm, ClientModel client, UserSessionModel userSession, AuthenticatedClientSessionModel clientSession, boolean offlineTokenRequested) {
        String clientLifespan;
        boolean implicitFlow = false;
        String responseType = clientSession.getNote("response_type");
        if (responseType != null) {
            implicitFlow = OIDCResponseType.parse(responseType).isImplicitFlow();
        }
        int tokenLifespan = implicitFlow ? realm.getAccessTokenLifespanForImplicitFlow() : ((clientLifespan = client.getAttribute("access.token.lifespan")) != null && !clientLifespan.trim().isEmpty() ? Integer.parseInt(clientLifespan) : realm.getAccessTokenLifespan());
        long expiration = tokenLifespan == -1 ? TimeUnit.SECONDS.toMillis(userSession.getStarted() + (userSession.isRememberMe() && realm.getSsoSessionMaxLifespanRememberMe() > 0 ? realm.getSsoSessionMaxLifespanRememberMe() : realm.getSsoSessionMaxLifespan())) : Time.currentTimeMillis() + TimeUnit.SECONDS.toMillis(tokenLifespan);
        long sessionExpires = SessionExpirationUtils.calculateClientSessionMaxLifespanTimestamp((userSession.isOffline() || offlineTokenRequested ? 1 : 0) != 0, (boolean)userSession.isRememberMe(), (long)TimeUnit.SECONDS.toMillis(clientSession.getStarted()), (long)TimeUnit.SECONDS.toMillis(userSession.getStarted()), (RealmModel)realm, (ClientModel)client);
        expiration = sessionExpires > 0L ? Math.min(expiration, sessionExpires) : expiration;
        return (int)TimeUnit.MILLISECONDS.toSeconds(expiration);
    }

    public AccessTokenResponseBuilder responseBuilder(RealmModel realm, ClientModel client, EventBuilder event, KeycloakSession session, UserSessionModel userSession, ClientSessionContext clientSessionCtx) {
        return new AccessTokenResponseBuilder(realm, client, event, session, userSession, clientSessionCtx);
    }

    private String formatTokenType(ClientModel client) {
        if (OIDCAdvancedConfigWrapper.fromClientModel(client).isUseLowerCaseInTokenResponse()) {
            return "Bearer".toLowerCase();
        }
        return "Bearer";
    }

    public LogoutTokenValidationCode verifyLogoutToken(KeycloakSession session, RealmModel realm, String encodedLogoutToken) {
        Optional<LogoutToken> logoutTokenOptional = this.toLogoutToken(encodedLogoutToken);
        if (!logoutTokenOptional.isPresent()) {
            return LogoutTokenValidationCode.DECODE_TOKEN_FAILED;
        }
        LogoutToken logoutToken = logoutTokenOptional.get();
        List identityProviders = this.getOIDCIdentityProviders(realm, session).collect(Collectors.toList());
        if (identityProviders.isEmpty()) {
            return LogoutTokenValidationCode.COULD_NOT_FIND_IDP;
        }
        Stream<OIDCIdentityProvider> validOidcIdentityProviders = this.validateLogoutTokenAgainstIdpProvider(identityProviders.stream(), encodedLogoutToken, logoutToken);
        if (validOidcIdentityProviders.count() == 0L) {
            return LogoutTokenValidationCode.TOKEN_VERIFICATION_WITH_IDP_FAILED;
        }
        if (logoutToken.getSubject() == null && logoutToken.getSid() == null) {
            return LogoutTokenValidationCode.MISSING_SID_OR_SUBJECT;
        }
        if (!this.checkLogoutTokenForEvents(logoutToken)) {
            return LogoutTokenValidationCode.BACKCHANNEL_LOGOUT_EVENT_MISSING;
        }
        if (logoutToken.getOtherClaims().get("nonce") != null) {
            return LogoutTokenValidationCode.NONCE_CLAIM_IN_TOKEN;
        }
        if (logoutToken.getId() == null) {
            return LogoutTokenValidationCode.LOGOUT_TOKEN_ID_MISSING;
        }
        if (logoutToken.getIat() == null) {
            return LogoutTokenValidationCode.MISSING_IAT_CLAIM;
        }
        return LogoutTokenValidationCode.VALIDATION_SUCCESS;
    }

    public Optional<LogoutToken> toLogoutToken(String encodedLogoutToken) {
        try {
            JWSInput jws = new JWSInput(encodedLogoutToken);
            return Optional.of((LogoutToken)jws.readJsonContent(LogoutToken.class));
        }
        catch (JWSInputException e) {
            return Optional.empty();
        }
    }

    public Stream<OIDCIdentityProvider> getValidOIDCIdentityProvidersForBackchannelLogout(RealmModel realm, KeycloakSession session, String encodedLogoutToken, LogoutToken logoutToken) {
        return this.validateLogoutTokenAgainstIdpProvider(this.getOIDCIdentityProviders(realm, session), encodedLogoutToken, logoutToken);
    }

    public Stream<OIDCIdentityProvider> validateLogoutTokenAgainstIdpProvider(Stream<OIDCIdentityProvider> oidcIdps, String encodedLogoutToken, LogoutToken logoutToken) {
        return oidcIdps.filter(oidcIdp -> ((OIDCIdentityProviderConfig)oidcIdp.getConfig()).getIssuer() != null).filter(oidcIdp -> oidcIdp.isIssuer(logoutToken.getIssuer(), null)).filter(oidcIdp -> {
            try {
                oidcIdp.validateToken(encodedLogoutToken);
                return true;
            }
            catch (IdentityBrokerException e) {
                logger.debugf("LogoutToken verification with identity provider failed", (Object)e.getMessage());
                return false;
            }
        });
    }

    private Stream<OIDCIdentityProvider> getOIDCIdentityProviders(RealmModel realm, KeycloakSession session) {
        try {
            return realm.getIdentityProvidersStream().map(idpModel -> IdentityBrokerService.getIdentityProviderFactory(session, idpModel).create(session, idpModel)).filter(OIDCIdentityProvider.class::isInstance).map(OIDCIdentityProvider.class::cast);
        }
        catch (IdentityBrokerException e) {
            logger.warnf("LogoutToken verification with identity provider failed", (Object)e.getMessage());
            return Stream.empty();
        }
    }

    private boolean checkLogoutTokenForEvents(LogoutToken logoutToken) {
        for (String eventKey : logoutToken.getEvents().keySet()) {
            if (!"http://schemas.openid.net/event/backchannel-logout".equals(eventKey)) continue;
            return true;
        }
        return false;
    }

    public static class NotBeforeCheck
    implements TokenVerifier.Predicate<JsonWebToken> {
        private final int notBefore;

        public NotBeforeCheck(int notBefore) {
            this.notBefore = notBefore;
        }

        public boolean test(JsonWebToken t) throws VerificationException {
            if (t.getIssuedAt() < this.notBefore) {
                throw new VerificationException("Stale token");
            }
            return true;
        }

        public static NotBeforeCheck forModel(ClientModel clientModel) {
            if (clientModel != null) {
                int notBeforeClient = clientModel.getNotBefore();
                int notBeforeRealm = clientModel.getRealm().getNotBefore();
                int notBefore = notBeforeClient == 0 ? notBeforeRealm : (notBeforeRealm == 0 ? notBeforeClient : Math.min(notBeforeClient, notBeforeRealm));
                return new NotBeforeCheck(notBefore);
            }
            return new NotBeforeCheck(0);
        }

        public static NotBeforeCheck forModel(RealmModel realmModel) {
            return new NotBeforeCheck(realmModel == null ? 0 : realmModel.getNotBefore());
        }

        public static NotBeforeCheck forModel(KeycloakSession session, RealmModel realmModel, UserModel userModel) {
            return LightweightUserAdapter.isLightweightUser((UserModel)userModel) ? new NotBeforeCheck((int)(((LightweightUserAdapter)userModel).getCreatedTimestamp() / 1000L)) : new NotBeforeCheck(session.users().getNotBeforeOfUser(realmModel, userModel));
        }
    }

    public static class TokenValidation {
        public final UserModel user;
        public final UserSessionModel userSession;
        public final ClientSessionContext clientSessionCtx;
        public final AccessToken newToken;

        public TokenValidation(UserModel user, UserSessionModel userSession, ClientSessionContext clientSessionCtx, AccessToken newToken) {
            this.user = user;
            this.userSession = userSession;
            this.clientSessionCtx = clientSessionCtx;
            this.newToken = newToken;
        }
    }

    public static class TokenRevocationCheck
    implements TokenVerifier.Predicate<JsonWebToken> {
        private final KeycloakSession session;

        public TokenRevocationCheck(KeycloakSession session) {
            this.session = session;
        }

        public boolean test(JsonWebToken token) {
            SingleUseObjectProvider singleUseStore = this.session.singleUseObjects();
            return !singleUseStore.contains(token.getId() + ".revoked");
        }
    }

    public class AccessTokenResponseBuilder {
        RealmModel realm;
        ClientModel client;
        EventBuilder event;
        KeycloakSession session;
        UserSessionModel userSession;
        ClientSessionContext clientSessionCtx;
        AccessToken accessToken;
        RefreshToken refreshToken;
        IDToken idToken;
        String responseTokenType;
        boolean generateAccessTokenHash = false;
        String codeHash;
        String stateHash;
        private AccessTokenResponse response;

        public AccessTokenResponseBuilder(RealmModel realm, ClientModel client, EventBuilder event, KeycloakSession session, UserSessionModel userSession, ClientSessionContext clientSessionCtx) {
            this.realm = realm;
            this.client = client;
            this.event = event;
            this.session = session;
            this.userSession = userSession;
            this.clientSessionCtx = clientSessionCtx;
            this.responseTokenType = TokenManager.this.formatTokenType(client);
        }

        public AccessToken getAccessToken() {
            return this.accessToken;
        }

        public RefreshToken getRefreshToken() {
            return this.refreshToken;
        }

        public IDToken getIdToken() {
            return this.idToken;
        }

        public AccessTokenResponseBuilder accessToken(AccessToken accessToken) {
            this.accessToken = accessToken;
            return this;
        }

        public AccessTokenResponseBuilder refreshToken(RefreshToken refreshToken) {
            this.refreshToken = refreshToken;
            return this;
        }

        public AccessTokenResponseBuilder responseTokenType(String responseTokenType) {
            this.responseTokenType = responseTokenType;
            return this;
        }

        public AccessTokenResponseBuilder generateAccessToken() {
            UserModel user = this.userSession.getUser();
            this.accessToken = TokenManager.this.createClientAccessToken(this.session, this.realm, this.client, user, this.userSession, this.clientSessionCtx);
            return this;
        }

        public AccessTokenResponseBuilder generateRefreshToken() {
            if (this.accessToken == null) {
                throw new IllegalStateException("accessToken not set");
            }
            ClientScopeModel offlineAccessScope = KeycloakModelUtils.getClientScopeByName((RealmModel)this.realm, (String)"offline_access");
            boolean offlineTokenRequested = offlineAccessScope == null ? false : this.clientSessionCtx.getClientScopeIds().contains(offlineAccessScope.getId());
            this.generateRefreshToken(offlineTokenRequested);
            return this;
        }

        public AccessTokenResponseBuilder generateRefreshToken(String scope) {
            if (this.accessToken == null) {
                throw new IllegalStateException("accessToken not set");
            }
            boolean offlineTokenRequested = Arrays.asList(scope.split(" ")).contains("offline_access");
            this.generateRefreshToken(offlineTokenRequested);
            this.refreshToken.setScope(scope);
            return this;
        }

        private void generateRefreshToken(boolean offlineTokenRequested) {
            this.refreshToken = new RefreshToken(this.accessToken);
            this.refreshToken.id(KeycloakModelUtils.generateId());
            this.refreshToken.issuedNow();
            int currentTime = Time.currentTime();
            AuthenticatedClientSessionModel clientSession = this.clientSessionCtx.getClientSession();
            clientSession.setTimestamp(currentTime);
            UserSessionModel userSession = clientSession.getUserSession();
            userSession.setLastSessionRefresh(currentTime);
            if (offlineTokenRequested) {
                UserSessionManager sessionManager = new UserSessionManager(this.session);
                if (!sessionManager.isOfflineTokenAllowed(this.clientSessionCtx)) {
                    this.event.error("not_allowed");
                    throw new ErrorResponseException("not_allowed", "Offline tokens not allowed for the user or client", Response.Status.BAD_REQUEST);
                }
                this.refreshToken.type("Offline");
                if (this.realm.isOfflineSessionMaxLifespanEnabled()) {
                    this.refreshToken.expiration(this.getExpiration(true));
                }
                sessionManager.createOrUpdateOfflineSession(this.clientSessionCtx.getClientSession(), userSession);
            } else {
                this.refreshToken.expiration(this.getExpiration(false));
            }
        }

        private int getExpiration(boolean offline) {
            long expiration = SessionExpirationUtils.calculateClientSessionIdleTimestamp((boolean)offline, (boolean)this.userSession.isRememberMe(), (long)TimeUnit.SECONDS.toMillis(this.clientSessionCtx.getClientSession().getTimestamp()), (RealmModel)this.realm, (ClientModel)this.client);
            long lifespan = SessionExpirationUtils.calculateClientSessionMaxLifespanTimestamp((boolean)offline, (boolean)this.userSession.isRememberMe(), (long)TimeUnit.SECONDS.toMillis(this.clientSessionCtx.getClientSession().getStarted()), (long)TimeUnit.SECONDS.toMillis(this.userSession.getStarted()), (RealmModel)this.realm, (ClientModel)this.client);
            expiration = lifespan > 0L ? Math.min(expiration, lifespan) : expiration;
            return (int)TimeUnit.MILLISECONDS.toSeconds(expiration);
        }

        public AccessTokenResponseBuilder generateIDToken() {
            return this.generateIDToken(false);
        }

        public AccessTokenResponseBuilder generateIDToken(boolean isIdTokenAsDetachedSignature) {
            if (this.accessToken == null) {
                throw new IllegalStateException("accessToken not set");
            }
            this.idToken = new IDToken();
            this.idToken.id(KeycloakModelUtils.generateId());
            this.idToken.type("ID");
            this.idToken.subject(this.accessToken.getSubject());
            this.idToken.audience(new String[]{this.client.getClientId()});
            this.idToken.issuedNow();
            this.idToken.issuedFor(this.accessToken.getIssuedFor());
            this.idToken.issuer(this.accessToken.getIssuer());
            this.idToken.setNonce(this.accessToken.getNonce());
            this.idToken.setAuthTime(this.accessToken.getAuthTime());
            this.idToken.setSessionState(this.accessToken.getSessionState());
            this.idToken.expiration(this.accessToken.getExpiration());
            if (!Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.STEP_UP_AUTHENTICATION)) {
                this.idToken.setAcr(this.accessToken.getAcr());
            }
            if (!isIdTokenAsDetachedSignature) {
                this.idToken = TokenManager.this.transformIDToken(this.session, this.idToken, this.userSession, this.clientSessionCtx);
            }
            return this;
        }

        public AccessTokenResponseBuilder generateAccessTokenHash() {
            this.generateAccessTokenHash = true;
            return this;
        }

        public AccessTokenResponseBuilder generateCodeHash(String code) {
            this.codeHash = this.generateOIDCHash(code);
            return this;
        }

        public AccessTokenResponseBuilder generateStateHash(String state) {
            this.stateHash = this.generateOIDCHash(state);
            return this;
        }

        public boolean isOfflineToken() {
            return this.refreshToken != null && "Offline".equals(this.refreshToken.getType());
        }

        public AccessTokenResponse build() {
            int userNotBefore;
            UserModel user;
            String encodedToken;
            if (this.response != null) {
                return this.response;
            }
            if (this.accessToken != null) {
                this.event.detail("token_id", this.accessToken.getId());
            }
            if (this.refreshToken != null) {
                if (this.event.getEvent().getDetails().containsKey("refresh_token_id")) {
                    this.event.detail("updated_refresh_token_id", this.refreshToken.getId());
                } else {
                    this.event.detail("refresh_token_id", this.refreshToken.getId());
                }
                this.event.detail("refresh_token_type", this.refreshToken.getType());
            }
            AccessTokenResponse res = new AccessTokenResponse();
            if (this.accessToken != null) {
                encodedToken = this.session.tokens().encode((Token)this.accessToken);
                res.setToken(encodedToken);
                res.setTokenType(this.responseTokenType);
                res.setSessionState(this.accessToken.getSessionState());
                if (this.accessToken.getExpiration() != 0) {
                    res.setExpiresIn((long)(this.accessToken.getExpiration() - Time.currentTime()));
                }
            }
            if (this.generateAccessTokenHash) {
                String atHash = this.generateOIDCHash(res.getToken());
                this.idToken.setAccessTokenHash(atHash);
            }
            if (this.codeHash != null) {
                this.idToken.setCodeHash(this.codeHash);
            }
            if (this.stateHash != null) {
                this.idToken.setStateHash(this.stateHash);
            }
            if (this.idToken != null) {
                encodedToken = this.session.tokens().encodeAndEncrypt((Token)this.idToken);
                res.setIdToken(encodedToken);
            }
            if (this.refreshToken != null) {
                encodedToken = this.session.tokens().encode((Token)this.refreshToken);
                res.setRefreshToken(encodedToken);
                if (this.refreshToken.getExpiration() != 0) {
                    res.setRefreshExpiresIn((long)(this.refreshToken.getExpiration() - Time.currentTime()));
                }
            }
            int notBefore = this.realm.getNotBefore();
            if (this.client.getNotBefore() > notBefore) {
                notBefore = this.client.getNotBefore();
            }
            if (!LightweightUserAdapter.isLightweightUser((UserModel)(user = this.userSession.getUser())) && (userNotBefore = this.session.users().getNotBeforeOfUser(this.realm, user)) > notBefore) {
                notBefore = userNotBefore;
            }
            res.setNotBeforePolicy(notBefore);
            res = TokenManager.this.transformAccessTokenResponse(this.session, res, this.userSession, this.clientSessionCtx);
            String responseScope = this.clientSessionCtx.getScopeString();
            res.setScope(responseScope);
            this.event.detail("scope", responseScope);
            this.response = res;
            return this.response;
        }

        private String generateOIDCHash(String input) {
            String signatureAlgorithm = this.session.tokens().signatureAlgorithm(TokenCategory.ID);
            SignatureProvider signatureProvider = (SignatureProvider)this.session.getProvider(SignatureProvider.class, signatureAlgorithm);
            String hashAlgorithm = signatureProvider.signer().getHashAlgorithm();
            HashProvider hashProvider = (HashProvider)this.session.getProvider(HashProvider.class, hashAlgorithm);
            byte[] hash = hashProvider.hash(input);
            return HashUtils.encodeHashToOIDC((byte[])hash);
        }
    }

    private static abstract class TokenCollector<T>
    implements Collector<Map.Entry<ProtocolMapperModel, ProtocolMapper>, TokenCollector<T>, T> {
        private T token;

        public TokenCollector(T token) {
            this.token = token;
        }

        @Override
        public Supplier<TokenCollector<T>> supplier() {
            return () -> this;
        }

        @Override
        public Function<TokenCollector<T>, T> finisher() {
            return idTokenWrapper -> idTokenWrapper.token;
        }

        @Override
        public Set<Collector.Characteristics> characteristics() {
            return Collections.emptySet();
        }

        @Override
        public BinaryOperator<TokenCollector<T>> combiner() {
            return (tMutableWrapper, tMutableWrapper2) -> {
                throw new IllegalStateException("can't combine");
            };
        }

        @Override
        public BiConsumer<TokenCollector<T>, Map.Entry<ProtocolMapperModel, ProtocolMapper>> accumulator() {
            return (idToken, mapper) -> {
                idToken.token = this.applyMapper(idToken.token, (Map.Entry<ProtocolMapperModel, ProtocolMapper>)mapper);
            };
        }

        protected abstract T applyMapper(T var1, Map.Entry<ProtocolMapperModel, ProtocolMapper> var2);
    }
}

