package liquibase.database;

import liquibase.exception.ValidationErrors;
import liquibase.util.StringUtil;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

public class DatabaseList {
    /**
     * Compares a given database to a database definition string.
     * <p></p>
     * Definition syntax: Comma separated list of database shortNames. Doesn't matter if it includes spaces.
     * Can exclude a database by prepending its name with a '!'.
     * The string "all" will match all databases. The string "none" will match no databases, even if others are listed.
     * If an empty or null definition or null is passed, it will return the passed returnValueIfEmpty value.
     */
    public static boolean definitionMatches(String definition, String databaseShortName, boolean returnValueIfEmpty) {
        return definitionMatches(StringUtil.splitAndTrim(StringUtil.trimToNull(definition), ","), databaseShortName, returnValueIfEmpty);
    }

    public static boolean definitionMatches(String definition, Database database, boolean returnValueIfEmpty) {
        return definitionMatches(StringUtil.splitAndTrim(StringUtil.trimToNull(definition), ","), database, returnValueIfEmpty);
    }

    /**
     * Same logic as {@link #definitionMatches(String, liquibase.database.Database, boolean)} but with a collection of definitions rather than a comma separated list.
     */
    public static boolean definitionMatches(Collection<String> definition, String databaseShortName, boolean returnValueIfEmptyList) {
        if ((definition == null) || definition.isEmpty()) {
            return returnValueIfEmptyList;
        }

        if (definition.contains("all")) {
            return true;
        }

        if (definition.contains("none")) {
            return false;
        }

        // !h2 would mean that the h2 database should be excluded
        if (definition.contains("!" + databaseShortName)) {
            return false;
        }

        Set<String> dbmsSupported = new HashSet<>();
        // add all dbms that do not start with ! to a list
        for (String dbms: definition) {
            if (!dbms.startsWith("!")) {
                dbmsSupported.add(dbms);
            }
        }


        return dbmsSupported.isEmpty() || dbmsSupported.contains(databaseShortName);

    }

    public static boolean definitionMatches(Collection<String> definition, Database database, boolean returnValueIfEmptyList) {
        String shortName;
        if (database == null) {
            shortName = "null";
        } else {
            shortName = database.getShortName();
        }
        return definitionMatches(definition, shortName, returnValueIfEmptyList);
    }

    public static Set<String> toDbmsSet(String dbmsList) {
        Set<String> dbmsSet = null;
        if (StringUtil.trimToNull(dbmsList) != null) {
            dbmsSet = new HashSet<>();
            for (String string : dbmsList.toLowerCase().split(",")) {
                dbmsSet.add(string.trim());
            }
        }
        return dbmsSet;
    }

    /**
     * This method will validate whether a dbms is valid and match with supported database, if it doesn't then
     * will add a validation error for it.
     * @param definition
     * @param vErrors
     */
    public static void validateDefinitions(String definition, ValidationErrors vErrors) {
        if(!definition.contentEquals("none") && !definition.contentEquals("all") && !definition.startsWith("!")) {
            Database database = DatabaseFactory.getInstance().getDatabase(definition.toLowerCase());
            if (database == null) {
                vErrors.addError(String.format("%s is not a supported DB", definition));
            }
        }
    }

    /**
     * This method will validate if a set of definitions/dbms are valid supported DBs.
     * @param definitions
     * @param vErrors
     */
    public static void validateDefinitions(Collection<String> definitions, ValidationErrors vErrors) {
        definitions.stream().forEach( definition -> validateDefinitions(definition, vErrors));
    }
}
