/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.kafka.connect.source;

import com.mongodb.MongoNamespace;
import com.mongodb.client.MongoClient;
import com.mongodb.kafka.connect.source.MongoSourceConfig;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.kafka.connect.errors.ConnectException;
import org.bson.BsonDocument;
import org.bson.conversions.Bson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class MongoCopyDataManager
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(MongoCopyDataManager.class);
    private volatile boolean closed;
    private volatile Exception errorException;
    private final AtomicInteger namespacesToCopy;
    private final MongoSourceConfig sourceConfig;
    private final MongoClient mongoClient;
    private final ExecutorService executor;
    private final ArrayBlockingQueue<BsonDocument> queue;

    MongoCopyDataManager(MongoSourceConfig sourceConfig, MongoClient mongoClient) {
        this.sourceConfig = sourceConfig;
        this.mongoClient = mongoClient;
        String database = sourceConfig.getString("database");
        String collection = sourceConfig.getString("collection");
        List<MongoNamespace> namespaces = database.isEmpty() ? MongoCopyDataManager.getCollections(mongoClient) : (collection.isEmpty() ? MongoCopyDataManager.getCollections(mongoClient, database) : Collections.singletonList(MongoCopyDataManager.createNamespace(database, collection)));
        LOGGER.info("Copying existing data on the following namespaces: {}", namespaces);
        this.namespacesToCopy = new AtomicInteger(namespaces.size());
        this.queue = new ArrayBlockingQueue(sourceConfig.getInt("copy.existing.queue.size"));
        this.executor = Executors.newFixedThreadPool(Math.min(this.namespacesToCopy.get(), sourceConfig.getInt("copy.existing.max.threads")));
        namespaces.forEach(n -> this.executor.submit(() -> this.copyDataFrom((MongoNamespace)n)));
    }

    Optional<BsonDocument> poll() {
        if (this.errorException != null) {
            if (!this.closed) {
                this.close();
            }
            throw new ConnectException((Throwable)this.errorException);
        }
        if (this.namespacesToCopy.get() == 0) {
            this.close();
        }
        return Optional.ofNullable(this.queue.poll());
    }

    boolean isCopying() {
        return this.namespacesToCopy.get() > 0 || !this.queue.isEmpty();
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closed = true;
            LOGGER.debug("Shutting down executors");
            this.executor.shutdownNow();
        }
    }

    private void copyDataFrom(MongoNamespace namespace) {
        LOGGER.debug("Copying existing data from: {}", (Object)namespace.getFullName());
        try {
            this.mongoClient.getDatabase(namespace.getDatabaseName()).getCollection(namespace.getCollectionName(), BsonDocument.class).aggregate(this.createPipeline(namespace)).forEach(this.queue::add);
            this.namespacesToCopy.decrementAndGet();
        }
        catch (Exception e) {
            this.errorException = e;
        }
    }

    private List<Bson> createPipeline(MongoNamespace namespace) {
        ArrayList<Bson> pipeline = new ArrayList<Bson>();
        pipeline.add(BsonDocument.parse("{$replaceRoot: {newRoot: {_id: {_id: '$_id', copyingData: true}, operationType: 'insert', ns: {db: '" + namespace.getDatabaseName() + "', coll: '" + namespace.getCollectionName() + "'}, documentKey: {_id: '$_id'}, fullDocument: '$$ROOT'}}}"));
        this.sourceConfig.getPipeline().map(pipeline::addAll);
        return pipeline;
    }

    private static List<MongoNamespace> getCollections(MongoClient mongoClient) {
        return mongoClient.listDatabaseNames().into(new ArrayList()).stream().filter(s -> !s.startsWith("admin") && !s.startsWith("config") && !s.startsWith("local")).map(d -> MongoCopyDataManager.getCollections(mongoClient, d)).flatMap(Collection::stream).collect(Collectors.toList());
    }

    private static List<MongoNamespace> getCollections(MongoClient mongoClient, String database) {
        return mongoClient.getDatabase(database).listCollectionNames().into(new ArrayList()).stream().filter(s -> !s.startsWith("system.")).map(c -> MongoCopyDataManager.createNamespace(database, c)).collect(Collectors.toList());
    }

    private static MongoNamespace createNamespace(String database, String collection) {
        return new MongoNamespace(database, collection);
    }
}

