import { ChangeAction } from "../../../utils/textChanges.js";
import { getNextPropertyPointer, getPreviousPropertyPointer } from "../parser.js";
import { JsonFix } from "./JsonFix.js";
export default class RemoveJsonPropertyFix extends JsonFix {
    constructor(options) {
        super();
        this.calculatePositions(options.key, options.pointers, options.removeEmptyDirectParent);
    }
    calculatePositions(key, pointers, removeParent = false) {
        const currentPointer = pointers[key];
        if (!currentPointer) {
            throw new Error(`Cannot find JSON pointer: '${key}'`);
        }
        if (!currentPointer.key) {
            throw new Error(`Unsupported removal of non-property value: '${key}'`);
        }
        const previousPointer = getPreviousPropertyPointer(pointers, currentPointer, key);
        if (previousPointer) {
            // Start removal from end of previous property to include the comma
            this.startPos = previousPointer.valueEnd.pos;
            this.endPos = currentPointer.valueEnd.pos;
            return;
        }
        const nextPointer = getNextPropertyPointer(pointers, currentPointer, key);
        if (nextPointer) {
            // End the removal at the start of the next property to include the comma
            this.startPos = currentPointer.key.pos;
            this.endPos = nextPointer.key.pos; // Key is present, as it's a sibling property
            return;
        }
        const parentKey = key.substring(0, key.lastIndexOf("/"));
        const parentPointer = pointers[parentKey];
        if (!parentPointer) {
            throw new Error(`Cannot find parent JSON pointer: '${parentKey}' (for '${key}')`);
        }
        // Never remove the root object (empty string key)
        // Only remove parent if it is a property (has a key) and not an array element
        if (removeParent && parentKey !== "" && parentPointer.key) {
            this.calculatePositions(parentKey, pointers);
        }
        else {
            // Empty the parent object to remove the property and potential whitespace
            this.startPos = parentPointer.value.pos + 1; // Skip opening brace '{'
            this.endPos = parentPointer.valueEnd.pos - 1; // Skip closing brace '}'
        }
    }
    generateChanges() {
        if (this.startPos === undefined || this.endPos === undefined) {
            return undefined;
        }
        return {
            action: ChangeAction.DELETE,
            start: this.startPos,
            end: this.endPos,
        };
    }
}
//# sourceMappingURL=RemoveJsonPropertyFix.js.map