import Fix from "./Fix.js";
import { getModuleTypeInfo } from "../Ui5TypeInfo.js";
export var FixScope;
(function (FixScope) {
    /**
     * Replace the whole expression
     *
     * Call expression example: "sap.module.method()" for a call expression
     * Access expression example: "sap.module.property"
     */
    FixScope[FixScope["FullExpression"] = 0] = "FullExpression";
    /**
     * Replace the first child of the expression
     *
     * Call expression example: "sap.module.method"
     * Access expression example: "sap.module"
     */
    FixScope[FixScope["FirstChild"] = 1] = "FirstChild";
    /**
     * Replace the second child of the expression
     *
     * Call expression example: "sap.module"
     * Access expression example: "sap"
     */
    FixScope[FixScope["SecondChild"] = 2] = "SecondChild";
    /**
     * Replace the third child of the expression
     *
     * Call expression example: "sap"
     */
    FixScope[FixScope["ThirdChild"] = 3] = "ThirdChild";
    /**
     * Replace the fourth child of the expression
     */
    FixScope[FixScope["FourthChild"] = 4] = "FourthChild";
})(FixScope || (FixScope = {}));
export default class BaseFix extends Fix {
    params;
    startPos;
    endPos;
    moduleIdentifierNames;
    globalIdentifierNames;
    sourcePosition;
    nodeTypes = [];
    requestsModuleOrGlobal;
    obsoleteModuleDependency;
    constructor(params, ui5TypeInfo) {
        super();
        this.params = params;
        this.requestsModuleOrGlobal = !!(params.globalName ?? params.moduleName);
        if (this.params.moduleName &&
            (this.params.obsoleteModuleName || this.params.inferObsoleteModuleName !== false)) {
            const ui5ModuleTypeInfo = getModuleTypeInfo(ui5TypeInfo);
            if (ui5ModuleTypeInfo && ui5ModuleTypeInfo.name !== this.params.moduleName) {
                // If a new module is imported and the flag for preserving the import of the
                // module this fix has been created for is not set, then mark it as obsolete
                this.obsoleteModuleDependency = ui5ModuleTypeInfo.name;
            }
        }
    }
    getAffectedSourceCodeRange() {
        if (this.startPos === undefined || this.endPos === undefined) {
            throw new Error("Start or end position is not defined");
        }
        return {
            start: this.startPos,
            end: this.endPos,
        };
    }
    getNodeSearchParameters() {
        if (this.sourcePosition === undefined) {
            throw new Error("Position for search is not defined");
        }
        if (this.nodeTypes === undefined) {
            throw new Error("Node types for search are not defined in subclass");
        }
        return {
            nodeTypes: this.nodeTypes,
            position: this.sourcePosition,
        };
    }
    setIdentifierForDependency(identifier, moduleName) {
        this.moduleIdentifierNames ??= new Map();
        this.moduleIdentifierNames.set(moduleName, identifier);
    }
    setIdentifierForGlobal(identifier, globalName) {
        this.globalIdentifierNames ??= new Map();
        this.globalIdentifierNames.set(globalName, identifier);
    }
    getIdentifiersForSingleRequest(moduleName, globalName) {
        if (moduleName) {
            if (!this.moduleIdentifierNames?.has(moduleName)) {
                // The identifier for the requested module has not been set
                // This can happen for example if the position of the autofix is not inside
                // a module definition or require block. Therefore the required dependency can not be added
                // and the fix can not be applied.
                return;
            }
            return this.moduleIdentifierNames.get(moduleName);
        }
        else if (globalName) {
            if (!this.globalIdentifierNames?.has(globalName)) {
                // This should not happen, globals can always be provided
                throw new Error("Global identifier has not been provided");
            }
            return this.globalIdentifierNames.get(globalName);
        }
    }
    /**
     * Helper method for fix classes that feature multiple imports/globals.
     *
     * Returns undefined if any of the requested identifiers could not be set, indicating that the
     * fix must not be applied
     */
    getIdentifiersForMultipleRequests(moduleNames, globalNames) {
        const identifiers = []; // Modules first, then globals. Both in the order they have been requested in
        if (moduleNames?.length) {
            if (!this.moduleIdentifierNames) {
                // No modules have been set. Change can not be applied
                return;
            }
            for (const moduleName of moduleNames) {
                if (!this.moduleIdentifierNames.has(moduleName)) {
                    // The identifier for the requested module has not been set
                    // Change can not be applied
                    return;
                }
                identifiers.push(this.moduleIdentifierNames.get(moduleName));
            }
        }
        if (globalNames?.length) {
            if (!this.globalIdentifierNames) {
                // This should not happen, globals can always be provided
                throw new Error("Global identifier has not been provided");
            }
            for (const globalName of globalNames) {
                if (!this.globalIdentifierNames.has(globalName)) {
                    // This should not happen, globals can always be provided
                    throw new Error("Global identifier has not been provided");
                }
                identifiers.push(this.globalIdentifierNames.get(globalName));
            }
        }
        return identifiers;
    }
    getNewModuleDependencies() {
        if (this.startPos === undefined) {
            throw new Error("Start position is not defined");
        }
        if (!this.params.moduleName) {
            return;
        }
        return {
            moduleName: this.params.moduleName,
            preferredIdentifier: this.params.preferredIdentifier,
            usagePosition: this.startPos,
        };
    }
    getObsoleteModuleDependencies() {
        if (this.startPos === undefined) {
            throw new Error("Start position is not defined");
        }
        if (this.obsoleteModuleDependency) {
            return {
                moduleName: this.obsoleteModuleDependency,
                usagePosition: this.startPos,
            };
        }
    }
    getNewGlobalAccess() {
        if (this.startPos === undefined) {
            throw new Error("Start position is not defined");
        }
        if (!this.params.globalName) {
            return;
        }
        return {
            globalName: this.params.globalName,
            usagePosition: this.startPos,
        };
    }
}
//# sourceMappingURL=BaseFix.js.map