// For Library Version: 1.136.10

declare module "sap/fe/navigation/library" {
  /**
   * A static enumeration type which indicates the Odata version used for runnning the Navigation Handler.
   *
   * @since 1.83.0
   */
  export enum Mode {
    /**
     * This is used for ODataV2 services to do some internal tasks like creation of appstate, removal of sensitive
     * data etc.,
     */
    ODataV2 = "undefined",
    /**
     * This is used for ODataV4 services to do some internal tasks like creation of appstate, removal of sensitive
     * data etc.,
     */
    ODataV4 = "undefined",
  }
  /**
   * This is the successor of {@link sap.ui.generic.app.navigation.service.NavType }.
   *
   * A static enumeration type which indicates the type of inbound navigation.
   *
   * @since 1.86.0
   */
  export enum NavType {
    /**
     * Back navigation with sap-iapp-state parameter
     */
    iAppState = "undefined",
    /**
     * Initial startup without any navigation or default parameters
     */
    initial = "undefined",
    /**
     * Basic cross-app navigation with URL parameters only (without sap-xapp-state) or initial start with default
     * parameters
     */
    URLParams = "undefined",
    /**
     * Cross-app navigation with sap-xapp-state parameter (and URL parameters), defaulted parameters may be
     * added
     */
    xAppState = "undefined",
  }
  /**
   * This is the successor of {@link sap.ui.generic.app.navigation.service.ParamHandlingMode }.
   *
   * A static enumeration type which indicates the conflict resolution method when merging URL parameters
   * into select options.
   *
   * @since 1.83.0
   */
  export enum ParamHandlingMode {
    /**
     * The conflict resolution adds URL parameters to the SelectionVariant
     */
    InsertInSelOpt = "undefined",
    /**
     * The conflict resolution favors the SelectionVariant over URL parameters
     */
    SelVarWins = "undefined",
    /**
     * The conflict resolution favors the URL parameters over the SelectionVariant. Caution: In case of cross-app
     * navigation
     * a navigation parameter value from the source app is overwritten by a default, if a default is maintained
     * in the launchpad
     * designer for this parameter in the target app!
     */
    URLParamWins = "undefined",
  }
  /**
   * This is the successor of {@link sap.ui.generic.app.navigation.service.SuppressionBehavior }.
   *
   * A static enumeration type which indicates whether semantic attributes with values `null`,
   * `undefined` or `""` (empty string) shall be suppressed, before they are mixed in to the selection variant
   * in the
   * method {@link sap.fe.navigation.NavigationHandler.mixAttributesAndSelectionVariant mixAttributesAndSelectionVariant}
   * of the {@link sap.fe.navigation.NavigationHandler NavigationHandler}.
   *
   * @since 1.83.0
   */
  export enum SuppressionBehavior {
    /**
     * Semantic attributes with an empty string are ignored, the remaining attributes are mixed in to the selection
     * variant.
     * Warning! Consider the impact on Boolean variable values!
     */
    ignoreEmptyString = "undefined",
    /**
     * Semantic attributes with a `null` value lead to an {@link sap.fin.central.lib.error.Error error} of type
     * NavigationHandler.INVALID_INPUT
     */
    raiseErrorOnNull = "undefined",
    /**
     * Semantic attributes with an `undefined` value lead to an {@link sap.fin.central.lib.error.Error error }
     * of type NavigationHandler.INVALID_INPUT
     */
    raiseErrorOnUndefined = "undefined",
    /**
     * Standard suppression behavior: semantic attributes with a `null` or an `undefined` value are ignored,
     * the remaining attributes are mixed in to the selection variant
     */
    standard = "undefined",
  }
}

declare module "sap/fe/navigation/NavError" {
  import BaseObject from "sap/ui/base/Object";

  /**
   * This is the successor of {@link sap.ui.generic.app.navigation.service.NavError }.
   *  An object that provides error handling information during runtime.
   *
   * @since 1.83.0
   */
  export default class NavError extends BaseObject {
    /**
     * Constructor requiring the error code as input.
     */
    constructor(
      /**
       * String based error code.
       */
      errorCode: string
    );

    /**
     * Returns the error code with which the instance has been created.
     *
     *
     * @returns The error code of the error
     */
    getErrorCode(): string;
  }
}

declare module "sap/fe/navigation/NavigationHandler" {
  import BaseObject from "sap/ui/base/Object";

  import ODataModel from "sap/ui/model/odata/v2/ODataModel";

  import ODataModel1 from "sap/ui/model/odata/v4/ODataModel";

  import SelectionVariant from "sap/fe/navigation/SelectionVariant";

  /**
   * This is the successor of {@link sap.ui.generic.app.navigation.service.NavigationHandler }.
   *  Creates a new NavigationHandler class by providing the required environment.
   *
   * The `NavigationHandler` supports the verification of sensitive information. All properties that are part
   * of
   * `selectionVariant` and `valueTexts` will be verified if they are annotated as
   * `com.sap.vocabularies.PersonalData.v1.IsPotentiallySensitive` or
   * `com.sap.vocabularies.UI.v1.ExcludeFromNavigationContext` and will be removed before the data is persisted
   * as the app
   * state.
   *
   * Also, all properties annotated as `com.sap.vocabularies.Analytics.v1.Measure` will be removed from the
   * data stored as the
   * xapp state.
   *
   * To verify the information to be removed, the `NavigationHandler` requires an unnamed model of type
   * {@link sap.ui.model.odata.v2.ODataModel } on component level. It is possible to set such a model using
   * the `setModel`
   * method.
   *
   * **Note:** The check for excluded data requires that the OData metadata has already been loaded completely.
   *
   * If the OData metadata model has not been loaded completely, all properties are removed from the application
   * context.
   *
   * **Note:** This class requires that the UShell {@link sap.ushell.services.CrossApplicationNavigation  }
   * is available and initialized.
   *
   * @since 1.83.0
   */
  export default class NavigationHandler extends BaseObject {
    /**
     * The method creates a context url based on provided data. This context url can either be used as
     * {@link sap.fe.navigation.NavigationHandler#setParameterContextUrl ParameterContextUrl} or
     * {@link sap.fe.navigation.NavigationHandler#setFilterContextUrl FilterContextUrl}.
     *
     * @ui5-protected Do not call from applications (only from related classes in the framework)
     *
     * @returns The context url for the given entities
     */
    constructContextUrl(
      /**
       * Used for url determination
       */
      sEntitySetName: string,
      /**
       * Used for url determination. If omitted, the NavigationHandler model is used.
       */
      oModel?: ODataModel | ODataModel1
    ): string;
    /**
     * Gets the application specific technical parameters. Technical parameters will not be added to the selection
     * variant passed to the
     * application. As a default the following values are considered as technical parameters:
     *
     *
     * 	 - `sap-system`
     *
     * 	 - `sap-ushell-defaultedParameterNames`
     *
     * 	 - `"hcpApplicationId"`
     * .
     *
     *
     * @returns An Array containing the technical parameters.
     */
    getTechnicalParameters(): string[];
    /**
     * Combines the given parameters and selection variant into a new selection variant containing properties
     * from both, with the parameters
     * overriding existing properties in the selection variant. The new selection variant does not contain any
     * parameters. All parameters are
     * merged into select options. The output of this function, converted to a JSON string, can be used for
     * the
     * {@link #.navigate NavigationHandler.navigate} method.
     *
     *
     * @returns Instance of {@link sap.fe.navigation.SelectionVariant }
     */
    mixAttributesAndSelectionVariant(
      /**
       * Object/(Array of Objects) containing key/value pairs
       */
      vSemanticAttributes: any[] | object,
      /**
       * The selection variant in string format as provided by the SmartFilterBar control
       */
      sSelectionVariant: /* was: sap.fe.navigation.SelectionVariant.SerializedSelectionVariant */
        | any
        | string,
      /**
       * Indicates whether semantic
       * attributes with special values (see {@link sap.fe.navigation.SuppressionBehavior suppression behavior})
       * must be
       * suppressed before they are combined with the selection variant; several
       * {@link sap.fe.navigation.SuppressionBehavior suppression behaviors} can be combined with the bitwise
       * OR operator
       * (|)
       */
      iSuppressionBehavior?: number
    ): SelectionVariant;
    /**
     * Triggers a cross-app navigation after saving the inner and the cross-app states. The navigation mode
     * based on
     * `sap-ushell-next-navmode` is taken into account. If set to `explace` the inner app state will not be
     * changed.
     * **Note:** The `sNavMode` argument can be used to overwrite the SAP Fiori launchpad default navigation
     * for opening a URL
     * in-place or ex-place.
     *
     *
     * **Node:** If the `oExternalAppData` parameter is not supplied, the external app data will be calculated
     * based on
     * the `oInnerAppData` data.
     *
     * SmartFilterBar control **Parameters:**
     *
     * {object}
     * **oError**
     * NavError object (instance of {@link sap.fe.navigation.NavError }) that describes which kind of error
     * occurred
     *
     * {string}
     * **oError.errorCode**
     * Code to identify the error
     *
     * {string}
     * **oError.type**
     * Severity of the error (info/warning/error)
     *
     * {array}
     * **oError.params**
     * An array of objects (typically strings) that describe additional value parameters required for generating
     * the message
     * .
     */
    navigate(
      /**
       * Name of the semantic object of the target app
       */
      sSemanticObject: string,
      /**
       * Name of the action of the target app
       */
      sActionName: string,
      /**
       * Navigation parameters as an object with key/value pairs or as a string representation of such an object.
       * If passed as an object, the properties are not checked against the `IsPotentialSensitive` or `Measure`
       * type.
       */
      vNavigationParameters: object | string,
      /**
       * Object for storing current state of the app
       */
      oInnerAppData?: /* was: sap.fe.navigation.NavigationHandler.InnerAppData */ any,
      /**
       * Callback that is called if an error occurs during navigation
       */
      fnOnError?: Function,
      /**
       * Object for storing the state which will be forwarded to the target component.
       */
      oExternalAppData?: {
        /**
         * Object containing the current ui state of the app which will be forwarded to the
         * target component.
         */
        presentationVariant?: object | undefined;
        /**
         * Stringified JSON object, which will be forwarded to the target component. If not
         * provided the selectionVariant will be constructed based on the vNavigationParameters.
         */
        selectionVariant?: object | undefined;
        /**
         * Object containing value descriptions which will be forwarded to the target component.
         */
        valueTexts?: object | undefined;
      },
      /**
       * Argument is used to overwrite the FLP-configured target for opening a URL. If used, only the
       * `explace` or `inplace` values are allowed. Any other value will lead to an exception
       * `NavigationHandler.INVALID_NAV_MODE`.
       */
      sNavMode?: string
    ): void;
    /**
     * Parses the incoming URL and returns a Promise. If this method detects a back navigation, the inner app
     * state is returned in the resolved
     * Promise. Otherwise startup parameters will be merged into the app state provided by cross app navigation,
     * and a combined app state will be
     * returned. The conflict resolution can be influenced with sParamHandlingMode defined in the constructor.
     *
     *
     * @returns A Promise object to monitor when all the actions of the function have been executed. If the
     * execution is successful, the
     * extracted app state, the startup parameters, and the type of navigation are returned, see also the example
     * above. The app state is
     * an object that contains the following information:
     *
     * 	`oAppData.oSelectionVariant`: An instance of {@link sap.fe.navigation.SelectionVariant }
     * containing only parameters/select options that are related to navigation
     *
     * 	 - `oAppData.selectionVariant`: The navigation-related selection variant as a JSON-formatted string
     * `oAppData.oDefaultedSelectionVariant`: An instance of
     * {@link sap.fe.navigation.SelectionVariant } containing only the parameters/select options that are set
     * by user
     * default data
     * `oAppData.bNavSelVarHasDefaultsOnly`: A Boolean flag that indicates whether only defaulted parameters
     * and no
     * navigation parameters are present.
     *
     * **Note:** If no navigation parameters are available, `bNavSelVarHasDefaultsOnly` is set to `true`,
     * even though parameters without default might be available as well.
     *
     * If the navigation-related selection variant is empty, it is replaced by a copy of the defaulted selection
     * variant.
     *
     * The navigation type is an enumeration type of type {@link sap.fe.navigation.NavType } (possible values
     * are
     * initial, URLParams, xAppState, and iAppState).
     *
     * **Note:** If the navigation type is {@link sap.fe.navigation.NavType.iAppState } oAppData has two
     * additional properties
     *
     *
     * 	 - `oAppData.tableVariantId`
     *
     * 	 - `oAppData.customData`
     *
     * which return the inner app data as stored in {@link  #.navigate navigate} or {@link  #.storeInnerAppStateAsync storeInnerAppStateAsync}.
     * `oAppData.oDefaultedSelectionVariant` is an empty selection variant and
     * `oAppData.bNavSelVarHasDefaultsOnly` is `false` in this case.
     *
     * **Note:** If the navigation type is {@link sap.fe.navigation.NavType.initial } oAppData is an empty object!
     *
     * If an error occurs, an error object of type {@link sap.fe.navigation.NavError }, URL parameters (if available)
     * and the type of navigation are returned.
     */
    parseNavigation(): JQuery.Promise<any>;
    /**
     * Processes navigation-related tasks related to beforePopoverOpens event handling for the SmartLink control
     * and returns a Promise object. In
     * particular, the following tasks are performed before the SmartLink popover can be opened:
     *
     *
     * 	 - If `mInnerAppData` is provided, this inner app state is saved for back navigation at a later time.
     * The table event parameters (semantic attributes) and the selection variant data are combined by calling
     * the method
     * {@link #.mixAttributesAndSelectionVariant mixAttributesAndSelectionVariant}.
     * The combined data is saved as the cross app state to be handed over to the target app, and the corresponding
     * sap-xapp-state key is set
     * in the URL.
     *
     * 	 - All single selections ("including equal") of the combined selection data are passed to the SmartLink
     *     popover as semantic attributes.
     * The method `oTableEventParameters.open()` is called. Note that this does not really open the popover,
     * but the SmartLink
     * control proceeds with firing the event `navigationTargetsObtained`.
     * .
     *
     *
     * **Node:** If the `oExternalAppData` parameter is not supplied, the external app data will be calculated
     * based on
     * the `mInnerAppData` data.
     * .
     *
     *
     * @returns A Promise object to monitor when all actions of the function have been executed; if the execution
     * is successful, the
     * modified oTableEventParameters is returned; if an error occurs, an error object of type
     * {@link sap.fe.navigation.NavError } is returned
     */
    processBeforeSmartLinkPopoverOpens(
      /**
       * The parameters made available by the SmartTable control when the SmartLink control has been clicked,
       * an instance of a PopOver object
       */
      oTableEventParameters: any,
      /**
       * Stringified JSON object as returned, for example, from getDataSuiteFormat() of the SmartFilterBar control
       */
      sSelectionVariant: string,
      /**
       * Object containing the current state of the app. If provided, opening the Popover is deferred until the
       * inner app data is saved in a consistent way.
       */
      mInnerAppData?: /* was: sap.fe.navigation.NavigationHandler.InnerAppData */ any,
      /**
       * Object containing the state which will be passed to the target screen.
       */
      oExternalAppData?: {
        /**
         * Object containing the current ui presentationVariant of the app, which will be
         * passed to the target screen
         */
        presentationVariant?: object;
        /**
         * Object containing selectionVariant, which will be passed to the target screen. If not
         * set the sSelectionVariant will be used.
         */
        selectionVariant?: object;
        /**
         * Object containing value descriptions, which will be passed to the target screen
         */
        valueTexts?: object;
      }
    ): JQuery.Promise<string[], any, any>;
    /**
     * Changes the URL according to the current sAppStateKey. As an reaction route change event will be triggered.
     */
    replaceHash(
      /**
       * The new app state key.
       */
      sAppStateKey: string
    ): void;
    /**
     * Sets the model that is used for verification of sensitive information. If the model is not set, the unnamed
     * component model is used for the
     * verification of sensitive information.
     */
    setModel(
      /**
       * For checking sensitive information
       */
      oModel: ODataModel | ODataModel1
    ): void;
    /**
     * Sets the application specific technical parameters. Technical parameters will not be added to the selection
     * variant passed to the
     * application. As a default the following values are considered as technical parameters:
     *
     *
     * 	 - `sap-system`
     *
     * 	 - `sap-ushell-defaultedParameterNames`
     *
     * 	 - `"hcpApplicationId"`
     * .
     */
    setTechnicalParameters(
      /**
       * List of parameter names to be considered as technical parameters. `null` or
       * `undefined` may be used to reset the complete list.
       */
      aTechnicalParameters?: any[]
    ): void;
    /**
     * Changes the URL according to the current app state and stores the app state for later retrieval.
     *
     * @deprecated As of version 1.104. Use the {@link sap.fe.navigation.NavigationHandler.storeInnerAppStateAsync  }
     * instead.
     *
     * @returns A Promise object to monitor when all the actions of the function have been executed; if the
     * execution is successful, the
     * app state key is returned; if an error occurs, an object of type {@link sap.fe.navigation.NavError  }
     * is
     * returned
     */
    storeInnerAppState(
      /**
       * Object containing the current state of the app
       */
      mInnerAppData: /* was: sap.fe.navigation.NavigationHandler.InnerAppData */ any,
      /**
       * If set to false, the inner app hash will not be replaced until storing is successful; do not
       * set to false if you cannot react to the resolution of the Promise, for example, when calling the beforeLinkPressed
       * event
       */
      bImmediateHashReplace?: boolean
    ): jQuery.Promise<string>;
    /**
     * Changes the URL according to the current app state and stores the app state for later retrieval.
     *
     *
     * @returns A Promise object to monitor when all the actions of the function have been executed; if the
     * execution is successful, the
     * app state key is returned; if an error occurs, an object of type {@link sap.fe.navigation.NavError  }
     * is
     * returned
     */
    storeInnerAppStateAsync(
      /**
       * Object containing the current state of the app
       */
      mInnerAppData: /* was: sap.fe.navigation.NavigationHandler.InnerAppData */ any,
      /**
       * If set to false, the inner app hash will not be replaced until storing is successful; do not
       * set to false if you cannot react to the resolution of the Promise, for example, when calling the beforeLinkPressed
       * event
       */
      bImmediateHashReplace?: boolean,
      /**
       * If set to true, the inner app hash will not be replaced in the storeInnerAppStateAsync. Also the bImmediateHashReplace
       * will be ignored.
       */
      bSkipHashReplace?: boolean
    ): jQuery.Promise<string>;
    /**
     * Changes the URL according to the current app state and stores the app state for later retrieval.
     *
     * @deprecated As of version 1.104. Use the {@link sap.fe.navigation.NavigationHandler.storeInnerAppStateAsync  }
     * instead.
     *
     * @returns An object containing the appStateId and a promise object to monitor when all the actions of
     * the function have been
     * executed; Please note that the appStateKey may be undefined or empty.
     */
    storeInnerAppStateWithImmediateReturn(
      /**
       * Object containing the current state of the app
       */
      mInnerAppData: /* was: sap.fe.navigation.NavigationHandler.InnerAppData */ any,
      /**
       * If set to false, the inner app hash will not be replaced until storing is successful; do not
       * set to false if you cannot react to the resolution of the Promise, for example, when calling the beforeLinkPressed
       * event. **Note:**If
       * not provided it will be treated as set to false. **Note:**If set to true, the calling instance has to
       * ensure that a follow-on
       * call to `replaceHash` will take place!
       */
      bImmediateHashReplace?: boolean
    ): object;
  }
}

declare module "sap/fe/navigation/PresentationVariant" {
  import BaseObject from "sap/ui/base/Object";

  /**
   * This is the successor of {@link sap.ui.generic.app.navigation.service.PresentationVariant }.
   *  Creates a new instance of a PresentationVariant class. If no parameter is passed, an new empty instance
   * is created whose ID has been set to `""`. Passing a JSON-serialized string complying to the Selection
   * Variant Specification will parse it, and the newly created instance will contain the same information.
   *
   * @since 1.83.0
   */
  export default class PresentationVariant extends BaseObject {
    /**
     * If no parameter is passed, a new empty instance is created whose ID has been set to `""`.
     * Passing a JSON-serialized string complying to the Selection Variant Specification will parse it,
     * and the newly created instance will contain the same information.
     */
    constructor(
      /**
       * If of type `string`, the selection variant is JSON-formatted;
       * if of type `object`, the object represents a selection variant
       */
      presentationVariant?: object | string
    );

    /**
     * Gets the chart visualization property.
     *
     *
     * @returns An object containing the properties to be used for the chart visualization.
     */
    getChartVisualization(): Visualization | undefined;
    /**
     * Gets the current context URL intended for the query.
     *
     *
     * @returns The current context URL for the query
     */
    getContextUrl(): string | undefined;
    /**
     * Returns the identification of the selection variant.
     *
     *
     * @returns The identification of the selection variant as made available during construction
     */
    getID(): string;
    /**
     * Gets the more trivial properties. Basically all properties with the exception of the Visualization.
     *
     *
     * @returns The current properties.
     */
    getProperties(): /* was: sap.fe.navigation.PresentationVariant.PvProperties */
      | any
      | undefined;
    /**
     * Gets the table visualization property.
     *
     *
     * @returns An object containing the properties to be used for the table visualization.
     */
    getTableVisualization(): Visualization | undefined;
    /**
     * Returns the current text / description of this selection variant.
     *
     *
     * @returns The current description of this selection variant.
     */
    getText(): string | undefined;
    /**
     * Returns `true` if the presentation variant does not contain any properties.
     * nor ranges.
     *
     *
     * @returns If set to `true` there are no current properties set; `false` otherwise.
     */
    isEmpty(): boolean;
    /**
     * Sets the chart visualization property.
     */
    setChartVisualization(
      /**
       * An object containing the properties to be used for the chart visualization.
       */
      properties: Visualization
    ): void;
    /**
     * Sets the context URL.
     */
    setContextUrl(
      /**
       * The URL of the context
       */
      url: string
    ): void;
    /**
     * Sets the identification of the selection variant.
     */
    setID(
      /**
       * The new identification of the selection variant
       */
      id: string
    ): void;
    /**
     * Sets the more trivial properties. Basically all properties with the exception of the Visualization.
     */
    setProperties(
      /**
       * The properties to be used.
       */
      properties: object
    ): void;
    /**
     * Sets the table visualization property.
     */
    setTableVisualization(
      /**
       * An object containing the properties to be used for the table visualization.
       */
      properties: Visualization
    ): void;
    /**
     * Sets the text / description of the selection variant.
     */
    setText(
      /**
       * The new description to be used
       */
      newText?: string
    ): void;
    /**
     * Returns the external representation of the selection variant as JSON object.
     *
     *
     * @returns The external representation of this instance as a JSON object
     */
    toJSONObject(): ExternalPresentationVariant;
    /**
     * Serializes this instance into a JSON-formatted string.
     *
     *
     * @returns The JSON-formatted representation of this instance in stringified format
     */
    toJSONString(): string;
  }
  /**
   * Structure of dimension type
   */
  export type DimensionType = {
    /**
     * Defines dimension.
     */
    Dimension: string;
    /**
     * Defines role.
     */
    Role: string;
  };

  /**
   * Structure of the external plain object representation of a PresentationVariant
   */
  export type ExternalPresentationVariant = {
    /**
     * Defines context URL.
     */
    ContextUrl: string;
    /**
     * Defines presentation variant ID.
     */
    PresentationVariantID: string;
    /**
     * Defines text.
     */
    Text: string;
    /**
     * Defines version.
     */
    Version: object;
    /**
     * Defines visualizations.
     */
    Visualizations: Visualization[];
  };

  /**
   * Structure of measure type
   */
  export type MeasureType = {
    /**
     * Defines measure.
     */
    Measure: string;
    /**
     * Defines role.
     */
    Role: string;
  };

  /**
   * Structure of a visualization object
   */
  export type VisChartContent = {
    /**
     * Defines chart type.
     */
    ChartType: string;
    /**
     * Defines dimension attributes.
     */
    DimensionAttributes: DimensionType[];
    /**
     * Defines dimensions.
     */
    Dimensions: string[];
    /**
     * Defines measure attributes.
     */
    MeasureAttributes: MeasureType[];
    /**
     * Defines measures.
     */
    Measures: string[];
  };

  /**
   * Structure of the external plain object representation of a PresentationVariant
   */
  export type VisContentType = {
    /**
     * Defines value.
     */
    Value: string;
  };

  /**
   * Structure of a visualization object
   */
  export type Visualization = {
    /**
     * Defines content of the object.
     */
    Content: VisChartContent | VisContentType[];
    /**
     * Defines type of the object.
     */
    Type: string;
  };
}

declare module "sap/fe/navigation/SelectionVariant" {
  import BaseObject from "sap/ui/base/Object";

  /**
   * This is the successor of {@link sap.ui.generic.app.navigation.service.SelectionVariant }.
   *
   * Creates a new instance of a SelectionVariant class. If no parameter is passed,
   * an new empty instance is created whose ID has been set to `""`.
   * Passing a JSON-serialized string complying to the Selection Variant Specification will parse it,
   * and the newly created instance will contain the same information.
   *
   * @since 1.83.0
   */
  export default class SelectionVariant extends BaseObject {
    /**
     * Sets the value of a parameter called `sName` to the new value `sValue`.
     * If the parameter has already been set before, its value is overwritten.
     *
     *
     * @returns This instance to allow method chaining
     */
    addParameter(
      /**
       * The name of the parameter to be set; the `null` value is not allowed
       */
      sName: string,
      /**
       * The value of the parameter to be set
       */
      sValue: string
    ): this;
    /**
     * Adds a new range to the list of select options for a given parameter.
     *
     *
     * @returns This instance to allow method chaining.
     */
    addSelectOption(
      /**
       * The name of the property for which the selection range is added
       */
      sPropertyName: string,
      /**
       * The sign of the range (**I**nclude or **E**xclude)
       */
      sSign: string,
      /**
       * The option of the range (**EQ** for "equals", **NE** for "not equals",
       * **LE** for "less or equals", **GE** for "greater or equals", **LT** for "less than" (and not equals),
       * **GT** for "greater than" (and not equals), **BT** for "between", or **CP** for "contains pattern"
       * (ABAP-styled pattern matching with the asterisk as wildcard)
       */
      sOption: string,
      /**
       * The single value or the lower boundary of the interval; the `null` value is not allowed
       */
      sLow: string,
      /**
       * Set only if sOption is **BT**: the upper boundary of the interval
       */
      sHigh?: null | string,
      /**
       * Text representing the SelectOption. This is an optional parameter. For an example in most Fiori applications
       * if the text is not provided, it is fetched based on the ID.
       * must be `undefined` or `null` in all other cases
       */
      sText?: string,
      /**
       * Object containing semanticDates filter information
       */
      semanticDates?: /* was: sap.fe.navigation.SelectionVariant.SemanticDateConfiguration */ any
    ): this;
    /**
     * Gets the current context URL intended for the filters.
     *
     *
     * @returns The current context URL for the filters
     */
    getFilterContextUrl(): string | undefined;
    /**
     * Returns the identification of the selection variant.
     *
     *
     * @returns The identification of the selection variant as made available during construction
     */
    getID(): string;
    /**
     * Returns the value of the parameter called `sName` if it has been set.
     * If the parameter has never been set or has been removed, `undefined` is returned.
     *
     *
     * @returns The value of parameter `sName`; returning the value `null` not possible
     */
    getParameter(
      /**
       * The name of the parameter to be returned
       */
      sName: string
    ): string;
    /**
     * Gets the current context URL intended for the parameters.
     *
     *
     * @returns The current context URL for the parameters
     */
    getParameterContextUrl(): string | undefined;
    /**
     * Returns the set of parameter names available in this selection variant.
     *
     *
     * @returns The list of parameter names which are valid
     */
    getParameterNames(): string[];
    /**
     * Returns the names of the parameter and select option properties available for this instance.
     *
     *
     * @returns The list of parameter and select option property names available for this instance
     */
    getPropertyNames(): string[];
    /**
     * Returns the set of select options/ranges available for a given property name.
     *
     *
     * @returns If `sPropertyName` is an invalid name of a property or no range exists, `undefined`
     * is returned; otherwise, an immutable array of ranges is returned. Each entry of the array is an object
     * with the
     * following properties:
     *
     *
     * 	 - `Sign`: The sign of the range
     *
     * 	 - `Option`: The option of the range
     *
     * 	 - `Low`: The low value of the range; returning value `null` is not possible
     *
     * 	 - `High`: The high value of the range; if this value is not necessary, `null` is used
     *
     * For further information about the meaning of the attributes, refer to method `addSelectOption`.
     */
    getSelectOption(
      /**
       * The name of the property for which the set of select options/ranges is returned
       */
      sPropertyName: string
    ): /* was: sap.fe.navigation.SelectionVariant.SelectOption */
      | any[]
      | undefined;
    /**
     * Returns the names of the properties available for this instance.
     *
     *
     * @returns The list of property names available for this instance
     */
    getSelectOptionsPropertyNames(): string[];
    /**
     * Returns the current text / description of this selection variant.
     *
     *
     * @returns The current description of this selection variant.
     */
    getText(): string | undefined;
    /**
     * First tries to retrieve the set of select options or ranges available for `sName` as the property name.
     * If successful,
     * this array of selections is returned. If it fails, an attempt to find a parameter with the name `sName`
     * is used. If the latter succeeds, the single value is converted to fit into an array of selections to
     * make it
     * type compatible with ranges. This array is then returned.
     * If neither a select option nor a parameter could be found, `undefined` is returned.
     *
     *
     * @returns The ranges in the select options for the specified property or a range-converted representation
     * of a parameter is returned.
     * If both lookups fail, `undefined` is returned.
     * The returned ranges have the format:
     *
     *
     * 	 - `Sign`: The sign of the range
     *
     * 	 - `Option`: The option of the range
     *
     * 	 - `Low`: The low value of the range; returning the value `null` is not possible
     *
     * 	 - `High`: The high value of the range; if this value is not necessary, `null` (but does exist)
     *
     * For further information on the meaning of the attributes, refer to method {@link  #.addSelectOption addSelectOption}.
     */
    getValue(
      /**
       * The name of the attribute for which the value is retrieved
       */
      sName: string
    ): /* was: sap.fe.navigation.SelectionVariant.SelectOption */
      | any[]
      | undefined;
    /**
     * Returns `true` if the selection variant does neither contain parameters
     * nor ranges.
     *
     *
     * @returns If set to `true`  there are no parameters and no select options available in
     * the selection variant; `false` otherwise.
     */
    isEmpty(): boolean;
    /**
     * Adds a set of select options to the list of select options for a given parameter.
     *
     *
     * @returns This instance to allow method chaining
     */
    massAddSelectOption(
      /**
       * The name of the property for which the set of select options is added
       */
      sPropertyName: string,
      /**
       * Set of select options to be added
       */
      aSelectOptions: /* was: sap.fe.navigation.SelectionVariant.SelectOption */ any[]
    ): this;
    /**
     * Removes a parameter called `sName` from the selection variant.
     *
     *
     * @returns This instance to allow method chaining
     */
    removeParameter(
      /**
       * The name of the parameter to be removed
       */
      sName: string
    ): this;
    /**
     * Removes a select option called `sName` from the selection variant.
     *
     *
     * @returns This instance to allow method chaining.
     */
    removeSelectOption(
      /**
       * The name of the select option to be removed
       */
      sName: string
    ): this;
    /**
     * Renames a parameter called `sNameOld` to `sNameNew`. If a parameter or a select option with
     * the name `sNameNew` already exist, an error is thrown. If a parameter with the name `sNameOld`
     * does not exist, nothing is changed.
     *
     *
     * @returns This instance to allow method chaining
     */
    renameParameter(
      /**
       * The current name of the parameter to be renamed
       */
      sNameOld: string,
      /**
       * The new name of the parameter
       */
      sNameNew: string
    ): this;
    /**
     * Renames a select option called `sNameOld` to `sNameNew`. If a select option or a parameter
     * with the name `sNameNew` already exist, an error is thrown. If a select option with the name `sNameOld`
     * does not exist, nothing is changed.
     *
     *
     * @returns This instance to allow method chaining
     */
    renameSelectOption(
      /**
       * The current name of the select option property to be renamed
       */
      sNameOld: string,
      /**
       * The new name of the select option property
       */
      sNameNew: string
    ): this;
    /**
     * Sets the context URL intended for the filters.
     */
    setFilterContextUrl(
      /**
       * The URL of the filters
       */
      sURL: string
    ): void;
    /**
     * Sets the identification of the selection variant.
     */
    setID(
      /**
       * The new identification of the selection variant
       */
      id: string
    ): void;
    /**
     * Sets the context URL intended for the parameters.
     */
    setParameterContextUrl(
      /**
       * The URL of the parameter context
       */
      sURL: string
    ): void;
    /**
     * Sets the text / description of the selection variant.
     */
    setText(
      /**
       * The new description to be used
       */
      newText?: string
    ): void;
    /**
     * Returns the external representation of the selection variant as JSON object.
     *
     *
     * @returns The external representation of this instance as a JSON object
     */
    toJSONObject(): /* was: sap.fe.navigation.SelectionVariant.SerializedSelectionVariant */ any;
    /**
     * Serializes this instance into a JSON-formatted string.
     *
     *
     * @returns The JSON-formatted representation of this instance in stringified format
     */
    toJSONString(): string;
  }
}

declare namespace sap {
  interface IUI5DefineDependencyNames {
    "sap/fe/navigation/library": undefined;

    "sap/fe/navigation/NavError": undefined;

    "sap/fe/navigation/NavigationHandler": undefined;

    "sap/fe/navigation/PresentationVariant": undefined;

    "sap/fe/navigation/SelectionVariant": undefined;
  }
}
