/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007,2008 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:CMIContext.java 914 2007-05-25 16:48:16Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.cmi.jndi.context;

import java.util.Collection;
import java.util.Hashtable;
import java.util.List;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NameClassPair;
import javax.naming.NameParser;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;

import net.jcip.annotations.NotThreadSafe;

import org.ow2.cmi.controller.common.ClusterViewManager;
import org.ow2.cmi.controller.server.ServerClusterViewManager;
import org.ow2.cmi.controller.server.ServerClusterViewManagerException;
import org.ow2.cmi.info.CMIInfoRepository;
import org.ow2.cmi.info.ClusteredObjectInfo;
import org.ow2.cmi.jndi.ClusteredObject;
import org.ow2.cmi.lb.LoadBalanceable;
import org.ow2.cmi.lb.NoLoadBalanceableException;
import org.ow2.cmi.lb.decision.DecisionManager;
import org.ow2.cmi.lb.decision.DecisionManager.Decision;
import org.ow2.cmi.lb.policy.FirstAvailable;
import org.ow2.cmi.lb.policy.IPolicy;
import org.ow2.cmi.reference.CMIReference;
import org.ow2.cmi.reference.ObjectNotFoundException;
import org.ow2.cmi.reference.ServerId;
import org.ow2.cmi.reference.ServerRef;
import org.ow2.cmi.rpc.CMIProxy;
import org.ow2.cmi.rpc.CMIProxyFactory;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


/**
 * The CMI context intercepts the accesses to JNDI
 * and delegates the management of cluster view at an instance of {@link ClusterViewManager} which is a singleton.
 * An instance of CMIContext is relative at a protocol.
 * Instances owned by a server must have a local registry to perform operations such a bind.
 * @author The new CMI team
 * @see ClusterViewManager
 */
@NotThreadSafe
public final class CMIContext implements Context {

    /**
     * Logger.
     */
    private static final Log LOGGER = LogFactory.getLog(CMIContext.class);

    /**
     * The local context for not load-balanceable tasks (e.g. bindings).
     */
    private final Context localContext;

    /**
     * The name of the associated InitialContextFactory.
     */
    private final String initialContextFactoryName;

    /**
     * The protocol name to use for connecting to the registry.
     */
    private final String protocol;

    /**
     * True if clustered object can be bound (used only when serverModeEnabled=true).
     */
    private final boolean replicationEnabled;

    /**
     * A reference on the local registry (null for a client).
     */
    private final ServerId localRegistry;

    /**
     * A list of reference on remote registries.
     */
    private final List<ServerRef> remoteRegistries;

    /**
     * Manager which provides a cluster view.
     */
    private final ClusterViewManager clusterViewManager;

    /**
     * Policy to use when lazy-lookup is not enabled.
     */
    private IPolicy<ServerRef> defaultPolicy;

    /**
     * True indicates that {@link #close()} is disabled.
     */
    private boolean notCloseable = false;

    private final String dummyRegistryName;


    /**
     * Constructs a new CMI context from a given environment.
     * Servers must call it.
     * @param localRegistry a reference on a local registry to perform binds
     * @param initialContextFactoryName a factory to construct real contexts
     * @param protocol protocol
     * @param clusterViewProviderUrls
     * @throws CMINamingException if the default LB policy for access to JNDI cannot be retrieved
     * or if the instance of ServerClusterViewManager cannot be retrieved (when replication is enabled)
     * @see CMIInitialContextFactory which contruct this
     */
    public CMIContext(
            final ClusterViewManager clusterViewManager,
            final ServerId localRegistry,
            final String initialContextFactoryName,
            final String protocol,
            final List<ServerRef> clusterViewProviderUrls) throws CMINamingException {

        // Keep data to retrieve it
        this.initialContextFactoryName = initialContextFactoryName;
        this.clusterViewManager = clusterViewManager;
        this.localRegistry = localRegistry;
        this.protocol = protocol;
        this.replicationEnabled = clusterViewManager instanceof ServerClusterViewManager;
        this.dummyRegistryName = clusterViewManager.getConfig().getDummyRegistryName();
        this.remoteRegistries = clusterViewProviderUrls;

        try {
            setDefaultPolicy();
        } catch (CMIContextException e) {
            LOGGER.error("Cannot set the default LB policy", e);
            throw new CMINamingException("Cannot set the default LB policy", e);
        }
        if(localRegistry != null) {
            // Create a local context
            this.localContext = getRealContext(localRegistry.getProviderURL(), initialContextFactoryName);
        } else {
            this.localContext = null;
        }
    }

    /**
     * Adds a new environment property to the environment of this context.
     * If the property already exists, its value is overwritten.
     * @param propName the name of the environment property to add; may not be null
     * @param propVal the value of the property to add; may not be null
     * @return the previous value of the property, or null if the property was not in the environment before
     * @throws NamingException if a naming exception is encountered.
     */
    public Object addToEnvironment(final String propName, final Object propVal) throws NamingException {
        LOGGER.debug("addToEnvironment {0}", propName);
        if(localContext != null) {
            return localContext.addToEnvironment(propName, propVal);
        } else {
            LOGGER.warn("Cannot addToEnvironment because server mode is not activated.");
            return null;
        }
    }

    /**
     * Bind a new clustered object.
     * @param name The name to bind; may not be empty.
     * @param obj The object to bind; possibly null.
     * @throws NamingException if a naming exception is encountered.
     */
    public void bind(final Name name, final Object obj) throws NamingException {
        // Just use the string version for now.
        bind(name.toString(), obj);
    }

    /**
     * Bind a new clustered object.
     * @param name The name to bind; may not be empty.
     * @param obj The object to bind; possibly null.
     * @throws NamingException if a naming exception is encountered.
     */
    public void bind(final String name, final Object obj) throws NamingException {
        LOGGER.debug("bind {0}", name);
        // TODO Security
        if(localContext != null) {
            LOGGER.debug("Server mode enabled");
            localContext.bind(name, obj);
            // Verify that replication is activated
            if(replicationEnabled && clusterViewManager != null
                    && clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE)) {
                // If CMI is activated, we insert a new entry into the cluster view thanks to the manager of the server-side
                try {
                    bindIntoCluster(name, obj);
                    LOGGER.debug("The object for name {0} has been added in the cluster.", name);
                } catch (CMIContextException e) {
                    LOGGER.error("Cannot add the object for name  {0} to the cluster", name);
                    CMINamingException ne = new CMINamingException(
                            "Cannot add the object for name {0} to the cluster" + name);
                    ne.setRootCause(e);
                    throw ne;
                }
            } else {
                LOGGER.debug("Replication is disabled");
            }
        } else {
            LOGGER.warn("Cannot bind because server mode is not activated.");
        }
    }

    public void close() throws NamingException {
        LOGGER.debug("close");
        if(localContext != null) {
            if(!notCloseable) {
                localContext.close();
            }
        } else {
            LOGGER.warn("Cannot close because server mode is not activated.");
        }
    }

    public Name composeName(final Name name, final Name prefix) throws NamingException {
        if(localContext != null) {
            return localContext.composeName(name, prefix);
        } else {
            LOGGER.warn("Cannot compose name because server mode is not activated.");
            return null;
        }
    }

    public String composeName(final String name, final String prefix) throws NamingException {
        if(localContext != null) {
            return localContext.composeName(name, prefix);
        } else {
            LOGGER.warn("Cannot compose name because server mode is not activated.");
            return null;
        }
    }

    public Context createSubcontext(final Name name) throws NamingException {
        if(localContext != null) {
            return localContext.createSubcontext(name);
        } else {
            LOGGER.warn("Cannot create subcontext because server mode is not activated.");
            return null;
        }
    }

    public Context createSubcontext(final String name) throws NamingException {
        if(localContext != null) {
            return localContext.createSubcontext(name);
        } else {
            LOGGER.warn("Cannot create subcontext because server mode is not activated.");
            return null;
        }
    }

    public void destroySubcontext(final Name name) throws NamingException {
        if(localContext != null) {
            localContext.destroySubcontext(name);
        } else {
            LOGGER.warn("Cannot destroy subcontext because server mode is not activated.");
        }
    }

    public void destroySubcontext(final String name) throws NamingException {
        if(localContext != null) {
            localContext.destroySubcontext(name);
        } else {
            LOGGER.warn("Cannot destroy subcontext because server mode is not activated.");
        }
    }

    public Hashtable<?, ?> getEnvironment() throws NamingException {
        if(localContext != null) {
            return localContext.getEnvironment();
        } else {
            LOGGER.warn("Cannot getEnvironement because server mode is not activated.");
            return null;
        }
    }

    public String getNameInNamespace() throws NamingException {
        if(localContext != null) {
            return localContext.getNameInNamespace();
        } else {
            LOGGER.warn("Cannot getNameInNamespace because server mode is not activated.");
            return null;
        }
    }

    public NameParser getNameParser(final Name name) throws NamingException {
        if(localContext != null) {
            return localContext.getNameParser(name);
        } else {
            LOGGER.warn("Cannot getNameParser because server mode is not activated.");
            return null;
        }
    }

    public NameParser getNameParser(final String name) throws NamingException {
        if(localContext != null) {
            return localContext.getNameParser(name);
        } else {
            LOGGER.warn("Cannot getNameParser because server mode is not activated.");
            return null;
        }
    }

    public NamingEnumeration<NameClassPair> list(final Name name) throws NamingException {
        // Just use the string version for now.
        return list(name.toString());
    }

    @SuppressWarnings("unchecked")
    public NamingEnumeration<NameClassPair> list(final String name) throws NamingException {
        // On server, the local registry is always used.
        if(localContext != null) {
            return localContext.list(name);
        }
        IPolicy<? extends LoadBalanceable> lbPolicy;
        Collection<? extends LoadBalanceable> loadBalanceables;

        // Check if the client is connected at a provider of the cluster view.
        boolean localMode = clusterViewManager == null
            || !clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE);

        if(!localMode) {
            // The up-to-date policy and provider urls can be used.
            try{
                lbPolicy = clusterViewManager.getPolicy(dummyRegistryName);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get LB policy for object {0}",dummyRegistryName, e);
                throw new CMINamingException("Cannot get LB policy for object " + dummyRegistryName, e);
            }
            LOGGER.debug("LB policy for dummy context: {0}", lbPolicy);
            try {
                loadBalanceables = clusterViewManager.getCMIReferences(dummyRegistryName, protocol);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get node list for {0}", dummyRegistryName, e);
                throw new CMINamingException("Cannot get node list for " + dummyRegistryName, e);
            }
            LOGGER.debug("CMIReferences for dummy context: {0}", loadBalanceables);
        } else {
            // The initial policy and provider urls must be used.
            lbPolicy = defaultPolicy;
            loadBalanceables = remoteRegistries;
        }

        // Choose a server to perform the request.
        LoadBalanceable loadBalanceable;
        do {
            try {
                loadBalanceable = ((IPolicy<LoadBalanceable>) lbPolicy).choose((List<LoadBalanceable>) loadBalanceables);
            } catch (NoLoadBalanceableException e) {
                LOGGER.error("Cannot choose CMIReference in the list {0} with LB policy {1}",
                        loadBalanceables, lbPolicy, e);
                throw new CMINamingException("Cannot choose CMIReference in the list " + loadBalanceables
                        + " with LB policy " + lbPolicy);
            }
            ServerRef serverRef = loadBalanceable.getServerRef();
            String pURL = serverRef.getProviderURL();
            LOGGER.debug("Chosen providerURL: {0}", pURL);
            Context context = getRealContext(pURL, initialContextFactoryName);
            try {
                return context.list(name);
            } catch(NamingException e) {
                LOGGER.debug("onLookupException: ", e);
                DecisionManager<Void> decisionManager =
                    ((IPolicy<LoadBalanceable>) lbPolicy).onLookupException(loadBalanceable, e);
                if(decisionManager.getDecision().equals(Decision.THROW)) {
                    LOGGER.debug("Throw: ", e);
                    throw e;
                }
                LOGGER.debug("Removing node {0}", loadBalanceable);
                loadBalanceables.remove(loadBalanceable);
                if(loadBalanceables.isEmpty()) {
                    LOGGER.debug("No more load balanceable - Throw: ", e);
                    throw e;
                }
                // Retry with an other server...
            }
        } while(true);
    }

    public NamingEnumeration<Binding> listBindings(final Name name) throws NamingException {
        // Just use the string version for now.
        return listBindings(name.toString());
    }

    @SuppressWarnings("unchecked")
    public NamingEnumeration<Binding> listBindings(final String name) throws NamingException {
        // On server, the local registry is always used.
        if(localContext != null) {
            return localContext.listBindings(name);
        }
        IPolicy<? extends LoadBalanceable> lbPolicy;
        Collection<? extends LoadBalanceable> loadBalanceables;

        // Check if the client is connected at a provider of the cluster view.
        boolean localMode = clusterViewManager == null
            || !clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE);

        if(!localMode) {
            // The up-to-date policy and provider urls can be used.
            try{
                lbPolicy = clusterViewManager.getPolicy(dummyRegistryName);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get LB policy for object {0}", dummyRegistryName, e);
                throw new CMINamingException("Cannot get LB policy for object " + dummyRegistryName, e);
            }
            LOGGER.debug("LB policy for dummy context: {0}", lbPolicy);
            try {
                loadBalanceables = clusterViewManager.getCMIReferences(dummyRegistryName, protocol);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get node list for {0}", dummyRegistryName, e);
                throw new CMINamingException("Cannot get node list for " + dummyRegistryName, e);
            }
            LOGGER.debug("CMIReferences for dummy context: {0}", loadBalanceables);
        } else {
            // The initial policy and provider urls must be used.
            lbPolicy = defaultPolicy;
            loadBalanceables = remoteRegistries;
        }

        // Choose a server to perform the request.
        LoadBalanceable loadBalanceable;
        do {
            try {
                loadBalanceable = ((IPolicy<LoadBalanceable>) lbPolicy).choose((List<LoadBalanceable>) loadBalanceables);
            } catch (NoLoadBalanceableException e) {
                LOGGER.error("Cannot choose CMIReference in the list {0} with LB policy {1}",
                        loadBalanceables, lbPolicy, e);
                throw new CMINamingException("Cannot choose CMIReference in the list " + loadBalanceables
                        + " with LB policy " + lbPolicy);
            }
            ServerRef serverRef = loadBalanceable.getServerRef();
            String pURL = serverRef.getProviderURL();
            LOGGER.debug("Chosen providerURL: {0}", pURL);
            Context context = getRealContext(pURL, initialContextFactoryName);
            try {
                return context.listBindings(name);
            } catch(NamingException e) {
                LOGGER.debug("onLookupException: ", e);
                DecisionManager<Void> decisionManager =
                    ((IPolicy<LoadBalanceable>) lbPolicy).onLookupException(loadBalanceable, e);
                if(decisionManager.getDecision().equals(Decision.THROW)) {
                    LOGGER.debug("Throw: ", e);
                    throw e;
                }
                LOGGER.debug("Removing node {0}", loadBalanceable);
                loadBalanceables.remove(loadBalanceable);
                if(loadBalanceables.isEmpty()) {
                    LOGGER.debug("No more load balanceable - Throw: ", e);
                    throw e;
                }
                // Retry with an other server...
            }
        } while(true);
    }

    /**
     * Looks up an object.
     * @param name the name of the object to look up.
     * @return A CMI proxy if CMI is activated, the bound object otherwise.
     * @throws NamingException if a naming exception is encountered.
     */
    public Object lookup(final Name name) throws NamingException {
        // Just use the string version for now.
        return lookup(name.toString());
    }

    /**
     * Looks up an object.
     * @param name the name of the object to look up.
     * @return A CMI proxy if CMI is activated, the bound object otherwise.
     * @throws NamingException if a naming exception is encountered.
     */
    @SuppressWarnings("unchecked")
    public Object lookup(final String name) throws NamingException {

        boolean localMode = clusterViewManager == null
            || !clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE);

        if(!localMode) {
            LOGGER.debug("lookup {0} using ClusterViewManager", name);

            // Tests if the object is clustered
            if(clusterViewManager.isClustered(name)) {
                LOGGER.debug("{0} is clustered", name);
                try {
                    return findClusteredObject(name);
                } catch (CMIContextException e) {
                    LOGGER.warn("Clustered object not available: " + name, e);
                }
            }
        }

        // If we are on a server, and we lookup an object that is not clustered, we always accesses to the local registry
        if(localContext != null) {
            return localContext.lookup(name);
        }

        // The object is not clustered: we don't use CMIProxy but lookup for it in all the known registries
        IPolicy<? extends LoadBalanceable> lbPolicy;
        Collection<? extends LoadBalanceable> loadBalanceables;

        if(!localMode) {
            // The up-to-date policy and provider urls can be used.
            try{
                lbPolicy = clusterViewManager.getPolicy(dummyRegistryName);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get LB policy for object {0}", dummyRegistryName, e);
                throw new CMINamingException(
                        "Cannot get LB policy for object " + dummyRegistryName, e);
            }
            LOGGER.debug("LB policy for dummy context: {0}", lbPolicy);
            try {
                loadBalanceables = clusterViewManager.getCMIReferences(dummyRegistryName, protocol);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get node list for {0}", dummyRegistryName, e);
                throw new CMINamingException("Cannot get node list for " + dummyRegistryName, e);
            }
            LOGGER.debug("CMIReferences for dummy context: {0}", loadBalanceables);
        } else {
            // The initial policy and provider urls must be used.
            lbPolicy = defaultPolicy;
            loadBalanceables = remoteRegistries;
        }

        // Choose a server to perform the request.
        LoadBalanceable loadBalanceable;
        do {
            try {
                loadBalanceable = ((IPolicy<LoadBalanceable>) lbPolicy).choose((List<LoadBalanceable>) loadBalanceables);
            } catch (NoLoadBalanceableException e) {
                LOGGER.error("Cannot choose CMIReference in the list {0} with LB policy {1}",
                        loadBalanceables, lbPolicy, e);
                throw new CMINamingException("Cannot choose CMIReference in the list " + loadBalanceables
                        +" with LB policy " + lbPolicy, e);
            }
            ServerRef serverRef = loadBalanceable.getServerRef();
            String pURL = serverRef.getProviderURL();
            LOGGER.debug("Chosen providerURL: {0}", pURL);
            Context context = getRealContext(pURL, initialContextFactoryName);
            try {
                return context.lookup(name);
            } catch(NamingException e) {
                LOGGER.debug("onLookupException: ", e);
                DecisionManager<Void> decisionManager =
                    ((IPolicy<LoadBalanceable>) lbPolicy).onLookupException(loadBalanceable, e);
                if(decisionManager.getDecision().equals(Decision.THROW)) {
                    LOGGER.debug("Throw: ", e);
                    throw e;
                }
                LOGGER.debug("Removing node {0}", loadBalanceable);
                loadBalanceables.remove(loadBalanceable);
                if(loadBalanceables.isEmpty()) {
                    LOGGER.debug("No more load balanceable - Throw: ", e);
                    throw e;
                }
                // Retry with an other server...
            }
        } while(true);
    }

    /**
     * Looks up for the link of an object.
     * @param name the name of the object to look up.
     * @return A CMI proxy if CMI is activated, the bound object otherwise.
     * @throws NamingException if a naming exception is encountered.
     */
    public Object lookupLink(final Name name) throws NamingException {
        // Just use the string version for now.
        return lookupLink(name.toString());
    }

    /**
     * Looks up for the link of an object.
     * @param name the name of the object to look up.
     * @return A CMI proxy if CMI is activated, the bound object otherwise.
     * @throws NamingException if a naming exception is encountered.
     */
    @SuppressWarnings("unchecked")
    public Object lookupLink(final String name) throws NamingException {

        boolean localMode = clusterViewManager == null
            || !clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE);

        if(!localMode) {
            LOGGER.debug("lookup {0} using ClusterViewManager", name);

            // Tests if the object is clustered
            if(clusterViewManager.isClustered(name)) {
                LOGGER.debug("{0} is clustered", name);
                try {
                    return findClusteredObject(name);
                } catch (CMIContextException e) {
                    LOGGER.warn("Clustered object not available: " + name, e);
                }
            }
        }

        // If we are on a server, and we lookup an object that is not clustered, we always accesses to the local registry
        if(localContext != null) {
            return localContext.lookupLink(name);
        }

        // The object is not clustered: we don't use CMIProxy but lookup for it in all the known registries
        IPolicy<? extends LoadBalanceable> lbPolicy;
        Collection<? extends LoadBalanceable> loadBalanceables;

        if(!localMode) {
            // The up-to-date policy and provider urls can be used.
            try{
                lbPolicy = clusterViewManager.getPolicy(dummyRegistryName);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get LB policy for object {0}", dummyRegistryName, e);
                throw new CMINamingException(
                        "Cannot get LB policy for object " + dummyRegistryName, e);
            }
            LOGGER.debug("LB policy for dummy context: {0}", lbPolicy);
            try {
                loadBalanceables = clusterViewManager.getCMIReferences(dummyRegistryName, protocol);
            } catch (ObjectNotFoundException e) {
                LOGGER.error("Cannot get node list for {0}", dummyRegistryName, e);
                throw new CMINamingException("Cannot get node list for " + dummyRegistryName, e);
            }
            LOGGER.debug("CMIReferences for dummy context: {0}", loadBalanceables);
        } else {
            // The initial policy and provider urls must be used.
            lbPolicy = defaultPolicy;
            loadBalanceables = remoteRegistries;
        }

        // Choose a server to perform the request.
        LoadBalanceable loadBalanceable;
        do {
            try {
                loadBalanceable = ((IPolicy<LoadBalanceable>) lbPolicy).choose((List<LoadBalanceable>) loadBalanceables);
            } catch (NoLoadBalanceableException e) {
                LOGGER.error("Cannot choose CMIReference in the list {0} with LB policy {1}",
                        loadBalanceables, lbPolicy, e);
                throw new CMINamingException("Cannot choose CMIReference in the list "
                        + loadBalanceables + " with LB policy " + lbPolicy, e);
            }
            ServerRef serverRef = loadBalanceable.getServerRef();
            String pURL = serverRef.getProviderURL();
            LOGGER.debug("Chosen providerURL: {0}", pURL);
            Context context = getRealContext(pURL, initialContextFactoryName);
            try {
                return context.lookupLink(name);
            } catch(NamingException e) {
                LOGGER.debug("onLookupException: ", e);
                DecisionManager<Void> decisionManager =
                    ((IPolicy<LoadBalanceable>) lbPolicy).onLookupException(loadBalanceable, e);
                if(decisionManager.getDecision().equals(Decision.THROW)) {
                    LOGGER.debug("Throw: ", e);
                    throw e;
                }
                LOGGER.debug("Removing node {0}", loadBalanceable);
                loadBalanceables.remove(loadBalanceable);
                if(loadBalanceables.isEmpty()) {
                    LOGGER.debug("No more load balanceable - Throw: ", e);
                    throw e;
                }
                // Retry with an other server...
            }
        } while(true);
    }

    public void rebind(final Name name, final Object obj) throws NamingException {
        // Just use the string version for now.
        rebind(name.toString(), obj);
    }

    public void rebind(final String name, final Object obj) throws NamingException {
        LOGGER.debug("rebind {0}", name);
        // TODO Security
        if(localContext != null) {
            LOGGER.debug("Server mode enabled");
            localContext.rebind(name, obj);
            // Verify that replication is activated
            if(replicationEnabled && clusterViewManager != null
                    && clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE)) {
                // If CMI is activated, we insert a new entry into the cluster view thanks to the manager of the server-side
                try {
                    bindIntoCluster(name, obj);
                    LOGGER.debug("The object for name {0} has been added in the cluster.", name);
                } catch (CMIContextException e) {
                    LOGGER.error("Cannot add the object for name  {0} to the cluster", name);
                    throw new CMINamingException(
                            "Cannot add the object for name {0} to the cluster" + name, e);
                }
            } else {
                LOGGER.debug("Replication is disabled");
            }
        } else {
            LOGGER.warn("Cannot rebind because server mode is not activated.");
        }

    }

    public Object removeFromEnvironment(final String propName) throws NamingException {
        if(localContext != null) {
            return localContext.removeFromEnvironment(propName);
        } else {
            LOGGER.warn("Cannot removeFromEnvironment because server mode is not activated.");
            return null;
        }
    }

    public void rename(final Name oldName, final Name newName) throws NamingException {
        // Just use the string version for now.
        rename(oldName.toString(), newName.toString());
    }

    public void rename(final String oldName, final String newName) throws NamingException {
        LOGGER.debug("renaming {0}", oldName, newName);
        if(localContext != null) {
            LOGGER.debug("Server mode enabled");
            // Verify that replication is activated
            if(replicationEnabled && clusterViewManager != null
                    && clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE)) {
                // Tests if the object is clustered
                boolean isClustered = clusterViewManager.isClustered(oldName);
                if(isClustered) {
                    // TODO ...
                    LOGGER.error("Cannot rename in the cluster for now !");
                    throw new UnsupportedOperationException("Cannot rename in the cluster for now !");
                }
            }
            localContext.rename(oldName, newName);
        } else {
            LOGGER.warn("Cannot rename because server mode is not activated.");
        }
    }

    public void unbind(final Name name) throws NamingException {
        // Just use the string version for now.
        unbind(name.toString());
    }

    public void unbind(final String name) throws NamingException {
        LOGGER.debug("unbind {0}", name);
        if(localContext != null) {
            LOGGER.debug("Server mode enabled");
            // Verify that replication is activated
            if(replicationEnabled && clusterViewManager != null
                    && clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE)) {
                // Tests if the object is clustered
                boolean isClustered = clusterViewManager.isClustered(name);
                if(isClustered) {
                    LOGGER.debug("The object for name {0} has been removed of the cluster.", name);
                    ((ServerClusterViewManager) clusterViewManager).removeCMIReference(
                            new CMIReference(localRegistry, name));
                }
            }
            localContext.unbind(name);
        } else {
            LOGGER.warn("Cannot unbind because server mode is not activated.");
        }
    }

    /**
     * Insert entry into the cluster view.
     * @param name the name of the object
     * @param obj an object
     * @throws CMIContextException if the object could not be added or updated
     */
    private void bindIntoCluster(final String name, final Object obj) throws CMIContextException {
        // Extracts informations from the object
        ClusteredObjectInfo clusteredObjectInfo;
        if(obj instanceof ClusteredObject) {
            // The object contains informations
            ClusteredObject clusteredObject = (ClusteredObject) obj;
            clusteredObjectInfo = clusteredObject.getClusteredObjectInfo();
        } else if(CMIInfoRepository.containClusteredObjectInfo(name)) {
            // The object doesn't contain informations, but these are registered into the repository
            clusteredObjectInfo = CMIInfoRepository.getClusteredObjectInfo(name);
        } else {
            // The object doesn't contain informations and these are not registered into the repository:
            // the object doesn't belong to a cluster
            LOGGER.debug("The object for name {0} will not be replicated.", name);
            return;
        }
        if(clusteredObjectInfo.isReplicated() &&
                !((ServerClusterViewManager) clusterViewManager).isReplicationManagerStarted()) {
            LOGGER.warn(
                    "the replication manager should be started because the clustered object {0}"
                    + "has a state that must be replicated.", name);
        }
        // Create a new CMIReference for this object
        CMIReference cmiReference = new CMIReference(localRegistry, name);
        // Add it to the cluster view
        try {
            ((ServerClusterViewManager) clusterViewManager).addObjectInstance(clusteredObjectInfo, cmiReference);
        } catch (ServerClusterViewManagerException e) {
            LOGGER.error("Cannot insert {0} into the cluster view", name, e);
            throw new CMIContextException("Cannot insert " + name + " into the cluster view", e);
        }
    }

    /**
     * Initializes the policy for load-balancing of accesses at registry.
     * This policy is used when the replication mode is not enable.
     * @throws CMIContextException if we cannot construct the default LB policy
     * @see #getStaticLBData()
     */
    @SuppressWarnings("unchecked")
    private void setDefaultPolicy() throws CMIContextException {
        try {
            defaultPolicy = new FirstAvailable();
        } catch (Exception e) {
            LOGGER.error("Cannot construct the default policy", e);
            throw new CMIContextException("Cannot construct the default policy", e);
        }
    }

    /**
     * @param providerUrl the provider URL
     * @param initialContextFactoryName a classname of InitialContextFactory to use to create the context
     * @return the context for this provider
     * @throws CMINamingException if the construction of an InitialContext failed
     */
    private Context getRealContext(final String providerUrl, final String initialContextFactoryName)
    throws CMINamingException {
        Hashtable<String, Object> env = new Hashtable<String, Object>();
        env.put(Context.INITIAL_CONTEXT_FACTORY, initialContextFactoryName);
        env.put(Context.PROVIDER_URL, providerUrl);
        try {
            return new InitialContext(env);
        } catch (NamingException e) {
            throw new CMINamingException("Cannot create an initial context", e);
        }
    }

    /**
     * Registers this context in the associated instance of ServerClusterViewManager.
     * @throws CMIContextException if this context is associated with an instance of ClientClusterViewManager
     */
    protected void register() throws CMIContextException {
        if(clusterViewManager != null
                && !(clusterViewManager instanceof ServerClusterViewManager)) {
            LOGGER.error("Cannot register a registry for client managers");
            throw new CMIContextException("Cannot register a registry for client managers");
        }
        if(clusterViewManager != null
                && ((ServerClusterViewManager) clusterViewManager).isRegistrationOpened()
                && (clusterViewManager.getState().equals(ClusterViewManager.State.STARTING)
                        || clusterViewManager.getState().equals(ClusterViewManager.State.STARTED)
                        || clusterViewManager.getState().equals(ClusterViewManager.State.AVAILABLE))) {
            try {
                if(((ServerClusterViewManager) clusterViewManager).addProtocol(
                        initialContextFactoryName, localRegistry, this)) {
                    notCloseable  = true;
                }
            } catch (ServerClusterViewManagerException e) {
                LOGGER.error("Cannot register the local registry", e);
                throw new CMIContextException("Cannot register the local registry", e);
            }
        }
    }

    private Object findClusteredObject(final String name) throws CMIContextException {
        try {
            clusterViewManager.addObjectToWatch(name);
            if(!clusterViewManager.isProxyEnabled(name)) {
                try {
                    return findStubOrProxy(name);
                } catch (NamingException e) {
                    LOGGER.error("Cannot get a proxy (or stub) for {0}.", name, e);
                    throw new CMIContextException("Cannot get a proxy (or stub) for " + name, e);
                }
            }

            // Builds a CMI proxy
            CMIProxy cmiProxy;
            try {
                cmiProxy = CMIProxyFactory.newCMIProxy(clusterViewManager, name, protocol);
            } catch (Exception e) {
                LOGGER.error("Cannot get a CMI proxy for {0}.", name, e);
                throw new CMIContextException("Cannot get a CMI proxy for " + name, e);
            }
            // Returns the CMI proxy
            return cmiProxy;

        } catch (Exception e) {
            LOGGER.error("Cannot find the clustered object object {0}", name, e);
            throw new CMIContextException(
                    "Cannot find the clustered object object " + name, e);
        }
    }

    private Object findStubOrProxy(final String objectName)
    throws CMIContextException, ObjectNotFoundException, NamingException {

        // Choose a server to perform the request.
        Collection<CMIReference> cmiReferences = clusterViewManager.getCMIReferences(objectName, protocol);
        IPolicy<CMIReference> policy = clusterViewManager.getPolicy(objectName);
        CMIReference cmiReference;
        try {
            cmiReference = policy.choose(cmiReferences);
        } catch (NoLoadBalanceableException e) {
            LOGGER.error("Cannot choose a CMIReference in the list {0} with LB policy {1}",
                    cmiReferences, policy, e);
            throw new CMIContextException(
                    "Cannot choose a CMIReference in the list "
                    + cmiReferences + " with LB policy " + policy, e);
        }
        ServerRef serverRef = cmiReference.getServerRef();
        String pURL = serverRef.getProviderURL();
        LOGGER.debug("Chosen providerURL: {0}", pURL);
        Context context;
        context = getRealContext(pURL, initialContextFactoryName);
        return context.lookup(objectName);
    }

}
