/**
 * CMI : Cluster Method Invocation
 * Copyright (C) 2007-2008 Bull S.A.S.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id:CMIInitialContextFactory.java 914 2007-05-25 16:48:16Z loris $
 * --------------------------------------------------------------------------
 */

package org.ow2.cmi.jndi.context;

import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;

import javax.naming.Context;
import javax.naming.spi.InitialContextFactory;

import org.ow2.cmi.admin.MBeanUtils;
import org.ow2.cmi.config.CMIConfig;
import org.ow2.cmi.config.CMIConfigException;
import org.ow2.cmi.config.CMIConfigurator;
import org.ow2.cmi.config.JNDIConfig;
import org.ow2.cmi.config.ProtocolConfig;
import org.ow2.cmi.config.ProviderUrls;
import org.ow2.cmi.controller.client.ClientClusterViewManager;
import org.ow2.cmi.controller.common.ClusterViewManager;
import org.ow2.cmi.controller.common.ClusterViewManagerException;
import org.ow2.cmi.controller.factory.ClusterViewManagerFactory;
import org.ow2.cmi.controller.factory.ClusterViewManagerFactoryConfig;
import org.ow2.cmi.controller.factory.ClusterViewManagerFactoryException;
import org.ow2.cmi.controller.server.ProtocolNotFoundException;
import org.ow2.cmi.controller.server.ServerClusterViewManager;
import org.ow2.cmi.reference.ServerId;
import org.ow2.cmi.reference.ServerRef;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;


/**
 * CMIInitialContext factory.
 * @author The new CMI team
 * @see CMIContext
 */
public final class CMIInitialContextFactory implements InitialContextFactory {

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(CMIInitialContextFactory.class);


    /**
     * Create a CMI context.<br>
     * Because the JNDI environment is provided in parameter,
     * if the expected properties are not set, a NamingException will be thrown.
     * @param environment The needed properties to create the CMI context.
     * @return A CMI context
     * @throws CMINamingException If cannot create an initial context.
     */
    public Context getInitialContext(final Hashtable<?, ?> environment) throws CMINamingException {

        ClusterViewManagerFactory clusterViewManagerFactory =
            ClusterViewManagerFactory.getFactory();
        ClusterViewManager clusterViewManager =
            clusterViewManagerFactory.getClusterViewManager();

        boolean replicationEnabled;
        CMIConfig<?> cmiConfig = null;

        // Required parameters to build a CMI Context
        ServerId registryId = null;
        String wrappedFactoryName = null;
        String protocol = null;
        List<ServerRef> serverRefs = null;

        // Check if a manager already exists
        if(clusterViewManager != null) {
            replicationEnabled = clusterViewManager instanceof ServerClusterViewManager;
            if(replicationEnabled) {
                try {
                    cmiConfig = CMIConfigurator.findAndExtractCMIConfig(environment, true);
                } catch (CMIConfigException e) {
                    logger.error("Unable to retrieve the CMI configuration", e);
                    throw new CMINamingException("Unable to retrieve the CMI configuration", e);
                }
                Set<String> protocols;
                protocol = JNDIConfig.getCurrentProtocolName();
                // Check it
                if(protocol == null) {
                    protocols = clusterViewManager.getProtocols();
                    if(protocols.size() == 1) {
                        protocol = (String) protocols.toArray()[0];
                    }
                }

                // Check if the protocol was already registered
                if(protocol != null && ((ServerClusterViewManager) clusterViewManager).hasProtocol(protocol)) {
                    try {
                        wrappedFactoryName =
                            ((ServerClusterViewManager) clusterViewManager).getInitialContextFactoryName(protocol);
                        registryId = ((ServerClusterViewManager) clusterViewManager).getRefOnLocalRegistry(protocol);
                    } catch (ProtocolNotFoundException e) {
                        logger.error("Unable to retrieve the protocol config", e);
                        throw new CMINamingException("Unable to retrieve the protocol config", e);
                    }
                } else {
                    // Retrieve the JNDI configuration to create a context
                    JNDIConfig jndiConfig = cmiConfig.getJNDIConfig();
                    // Check it
                    if(jndiConfig == null) {
                        logger.error("JNDI config missing!");
                        throw new CMINamingException("JNDI config missing!");
                    }
                    Set<ProtocolConfig> protocolConfigs = jndiConfig.getProtocolConfigs();
                    if(protocolConfigs == null) {
                        logger.error("Protocol configs missing!");
                        throw new CMINamingException("Protocol configs missing!");
                    }
                    ProtocolConfig currentProtocolConfig = null;
                    if(protocol != null) {
                        for(ProtocolConfig protocolConfig : protocolConfigs) {
                            if(protocolConfig.getName().equals(protocol)) {
                                currentProtocolConfig = protocolConfig;
                            }
                        }
                    } else if(protocolConfigs.size() == 1) {
                        currentProtocolConfig = (ProtocolConfig) protocolConfigs.toArray()[0];
                        protocol = currentProtocolConfig.getName();
                    }
                    if(currentProtocolConfig == null) {
                        logger.error("Protocol config missing!");
                        throw new CMINamingException("Protocol config missing!");
                    }
                    wrappedFactoryName = currentProtocolConfig.getInitialContextFactoryName();
                    // Check it
                    if(wrappedFactoryName == null) {
                        logger.error("Missing the initial context factory for protocol " + protocol);
                        throw new CMINamingException("Missing the initial context factory for protocol " + protocol);
                    } else if(wrappedFactoryName.equals(CMIInitialContextFactory.class.getName())) {
                        logger.error("A CMI context cannot wrap an other CMI context!!");
                        throw new CMINamingException("A CMI context cannot wrap an other CMI context!!");
                    }
                    String localRegistry;
                    localRegistry = currentProtocolConfig.getLocalRegistryUrl();
                    // Check it
                    if(localRegistry == null) {
                        logger.error("Server cluster view manager requires a local registry!");
                        throw new CMINamingException("Server cluster view manager requires a local registry!");
                    }
                    try {
                        registryId =
                            new ServerId(MBeanUtils.getMBeanDomainName(),
                                    MBeanUtils.getMBeanServerName(), protocol, localRegistry);
                    } catch (MalformedURLException e) {
                        logger.error("The following provider URL is malformed {0}", localRegistry, e);
                        throw new CMINamingException("The following provider URL is malformed " + localRegistry, e);
                    } catch (UnknownHostException e) {
                        logger.error("The host name of the provider URL {0} cannot be resolved", localRegistry, e);
                        throw new CMINamingException(
                                "The host name of the provider URL " + localRegistry + " cannot be resolved", e);
                    }
                }
            } else {
                protocol = ((ClientClusterViewManager) clusterViewManager).getProtocolName();
                wrappedFactoryName = ((ClientClusterViewManager) clusterViewManager).getInitialContextFactoryName();
                serverRefs = ((ClientClusterViewManager) clusterViewManager).getProviderURLs();
            }
        } else {
            try {
                cmiConfig = CMIConfigurator.findAndExtractCMIConfig(environment, false);
            } catch (CMIConfigException e) {
                logger.error("Unable to retrieve the CMI configuration", e);
                throw new CMINamingException("Unable to retrieve the CMI configuration", e);
            }
            ClusterViewManagerFactoryConfig<?> clusterViewManagerFactoryConfig =
                cmiConfig.getClusterViewManagerFactoryConfig();
            replicationEnabled =
                 clusterViewManagerFactoryConfig instanceof org.ow2.cmi.controller.factory.server.ClusterViewManagerFactoryConfig;
            if(replicationEnabled && clusterViewManagerFactoryConfig == null) {
                logger.error("Unable to retrieve the CMI configuration");
                throw new CMINamingException("Unable to retrieve the CMI configuration");
            }
            protocol = JNDIConfig.getCurrentProtocolName();
            // Retrieve the JNDI configuration to create a context
            JNDIConfig jndiConfig = cmiConfig.getJNDIConfig();
            // Check it
            if(jndiConfig == null) {
                logger.error("JNDI config missing!");
                throw new CMINamingException("JNDI config missing!");
            }
            Set<ProtocolConfig> protocolConfigs = jndiConfig.getProtocolConfigs();
            if(protocolConfigs == null) {
                logger.error("Protocol configs missing!");
                throw new CMINamingException("Protocol configs missing!");
            }
            ProtocolConfig currentProtocolConfig = null;
            for(ProtocolConfig protocolConfig : protocolConfigs) {
                if(protocolConfig.getName().equals(protocol)) {
                    currentProtocolConfig = protocolConfig;
                }
            }
            // Check it
            if(currentProtocolConfig == null) {
                logger.error("Protocol config missing!");
                throw new CMINamingException("Protocol config missing!");
            }

            wrappedFactoryName = currentProtocolConfig.getInitialContextFactoryName();
            // Check it
            if(wrappedFactoryName == null) {
                logger.error("Missing the initial context factory for protocol " + protocol);
                throw new CMINamingException("Missing the initial context factory for protocol " + protocol);
            }

            ProviderUrls providerURLs = currentProtocolConfig.getProviderUrls();
            // Check it
            if(!replicationEnabled && providerURLs != null && !providerURLs.getProviderUrls().isEmpty()) {
                try {
                    serverRefs = CMIConfigurator.extractServerRefs(protocol, providerURLs.getProviderUrls());
                } catch (CMIConfigException e) {
                    logger.error("Unable to extract some server refs from the provider URL " + providerURLs.getProviderUrls(), e);
                    throw new CMINamingException(
                            "Unable to extract some server refs from the provider URL " + providerURLs.getProviderUrls(), e);
                }
            } else if(!replicationEnabled) {
                logger.error("Missing the initial provider URLs for protocol " + protocol);
                throw new CMINamingException("Missing the initial provider URLs for protocol " + protocol);
            }

            if(replicationEnabled) {
                String localRegistry = currentProtocolConfig.getLocalRegistryUrl();
                // Check it
                if(localRegistry == null) {
                    logger.error("Server cluster view manager requires a local registry!");
                    throw new CMINamingException("Server cluster view manager requires a local registry!");
                }
                try {
                    registryId =
                        new ServerId(MBeanUtils.getMBeanDomainName(),
                                MBeanUtils.getMBeanServerName(), protocol, localRegistry);
                } catch (MalformedURLException e) {
                    logger.error("The following provider URL is malformed {0}", localRegistry, e);
                    throw new CMINamingException("The following provider URL is malformed " + localRegistry, e);
                } catch (UnknownHostException e) {
                    logger.error("The host name of the provider URL {0} cannot be resolved", localRegistry, e);
                    throw new CMINamingException(
                            "The host name of the provider URL " + localRegistry + " cannot be resolved", e);
                }
            }
        }

        if(clusterViewManager == null) {
            if(cmiConfig == null) {
                logger.error("No CMI configuration");
                throw new CMINamingException("No CMI configuration");
            }
            // The manager doesn't yet exist
            try {
                CMIConfigurator.configure(clusterViewManagerFactory, cmiConfig);
            } catch (CMIConfigException e) {
                logger.error("Unable to configure the cluster view manager factory", e);
                throw new CMINamingException(
                        "Unable to configure the cluster view manager factory", e);
            }
            try {
                clusterViewManager = clusterViewManagerFactory.create();
            } catch (ClusterViewManagerFactoryException e) {
                logger.error("Unable to get the cluster view manager", e);
                throw new CMINamingException(
                        "Unable to get the cluster view manager", e);
            }
        }

        if(clusterViewManager != null
                && clusterViewManager.getState().equals(ClusterViewManager.State.STOPPED)
                && cmiConfig.getClusterViewManagerFactoryConfig().isAutoStarted()) {
            try {
                clusterViewManager.start();
            } catch (ClusterViewManagerException e) {
                logger.error("Unable to start the cluster view manager", e);
                throw new CMINamingException(
                        "Unable to start the cluster view manager", e);
            }
        }

        if(wrappedFactoryName == null || protocol == null) {
            logger.error("Unable to build a CMI context!!");
            throw new CMINamingException("Unable to build a CMI context!!");
        }
        CMIContext cmiContext = new CMIContext(
                clusterViewManager, registryId, wrappedFactoryName, protocol, serverRefs);

        if(replicationEnabled) {
            try {
                cmiContext.register();
            } catch (CMIContextException e) {
                logger.error("Cannot register the context", e);
                throw new CMINamingException("Cannot register the context", e);
            }
        }
        return cmiContext;
    }

}
