/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.api.relationship;

import org.reflext.api.TypeInfo;
import org.reflext.api.ArrayTypeInfo;
import org.reflext.api.ClassTypeInfo;
import org.reflext.api.ParameterizedTypeInfo;
import org.reflext.api.VoidTypeInfo;
import org.reflext.api.TypeVariableInfo;
import org.reflext.api.WildcardTypeInfo;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class SubType implements TypeRelationship<TypeInfo, TypeInfo> {

  /**
   * The sub type relationship is satisfied when the <tt>t1</tt> type extends the
   * <tt>t2</tt> type.
   *
   * @param t1 the first type
   * @param t2 the second type
   * @return the satisfaction of the relationship
   */
  public boolean isSatisfied(TypeInfo t1, TypeInfo t2) {
    return isSubType(t1, t2);
  }

  public boolean isSubType(TypeInfo subTI, TypeInfo superTI) {
    if (superTI instanceof ArrayTypeInfo) {
      return isSubType(subTI, (ArrayTypeInfo)superTI);
    } else if (superTI instanceof ClassTypeInfo) {
      return isSubType(subTI, (ClassTypeInfo)superTI);
    } else if (superTI instanceof ParameterizedTypeInfo) {
      return isSubType(subTI, (ParameterizedTypeInfo)superTI);
    } else if (superTI instanceof VoidTypeInfo) {
      return isSubType(subTI, (VoidTypeInfo)superTI);
    } else if (superTI instanceof TypeVariableInfo) {
      return isSubType(subTI, (TypeVariableInfo)superTI);
    } else {
      throw new AssertionError();
    }
  }

  public boolean isSubType(TypeInfo subTI, ClassTypeInfo superCTI) {
    if (subTI instanceof ClassTypeInfo) {
      ClassTypeInfo cti = (ClassTypeInfo)subTI;
      return superCTI.isAssignableFrom(cti);
    } else if (subTI instanceof TypeVariableInfo) {
      TypeVariableInfo tvi = (TypeVariableInfo)subTI;
      return isSubType(tvi.getBounds().get(0), superCTI);
    } else if (subTI instanceof ParameterizedTypeInfo) {
      ParameterizedTypeInfo pti = (ParameterizedTypeInfo)subTI;
      return isSubType(pti.getRawType(), superCTI);
    } else if (subTI instanceof WildcardTypeInfo) {
      throw new AssertionError();
    }
    return false;
  }

  public boolean isSubType(TypeInfo subTI, VoidTypeInfo superVTI) {
    return subTI instanceof VoidTypeInfo;
  }

  public boolean checkVariableExtension(TypeVariableInfo superTVI, TypeVariableInfo subTVI, TypeInfo ti) {
    if (ti instanceof ParameterizedTypeInfo) {
      ParameterizedTypeInfo pti = (ParameterizedTypeInfo)ti;
      for (int i = 0;i < pti.getTypeArguments().size();i++) {
        TypeInfo ptiTypeArg = pti.getTypeArguments().get(i);
        if (ptiTypeArg.equals(subTVI)) {
          TypeInfo ptiRawType = pti.getRawType();
          if (ptiRawType instanceof ClassTypeInfo) {
            ClassTypeInfo ptiRawTypeCTI = (ClassTypeInfo)ptiRawType;
            if (isSubType(ptiRawTypeCTI.getTypeParameters().get(i), superTVI)) {
              return true;
            }
          }
        }
      }
    }
    return false;
  }

  public boolean isSubType(TypeInfo subTI, TypeVariableInfo superTVI) {
    if (subTI instanceof TypeVariableInfo) {
      TypeVariableInfo extendsTVI = (TypeVariableInfo)subTI;
      if (superTVI.equals(subTI)) {
        return true;
      } else {
        ClassTypeInfo decl = (ClassTypeInfo)extendsTVI.getGenericDeclaration();
        TypeInfo declSuperTI = decl.getSuperType();
        if (checkVariableExtension(superTVI, extendsTVI, declSuperTI)) {
          return true;
        }
        for (TypeInfo declInterfaceTI : decl.getInterfaces()) {
          if (checkVariableExtension(superTVI, extendsTVI, declInterfaceTI)) {
            return true;
          }
        }
      }
      return false;
    }  else if (subTI instanceof ClassTypeInfo) {
      TypeInfo bound = superTVI.getBounds().get(0);
      return isSubType(subTI, bound);
    } else {
      throw new UnsupportedOperationException();
    }
  }

  public boolean isSubType(TypeInfo subTI, ParameterizedTypeInfo superPTI) {
    return isSubType(subTI, superPTI.getRawType());
  }

  public boolean isSubType(TypeInfo subTI, ArrayTypeInfo superATI) {
    if (subTI instanceof ArrayTypeInfo) {
      ArrayTypeInfo extendsATI = (ArrayTypeInfo)subTI;
      return isSubType(extendsATI.getComponentType(), superATI.getComponentType());
    } else {
      return false;
    }
  }
}
