/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.api;

import java.util.*;

/**
 * The signature of method, identifies a method.
 *
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class MethodSignature {

  /** . */
  private static final String[] NO_PARAMETERS = new String[0];

  /** . */
  private final String name;

  /** . */
  private final String[] parameterRawNames;

  public MethodSignature(String name, Class<?>... parameterTypes) {
    if (name == null) {
      throw new NullPointerException("No null name accepted");
    }
    if (parameterTypes == null) {
      throw new NullPointerException("No null name accepted");
    }

    //
    String[] parameterRawNames;
    if (parameterTypes.length == 0) {
      parameterRawNames = NO_PARAMETERS;
    } else {
      parameterRawNames = new String[parameterTypes.length];
      for (int i = 0;i < parameterTypes.length;i++) {
        Class<?> parameterType = parameterTypes[i];
        if (parameterType == null) {
          throw new IllegalArgumentException("No null parameter type allowed");
        }
        parameterRawNames[i] = parameterType.getName();
      }
    }

    //
    this.name = name;
    this.parameterRawNames = parameterRawNames;
  }

  public MethodSignature(String name, List<TypeInfo> parameterTypes) {
    if (name == null) {
      throw new NullPointerException("No null name accepted");
    }
    if (parameterTypes == null) {
      throw new NullPointerException("No null name accepted");
    }

    //
    String[] parameterRawNames;
    if (parameterTypes.size() == 0) {
      parameterRawNames = NO_PARAMETERS;
    } else {
      parameterRawNames = new String[parameterTypes.size()];
      int j = 0;
      for (TypeInfo parameterType : parameterTypes) {
        if (parameterType == null) {
          throw new IllegalArgumentException("No null parameter type allowed");
        }
        parameterRawNames[j++] = computeRawName(parameterType);
      }
    }

    //
    this.name = name;
    this.parameterRawNames = parameterRawNames;
  }

  public String getName() {
    return name;
  }

  private String computeRawName(TypeInfo type) {
    if (type instanceof ClassTypeInfo) {
      return ((ClassTypeInfo)type).getName();
    } else if (type instanceof ParameterizedTypeInfo) {
      return computeRawName(((ParameterizedTypeInfo)type).getRawType());
    } else if (type instanceof TypeVariableInfo) {
      return computeRawName(((TypeVariableInfo)type).getBounds().get(0));
    } else if (type instanceof ArrayTypeInfo) {
      return "[L" + computeRawName(((ArrayTypeInfo)type).getComponentType()) + ";";
    } else {
      throw new AssertionError();
    }
  }

  @Override
  public int hashCode() {
    return name.hashCode() ^ Arrays.hashCode(parameterRawNames);
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj instanceof MethodSignature) {
      MethodSignature that = (MethodSignature)obj;
      return name.equals(that.name) && Arrays.equals(parameterRawNames, that.parameterRawNames);
    }
    return false;
  }

  @Override
  public String toString() {
    return "MethodSignature[name=" + name + ",parameterRawNames=" + Arrays.toString(parameterRawNames) + "]";
  }
}
