/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.core;

import org.reflext.api.AccessScope;
import org.reflext.api.ClassTypeInfo;
import org.reflext.api.MethodInfo;
import org.reflext.api.MethodSignature;
import org.reflext.api.TypeInfo;
import org.reflext.api.TypeVariableInfo;

import java.util.List;
import java.util.ArrayList;
import java.lang.annotation.Annotation;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
class MethodInfoImpl<T, M> extends ReflectedObject<T, M> implements MethodInfo {

  /** . */
  private final M method;

  /** . */
  private final String name;

  /** . */
  private TypeInfo returnType;

  /** . */
  private List<TypeInfo> parameterTypes;

  /** . */
  private final AccessScope access;

  /** . */
  private final boolean _abstract;

  /** . */
  private final boolean _final;

  /** . */
  private final boolean _static;

  /** . */
  private final ClassTypeInfo owner;

  /** . */
  private MethodSignature signature;

  /** . */
  private List<TypeVariableInfo> typeParameters;

  public MethodInfoImpl(ClassTypeInfo owner, TypeDomain<T, M> domain, M method) {
    super(domain);

    //
    this.method = method;
    this.name = domain.methodModel.getName(method);
    this.access = domain.methodModel.getAccess(method);
    this._abstract = domain.methodModel.isAbstract(method);
    this._final = domain.methodModel.isFinal(method);
    this._static = domain.methodModel.isStatic(method);
    this.owner = owner;
    this.signature = null;
    this.typeParameters = null;
  }

  public M getMethod() {
    return method;
  }

  public TypeInfo getReturnType() {
    if (returnType == null) {
      T rt = domain.methodModel.getReturnType(method);
      returnType = domain.getType(rt);
    }
    return returnType;
  }

  public List<TypeInfo> getParameterTypes() {
    if (parameterTypes == null) {
      List<TypeInfo> parameterTypes = new ArrayList<TypeInfo>();
      for (T parameterType : domain.methodModel.getParameterTypes(method)) {
        parameterTypes.add(domain.getType(parameterType));
      }
      this.parameterTypes = parameterTypes;
    }
    return parameterTypes;
  }

  public AccessScope getAccess() {
    return access;
  }

  public boolean isAbstract() {
    return _abstract;
  }

  public boolean isStatic() {
    return _static;
  }

  public boolean isFinal() {
    return _final;
  }

  public String getName() {
    return name;
  }

  public ClassTypeInfo getOwner() {
    return owner;
  }

  public List<TypeVariableInfo> getTypeParameters() {
    if (typeParameters == null) {
      ArrayList<TypeVariableInfo> typeParameters = new ArrayList<TypeVariableInfo>();
      for (T tv : domain.methodModel.getTypeParameters(method)) {
        TypeVariableInfoImpl<T, M> typeParameter = (TypeVariableInfoImpl<T, M>)domain._getType(tv);
        typeParameters.add(typeParameter);
      }
      this.typeParameters = typeParameters;
    }
    return typeParameters;
  }

  public MethodSignature getSignature() {
    if (signature == null) {
/*
      List<TypeInfo> parameterTypes = new ArrayList<TypeInfo>();
      for (TypeInfo parameterType : getParameterTypes()) {
        ClassTypeInfo rawClassType = parameterType.getRawClassType();
        parameterTypes.add(rawClassType);
      }
*/
      signature = new MethodSignature(name, getParameterTypes());
    }
    return signature;
  }

  @Override
  public int hashCode() {
    return name.hashCode();
  }

  @Override
  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj instanceof MethodInfo) {
      MethodInfo that = (MethodInfo)obj;
      String thatName = that.getName();
      MethodSignature signature = getSignature();
      MethodSignature thatSignature = that.getSignature();
      return name.equals(thatName) && signature.equals(thatSignature);
    }
    return false;
  }

  public <A extends Annotation> A getDeclaredAnnotation(Class<A> annotationClass) {
    return domain.methodModel.getDeclaredAnnotation(method, annotationClass);
  }

  @Override
  public String toString() {
    return "MethodInfo[name=" + name + "]";
  }
}
