/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.core;

import org.reflext.api.ClassTypeInfo;
import org.reflext.api.TypeInfo;
import org.reflext.api.TypeVariableInfo;
import org.reflext.api.GenericDeclarationInfo;
import org.reflext.api.MethodInfo;
import org.reflext.spi.model.GenericDeclarationKind;

import java.util.List;
import java.util.ArrayList;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
class TypeVariableInfoImpl<T, M> extends TypeInfoImpl<T, M> implements TypeVariableInfo {

  /** . */
  private final T type;

  /** . */
  private final String name;

  /** . */
  private GenericDeclarationInfo genericDeclaration;

  /** . */
  private List<TypeInfo> bounds;

  public TypeVariableInfoImpl(TypeDomain<T, M> domain, T type) {
    super(domain);

    //
    this.name = domain.typeModel.getName(type);
    this.type = type;
    this.genericDeclaration = null;
    this.bounds = null;
  }

  public T getType() {
    return type;
  }

  public List<TypeInfo> getBounds() {
    if (bounds == null) {
      ArrayList<TypeInfo> bounds = new ArrayList<TypeInfo>();
      for (T b : domain.typeModel.getBounds(type)) {
        TypeInfoImpl<T, M> bound = domain._getType(b);
        bounds.add(bound);
      }
      this.bounds = bounds;
    }
    return bounds;
  }

  public GenericDeclarationInfo getGenericDeclaration() {
    if (genericDeclaration == null) {
      GenericDeclarationKind kind = domain.typeModel.getGenericDeclarationKind(type);
      switch (kind) {
        case TYPE:
          T gd = domain.typeModel.getGenericDeclaration(type);
          genericDeclaration = (ClassTypeInfo)domain.getType(gd);
          break;
        case METHOD:
          M mgd = domain.methodModel.getGenericDeclaration(type);
          T omgd = domain.methodModel.getOwner(mgd);
          ClassTypeInfo tmp = (ClassTypeInfo)domain.getType(omgd);
          for (MethodInfo mi : tmp.getDeclaredMethods()) {
            if (mi.getMethod().equals(mgd)) {
              genericDeclaration = mi;
            }
          }
          if (genericDeclaration == null) {
            throw new AssertionError("Need to handle that case which could happen due to covariant return types...");
          }
          break;
        default:
          throw new UnsupportedOperationException();
      }
    }
    return genericDeclaration;
  }

  public String getName() {
    return name;
  }

  public int hashCode() {
    return name.hashCode() ^ getGenericDeclaration().hashCode();
  }

  public boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (obj instanceof TypeVariableInfo) {
      TypeVariableInfo that = (TypeVariableInfo)obj;
      GenericDeclarationInfo generidDeclaration = getGenericDeclaration();
      GenericDeclarationInfo thatGenericDeclaration = that.getGenericDeclaration();
      String thatName = that.getName();
      return name.equals(thatName) && generidDeclaration.equals(thatGenericDeclaration);
    }
    return false;
  }

  @Override
  public String toString() {
    return "TypeVariableInfo[name=" + name + "]";
  }
}
