/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.jlr;

import org.reflext.spi.model.TypeModel;
import org.reflext.api.LiteralType;
import org.reflext.api.ClassKind;
import org.reflext.spi.model.TypeKind;
import org.reflext.spi.model.GenericDeclarationKind;

import java.lang.reflect.Type;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.GenericDeclaration;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.HashMap;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collections;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class JavaLangReflectTypeModel implements TypeModel<Type> {

  /** . */
  private static final Map<Class<?>, LiteralType> primitiveMap = new HashMap<Class<?>, LiteralType>();

  {
    primitiveMap.put(Boolean.class, LiteralType.BOOLEAN);
    primitiveMap.put(boolean.class, LiteralType.BOOLEAN);
    primitiveMap.put(Byte.class, LiteralType.BYTE);
    primitiveMap.put(byte.class, LiteralType.BYTE);
    primitiveMap.put(Short.class, LiteralType.SHORT);
    primitiveMap.put(short.class, LiteralType.SHORT);
    primitiveMap.put(Integer.class, LiteralType.INT);
    primitiveMap.put(int.class, LiteralType.INT);
    primitiveMap.put(Long.class, LiteralType.LONG);
    primitiveMap.put(long.class, LiteralType.LONG);
    primitiveMap.put(Float.class, LiteralType.FLOAT);
    primitiveMap.put(float.class, LiteralType.FLOAT);
    primitiveMap.put(Double.class, LiteralType.DOUBLE);
    primitiveMap.put(double.class, LiteralType.DOUBLE);
  }

  public TypeKind getKind(Type type) {
    if (type instanceof Class) {
      Class classType = (Class)type;
      if (classType.isArray()) {
        return TypeKind.ARRAY;
      } else {
        if (classType.isPrimitive()) {
          if (classType == void.class) {
            return TypeKind.VOID;
          } else {
            return TypeKind.SIMPLE;
          }
        } else if (
          classType == Boolean.class ||
          classType == Float.class ||
          classType == Byte.class ||
          classType == Short.class ||
          classType == Integer.class ||
          classType == Long.class ||
          classType == Double.class) {
          return TypeKind.SIMPLE;
        } else if (classType == Void.class) {
          return TypeKind.VOID;
        } else {
          return TypeKind.CLASS;
        }
      }
    } else if (type instanceof ParameterizedType) {
      return TypeKind.PARAMETERIZED;
    } else if (type instanceof TypeVariable) {
      return TypeKind.VARIABLE;
    } else if (type instanceof WildcardType) {
      return TypeKind.WILDCARD;
    } else if (type instanceof GenericArrayType) {
      return TypeKind.ARRAY;
    } else {
      throw new AssertionError();
    }
  }

  public LiteralType getLiteralType(Type simpleType) {
    Class classType = (Class)simpleType;
    return primitiveMap.get(classType);
  }

  public Type getComponentType(Type arrayType) {
    if (arrayType instanceof Class) {
      return ((Class)arrayType).getComponentType();
    } else {
      return ((GenericArrayType)arrayType).getGenericComponentType();
    }
  }

  public boolean isPrimitive(Type simpleType) {
    Class classType = (Class)simpleType;
    return classType.isPrimitive();
  }

  public String getClassName(Type classType) {
    return ((Class<?>)classType).getName();
  }

  public ClassKind getClassKind(Type classType) {
    Class tmp = (Class)classType;
    if (tmp.isAnnotation()) {
      return ClassKind.ANNOTATION;
    } else if (tmp.isEnum()) {
      return ClassKind.ENUM;
    } else if (tmp.isInterface()) {
      return ClassKind.INTERFACE;
    } else {
      return ClassKind.CLASS;
    }
  }

  public Iterable<Type> getInterfaces(Type classType) {
    return Arrays.asList(((Class<?>)classType).getGenericInterfaces());
  }

  public Type getSuperClass(Type classType) {
    return ((Class<?>)classType).getGenericSuperclass();
  }

  public Iterable<Type> getTypeParameters(Type classType) {
    ArrayList<Type> typeParameters = new ArrayList<Type>();
    for (TypeVariable typeParameter : ((Class)classType).getTypeParameters()) {
      typeParameters.add(typeParameter);
    }
    return typeParameters;
  }

  public Type getGenericDeclaration(Type typeVariable) {
    return  (Type)((TypeVariable)typeVariable).getGenericDeclaration();
  }

  public GenericDeclarationKind getGenericDeclarationKind(Type typeVariable) {
    GenericDeclaration genDecl = ((TypeVariable)typeVariable).getGenericDeclaration();
    if (genDecl instanceof Type) {
      return GenericDeclarationKind.TYPE;
    } else if (genDecl instanceof Method) {
      return GenericDeclarationKind.METHOD;
    } else {
      throw new UnsupportedOperationException("Not yet implemented");
    }
  }

  public String getName(Type typeVariable) {
    return ((TypeVariable)typeVariable).getName();
  }

  public Iterable<Type> getBounds(Type typeVariable) {
    return Arrays.asList(((TypeVariable)typeVariable).getBounds());
  }

  public Type getRawType(Type parameterizedType) {
    return ((ParameterizedType)parameterizedType).getRawType();
  }

  public Iterable<Type> getTypeArguments(Type parameterizedType) {
    return Arrays.asList(((ParameterizedType)parameterizedType).getActualTypeArguments());
  }

  public Iterable<Type> getUpperBounds(Type wildcardType) {
    Type[] upperBounds = ((WildcardType)wildcardType).getUpperBounds();
    if (upperBounds.length == 1 && upperBounds[0] == Object.class) {
      return Collections.emptyList();
    } else {
      return Arrays.asList(upperBounds);
    }
  }

  public Iterable<Type> getLowerBounds(Type wildcardType) {
    return Arrays.asList(((WildcardType)wildcardType).getLowerBounds());
  }
}
