/*
 * Copyright (C) 2003-2009 eXo Platform SAS.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.reflext.test.introspector.method;

import org.reflext.api.ClassTypeInfo;
import org.reflext.api.visit.HierarchyScope;
import org.reflext.test.ReflectUnitTest;
import org.reflext.api.TypeInfo;
import org.reflext.api.MethodInfo;
import org.reflext.api.introspection.MethodIntrospector;

import java.util.Collection;
import java.util.Collections;
import java.util.Set;
import java.util.HashSet;

/**
 * @author <a href="mailto:julien.viet@exoplatform.com">Julien Viet</a>
 * @version $Revision$
 */
public class MethodIntrospectorUnitTest extends ReflectUnitTest {

  protected void execute() throws Exception {
    testGetterWithScopeAll();
    testGetterWithScopeAncestors();
    testGetterWithScopeClass();
    testExactOverrideClass();
    testCovariantReturnTypeClass();
    testCovariantReturnTypeFull();
    testTypeVariableOverrideClass();
    testTypeVariableOverrideFull();
    testCovariantReturnTypeWithParallelHierarchy();
    testNPE();
  }

  private static <T> Set<T> set(Iterable<T> iterable) {
    HashSet<T> set = new HashSet<T>();
    for (T t : iterable) {
      if (set.contains(t)) {
        fail("Set " + set + " should not contain the element " + t);
      }
      set.add(t);
    }
    return set;
  }

  private void testGetterWithScopeAll() {
    ClassTypeInfo ti = (ClassTypeInfo)getTypeInfo("F1");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.ALL, true);

    //
    Set<MethodInfo> expectedGetters = new HashSet<MethodInfo>();
    expectedGetters.add(getMethodInfo("F1#getDeclaredString"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredPrimitiveBoolean1"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredPrimitiveBoolean2"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredPrimitiveBoolean3"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredPrimitiveBoolean3"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredBoolean1"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredBoolean2"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredBoolean3"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredBoolean3"));
    expectedGetters.add(getMethodInfo("F1#getOverridenInterface"));
    expectedGetters.add(getMethodInfo("F1#getOverridenParent"));
    expectedGetters.add(getMethodInfo("F2#getParentString"));
    expectedGetters.add(getMethodInfo("F3#getInterfaceString"));
    assertEquals(expectedGetters, set(spector.getGetters(ti)));

    //
    Set<MethodInfo> expectedSetters = new HashSet<MethodInfo>();
    expectedSetters.add(getMethodInfo("F1#setDeclaredString"));
    expectedSetters.add(getMethodInfo("F1#setDeclaredPrimitiveBoolean"));
    expectedSetters.add(getMethodInfo("F1#setDeclaredBoolean"));
    expectedSetters.add(getMethodInfo("F1#setOverridenInterface"));
    expectedSetters.add(getMethodInfo("F1#setOverridenParent"));
    expectedSetters.add(getMethodInfo("F2#setParentString"));
    expectedSetters.add(getMethodInfo("F3#setInterfaceString"));
    assertEquals(expectedSetters, set(spector.getSetters(ti)));
  }

  private void testGetterWithScopeAncestors() {
    ClassTypeInfo ti = (ClassTypeInfo)getTypeInfo("F1");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.ANCESTORS, true);

    //
    Set<MethodInfo> expectedGetters = new HashSet<MethodInfo>();
    expectedGetters.add(getMethodInfo("F1#getDeclaredString"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredPrimitiveBoolean1"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredPrimitiveBoolean2"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredPrimitiveBoolean3"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredPrimitiveBoolean3"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredBoolean1"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredBoolean2"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredBoolean3"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredBoolean3"));
    expectedGetters.add(getMethodInfo("F1#getOverridenInterface"));
    expectedGetters.add(getMethodInfo("F1#getOverridenParent"));
    expectedGetters.add(getMethodInfo("F2#getParentString"));
    assertEquals(expectedGetters, set(spector.getGetters(ti)));

    //
    Set<MethodInfo> expectedSetters = new HashSet<MethodInfo>();
    expectedSetters.add(getMethodInfo("F1#setDeclaredString"));
    expectedSetters.add(getMethodInfo("F1#setDeclaredPrimitiveBoolean"));
    expectedSetters.add(getMethodInfo("F1#setDeclaredBoolean"));
    expectedSetters.add(getMethodInfo("F1#setOverridenInterface"));
    expectedSetters.add(getMethodInfo("F1#setOverridenParent"));
    expectedSetters.add(getMethodInfo("F2#setParentString"));
    assertEquals(expectedSetters, set(spector.getSetters(ti)));
  }

  private void testGetterWithScopeClass() {
    ClassTypeInfo ti = (ClassTypeInfo)getTypeInfo("F1");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.CLASS, true);

    //
    Set<MethodInfo> expectedGetters = new HashSet<MethodInfo>();
    expectedGetters.add(getMethodInfo("F1#getDeclaredString"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredPrimitiveBoolean1"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredPrimitiveBoolean2"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredPrimitiveBoolean3"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredPrimitiveBoolean3"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredBoolean1"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredBoolean2"));
    expectedGetters.add(getMethodInfo("F1#getDeclaredBoolean3"));
    expectedGetters.add(getMethodInfo("F1#isDeclaredBoolean3"));
    expectedGetters.add(getMethodInfo("F1#getOverridenInterface"));
    expectedGetters.add(getMethodInfo("F1#getOverridenParent"));
    assertEquals(expectedGetters, set(spector.getGetters(ti)));

    //
    Set<MethodInfo> expectedSetters = new HashSet<MethodInfo>();
    expectedSetters.add(getMethodInfo("F1#setDeclaredString"));
    expectedSetters.add(getMethodInfo("F1#setDeclaredPrimitiveBoolean"));
    expectedSetters.add(getMethodInfo("F1#setDeclaredBoolean"));
    expectedSetters.add(getMethodInfo("F1#setOverridenInterface"));
    expectedSetters.add(getMethodInfo("F1#setOverridenParent"));
    assertEquals(expectedSetters, set(spector.getSetters(ti)));
  }

  private void testCovariantReturnTypeWithParallelHierarchy() {
    TypeInfo bti = getTypeInfo("E2Impl");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.ALL, true);
    Collection mis = spector.getMethods(bti);
    assertEquals(Collections.singleton(getMethodInfo("E2Impl_m1")), mis);
  }

  public void testExactOverrideClass() {
    TypeInfo bti = getTypeInfo("D2");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.CLASS);
    Collection mis = spector.getMethods(bti);
    assertEquals(Collections.singleton(getMethodInfo("D2_m1")), mis);
  }

  public void testExactOverrideFull() {
    TypeInfo bti = getTypeInfo("D2");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.ALL);
    Collection mis = spector.getMethods(bti);
    assertEquals(Collections.singleton(getMethodInfo("D2_m1")), mis);
  }

  private void testCovariantReturnTypeClass() {
    TypeInfo bti = getTypeInfo("A2");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.CLASS);
    Collection mis = spector.getMethods(bti);
    assertEquals(Collections.singleton(getMethodInfo("A2_m1")), mis);
  }

  private void testCovariantReturnTypeFull() {
    TypeInfo bti = getTypeInfo("A2");
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.ALL);
    Collection mis = spector.getMethods(bti);
    assertEquals(Collections.singleton(getMethodInfo("A2_m1")), mis);
  }

  private void testTypeVariableOverrideClass() {
    TypeInfo bti = getTypeInfo("B2");

    //
    Collection mis = new MethodIntrospector(HierarchyScope.CLASS).getMethods(bti);
    assertEquals(Collections.singleton(getMethodInfo("B2_m1")), mis);

    //
    mis = new MethodIntrospector(HierarchyScope.CLASS, true).getMethods(bti);
    assertEquals(Collections.singleton(getMethodInfo("B2_m1")), mis);

    //
    TypeInfo cti = getTypeInfo("C2");

    //
    mis = new MethodIntrospector(HierarchyScope.CLASS).getMethods(cti);
    assertEquals(Collections.singleton(getMethodInfo("C2_m1")), mis);

    //
    mis = new MethodIntrospector(HierarchyScope.CLASS, true).getMethods(cti);
    assertEquals(Collections.singleton(getMethodInfo("C2_m1")), mis);
  }

  private void testTypeVariableOverrideFull() {
    TypeInfo bti = getTypeInfo("B2");

    //
    Collection mis = new MethodIntrospector(HierarchyScope.ALL).getMethods(bti);
    Set<MethodInfo> methods = new HashSet<MethodInfo>();
    methods.add(getMethodInfo("B1_m1"));
    methods.add(getMethodInfo("B2_m1"));
    assertEquals(methods, mis);

    //
    mis = new MethodIntrospector(HierarchyScope.ALL, true).getMethods(bti);
    methods = new HashSet<MethodInfo>();
    methods.add(getMethodInfo("B2_m1"));
    assertEquals(methods, mis);

    //
    TypeInfo cti = getTypeInfo("C2");

    //
    mis = new MethodIntrospector(HierarchyScope.ALL).getMethods(cti);
    methods = new HashSet<MethodInfo>();
    methods.add(getMethodInfo("C1_m1"));
    methods.add(getMethodInfo("C2_m1"));
    assertEquals(methods, mis);

    //
    mis = new MethodIntrospector(HierarchyScope.ALL, true).getMethods(cti);
    methods = new HashSet<MethodInfo>();
    methods.add(getMethodInfo("C2_m1"));
    assertEquals(methods, mis);
  }

  private void testNPE() {
    MethodIntrospector spector = new MethodIntrospector(HierarchyScope.CLASS);
    try {
      spector.getMethods(null);
      fail();
    }
    catch (NullPointerException ignore) {
    }
    try {
      spector.getGetterMap(null);
      fail();
    }
    catch (NullPointerException ignore) {
    }
    try {
      spector.getSetters(null);
      fail();
    }
    catch (NullPointerException ignore) {
    }
    try {
      spector.getGetters(null);
      fail();
    }
    catch (NullPointerException ignore) {
    }
    try {
      spector.getSetterMap(null);
      fail();
    }
    catch (NullPointerException ignore) {
    }
    try {
      new MethodIntrospector(null);
      fail();
    }
    catch (NullPointerException ignore) {
    }
    try {
      new MethodIntrospector(null, true);
      fail();
    }
    catch (NullPointerException ignore) {
    }
  }
}