/**
 * Copyright 2005-2011 Noelios Technologies.
 * 
 * The contents of this file are subject to the terms of one of the following
 * open source licenses: LGPL 3.0 or LGPL 2.1 or CDDL 1.0 or EPL 1.0 (the
 * "Licenses"). You can select the license that you prefer but you may not use
 * this file except in compliance with one of these Licenses.
 * 
 * You can obtain a copy of the LGPL 3.0 license at
 * http://www.opensource.org/licenses/lgpl-3.0.html
 * 
 * You can obtain a copy of the LGPL 2.1 license at
 * http://www.opensource.org/licenses/lgpl-2.1.php
 * 
 * You can obtain a copy of the CDDL 1.0 license at
 * http://www.opensource.org/licenses/cddl1.php
 * 
 * You can obtain a copy of the EPL 1.0 license at
 * http://www.opensource.org/licenses/eclipse-1.0.php
 * 
 * See the Licenses for the specific language governing permissions and
 * limitations under the Licenses.
 * 
 * Alternatively, you can obtain a royalty free commercial license with less
 * limitations, transferable or non-transferable, directly at
 * http://www.noelios.com/products/restlet-engine
 * 
 * Restlet is a registered trademark of Noelios Technologies.
 */

package org.restlet.test;

import org.restlet.Client;
import org.restlet.Component;
import org.restlet.Request;
import org.restlet.Response;
import org.restlet.Restlet;
import org.restlet.data.Form;
import org.restlet.data.MediaType;
import org.restlet.data.Method;
import org.restlet.data.Parameter;
import org.restlet.data.Protocol;
import org.restlet.data.Status;
import org.restlet.representation.StringRepresentation;

public class HeaderTestCase extends RestletTestCase {

    /**
     * Restlet that returns as a new Representation the list of values of
     * "testHeader" header.
     * 
     */
    public static class TestHeaderRestlet extends Restlet {
        @Override
        public void handle(Request request, Response response) {
            final StringBuilder stb = new StringBuilder();
            final Form headers = getHttpHeaders(request);
            for (final Parameter header : headers) {
                if (header.getName().equals(TEST_HEADER)) {
                    stb.append(header.getValue());
                    stb.append('\n');
                }
            }
            response.setEntity(new StringRepresentation(stb,
                    MediaType.TEXT_PLAIN));
        }
    }

    private static final String HTTP_HEADERS = "org.restlet.http.headers";

    /**
     * Name of a test header field
     */
    private static final String TEST_HEADER = "testHeader";

    /**
     * Returns the list of http headers of a request as a Form.
     * 
     * @param request
     *            The request.
     * @return The list of headers as a Form object.
     */
    private static Form getHttpHeaders(Request request) {
        Form headers = (Form) request.getAttributes().get(HTTP_HEADERS);
        if (headers == null) {
            headers = new Form();
            request.getAttributes().put(HTTP_HEADERS, headers);
        }
        return headers;
    }

    private Component component;

    private Client client;

    /**
     * Handle a new request built according to the parameters and return the
     * response object.
     * 
     * @param parameters
     *            The list of parameters used to build the request.
     * @return The response of the request.
     * @throws Exception
     */
    private Response getWithParams(Parameter... parameters) throws Exception {
        Request request = new Request(Method.GET, "http://localhost:"
                + TEST_PORT);
        Form headers = getHttpHeaders(request);

        for (Parameter p : parameters) {
            headers.add(p);
        }

        Response result = client.handle(request);
        return result;
    }

    @Override
    public void setUp() throws Exception {
        super.setUp();
        this.client = new Client(Protocol.HTTP);

        if (this.component == null) {
            this.component = new Component();
            this.component.getServers().add(Protocol.HTTP, TEST_PORT);
            this.component.getDefaultHost().attachDefault(
                    new TestHeaderRestlet());
        }

        if (!this.component.isStarted()) {
            this.component.start();
        }
    }

    @Override
    public void tearDown() throws Exception {
        this.client.stop();
        this.component.stop();
        this.component = null;
        super.tearDown();
    }

    /** test with no test header */
    public void test0() throws Exception {
        Response response = getWithParams();
        assertEquals(Status.SUCCESS_OK, response.getStatus());
        assertEquals(null, response.getEntity().getText());
    }

    /** test with one test header */
    public void test1() throws Exception {
        Response response = getWithParams(new Parameter(TEST_HEADER, "a"));
        assertEquals(Status.SUCCESS_OK, response.getStatus());
        assertEquals("a\n", response.getEntity().getText());
    }

    /** test with two test headers */
    public void test2() throws Exception {
        Response response = getWithParams(new Parameter(TEST_HEADER, "a"),
                new Parameter(TEST_HEADER, "b"));
        assertEquals(Status.SUCCESS_OK, response.getStatus());
        assertEquals("a\nb\n", response.getEntity().getText());
    }
}
