/**
 * Copyright 2005-2011 Noelios Technologies.
 * 
 * The contents of this file are subject to the terms of one of the following
 * open source licenses: LGPL 3.0 or LGPL 2.1 or CDDL 1.0 or EPL 1.0 (the
 * "Licenses"). You can select the license that you prefer but you may not use
 * this file except in compliance with one of these Licenses.
 * 
 * You can obtain a copy of the LGPL 3.0 license at
 * http://www.opensource.org/licenses/lgpl-3.0.html
 * 
 * You can obtain a copy of the LGPL 2.1 license at
 * http://www.opensource.org/licenses/lgpl-2.1.php
 * 
 * You can obtain a copy of the CDDL 1.0 license at
 * http://www.opensource.org/licenses/cddl1.php
 * 
 * You can obtain a copy of the EPL 1.0 license at
 * http://www.opensource.org/licenses/eclipse-1.0.php
 * 
 * See the Licenses for the specific language governing permissions and
 * limitations under the Licenses.
 * 
 * Alternatively, you can obtain a royalty free commercial license with less
 * limitations, transferable or non-transferable, directly at
 * http://www.noelios.com/products/restlet-engine
 * 
 * Restlet is a registered trademark of Noelios Technologies.
 */

package org.restlet.test.ext.sip;

import junit.framework.TestCase;

import org.junit.Test;
import org.restlet.data.Parameter;
import org.restlet.data.Protocol;
import org.restlet.ext.sip.SipRecipientInfo;
import org.restlet.ext.sip.internal.SipRecipientInfoReader;

/**
 * Test case for the Via header.
 * 
 * @author Thierry Boileau
 * 
 */
public class SipRecipientInfoTestCase extends TestCase {

    @Test
    public void testParsing() throws Exception {
        String str = "SIP/2.0/UDP 192.0.2.1:5060 ;received=192.0.2.207;branch=z9hG4bK77asjd";
        SipRecipientInfoReader r = new SipRecipientInfoReader(str);
        SipRecipientInfo s = r.readValue();

        assertEquals(Protocol.SIP, s.getProtocol());
        assertEquals("UDP", s.getTransport());
        assertEquals("192.0.2.1:5060", s.getName());

        assertEquals(2, s.getParameters().size());
        Parameter parameter = s.getParameters().get(0);
        assertEquals("received", parameter.getName());
        assertEquals("192.0.2.207", parameter.getValue());
        parameter = s.getParameters().get(1);
        assertEquals("branch", parameter.getName());
        assertEquals("z9hG4bK77asjd", parameter.getValue());
        assertNull(s.getComment());

        str = "SIP/2.0/UDP 192.0.2.1:5060 ;received=192.0.2.207;branch=z9hG4bK77asjd (this is a comment)";
        r = new SipRecipientInfoReader(str);
        s = r.readValue();

        assertEquals(Protocol.SIP, s.getProtocol());
        assertEquals("UDP", s.getTransport());
        assertEquals("192.0.2.1:5060", s.getName());

        assertEquals(2, s.getParameters().size());
        parameter = s.getParameters().get(0);
        assertEquals("received", parameter.getName());
        assertEquals("192.0.2.207", parameter.getValue());
        parameter = s.getParameters().get(1);
        assertEquals("branch", parameter.getName());
        assertEquals("z9hG4bK77asjd", parameter.getValue());
        assertEquals("this is a comment", s.getComment());

        str = "SIP/2.0/TCP 127.0.0.1:5061;branch=z9hG4bK-6503-1-0";
        r = new SipRecipientInfoReader(str);
        s = r.readValue();

        assertEquals(Protocol.SIP, s.getProtocol());
        assertEquals("TCP", s.getTransport());
        assertEquals("127.0.0.1:5061", s.getName());

        assertEquals(1, s.getParameters().size());
        parameter = s.getParameters().get(0);
        assertEquals("branch", parameter.getName());
        assertEquals("z9hG4bK-6503-1-0", parameter.getValue());

        str = "SIP/2.0/TCP [fe80::223:dfff:fe7f:7b1a%en0]:5061;branch=z9hG4bK-409-1-0";
        r = new SipRecipientInfoReader(str);
        s = r.readValue();

        assertEquals(Protocol.SIP, s.getProtocol());
        assertEquals("TCP", s.getTransport());
        assertEquals("[fe80::223:dfff:fe7f:7b1a%en0]:5061", s.getName());
        assertEquals(1, s.getParameters().size());
        parameter = s.getParameters().get(0);
        assertEquals("branch", parameter.getName());
        assertEquals("z9hG4bK-409-1-0", parameter.getValue());
        assertNull(s.getComment());
    }

}
