package org.robolectric.shadows;

import android.view.View;
import android.widget.Adapter;
import android.widget.AdapterView;
import android.widget.FrameLayout;

import org.robolectric.annotation.Implementation;
import org.robolectric.annotation.Implements;
import org.robolectric.annotation.RealObject;
import org.robolectric.util.ReflectionHelpers.ClassParameter;

import static org.robolectric.internal.Shadow.directlyOn;
import static org.robolectric.Shadows.shadowOf;

/**
 * Shadow for {@link android.widget.AdapterView}.
 */
@SuppressWarnings({"UnusedDeclaration"})
@Implements(AdapterView.class)
public class ShadowAdapterView<T extends Adapter> extends ShadowViewGroup {
  private static int ignoreRowsAtEndOfList = 0;
  private static boolean automaticallyUpdateRowViews = true;

  @RealObject
  private AdapterView<T> realAdapterView;

  private AdapterView.OnItemSelectedListener itemSelectedListener;

  @Implementation
  public void setOnItemSelectedListener(AdapterView.OnItemSelectedListener itemSelectedListener) {
    this.itemSelectedListener = itemSelectedListener;
    directlyOn(realAdapterView, AdapterView.class, "setOnItemSelectedListener", ClassParameter.from(AdapterView.OnItemSelectedListener.class, itemSelectedListener));
  }

  public AdapterView.OnItemSelectedListener getItemSelectedListener() {
    return itemSelectedListener;
  }

  /**
   * Check if our adapter's items have changed without {@code onChanged()} or {@code onInvalidated()} having been called.
   *
   * @deprecated No longer supported.
   * @return true if the object is valid, false if not
   * @throws RuntimeException if the items have been changed without notification
   */
  public boolean checkValidity() {
    throw new UnsupportedOperationException();
  }

  /**
   * Use this static method to turn off the feature of this class which calls getView() on all of the
   * adapter's rows in setAdapter() and after notifyDataSetChanged() or notifyDataSetInvalidated() is
   * called on the adapter. This feature is turned on by default. This sets a static on the class, so
   * set it back to true at the end of your test to avoid test pollution.
   *
   * @param shouldUpdate false to turn off the feature, true to turn it back on
   * @deprecated Not supported as of Robolectric 2.0-alpha-3.
   */
  public static void automaticallyUpdateRowViews(boolean shouldUpdate) {
    automaticallyUpdateRowViews = shouldUpdate;
  }

  public boolean performItemClick(int position) {
    return realAdapterView.performItemClick(realAdapterView.getChildAt(position),
        position, realAdapterView.getItemIdAtPosition(position));
  }

  public int findIndexOfItemContainingText(String targetText) {
    for (int i = 0; i < realAdapterView.getCount(); i++) {
      View childView = realAdapterView.getAdapter().getView(i, null, new FrameLayout(realAdapterView.getContext()));
      String innerText = shadowOf(childView).innerText();
      if (innerText.contains(targetText)) {
        return i;
      }
    }
    return -1;
  }

  public View findItemContainingText(String targetText) {
    int itemIndex = findIndexOfItemContainingText(targetText);
    if (itemIndex == -1) {
      return null;
    }
    return realAdapterView.getChildAt(itemIndex);
  }

  public void clickFirstItemContainingText(String targetText) {
    int itemIndex = findIndexOfItemContainingText(targetText);
    if (itemIndex == -1) {
      throw new IllegalArgumentException("No item found containing text \"" + targetText + "\"");
    }
    performItemClick(itemIndex);
  }

  public void populateItems() {
    realView.measure(0, 0);
    realView.layout(0, 0, 100, 10000);
  }

  public void selectItemWithText(String s) {
    int itemIndex = findIndexOfItemContainingText(s);
    realAdapterView.setSelection(itemIndex);
  }
}
