package org.robolectric.shadows;

import android.accounts.AccountManager;
import android.content.Context;
import android.os.Build;
import android.os.FileUtils;
import android.os.Handler;
import android.view.Display;
import android.view.accessibility.AccessibilityManager;
import org.robolectric.RuntimeEnvironment;
import org.robolectric.annotation.Implementation;
import org.robolectric.annotation.Implements;
import org.robolectric.annotation.RealObject;
import org.robolectric.util.ReflectionHelpers;
import org.robolectric.util.ReflectionHelpers.ClassParameter;

import java.io.File;

import java.util.HashMap;
import java.util.Map;

import static org.robolectric.internal.Shadow.directlyOn;
import static org.robolectric.internal.Shadow.newInstanceOf;
import static org.robolectric.util.ReflectionHelpers.ClassParameter.from;

/**
 * Shadow for {@code android.content.ContextImpl}.
 */
@Implements(className = ShadowContextImpl.CLASS_NAME)
public class ShadowContextImpl extends ShadowContext {
  public static final String CLASS_NAME = "android.app.ContextImpl";
  private static final Map<String, String> SYSTEM_SERVICE_MAP = new HashMap<String, String>();

  @RealObject
  private Context realObject;

  static {
    // note that these are different!
    // They specify concrete classes within Robolectric for interfaces or abstract classes defined by Android
    SYSTEM_SERVICE_MAP.put(Context.WINDOW_SERVICE, "android.view.WindowManagerImpl");
    SYSTEM_SERVICE_MAP.put(Context.CLIPBOARD_SERVICE, "android.content.ClipboardManager");
    SYSTEM_SERVICE_MAP.put(Context.SENSOR_SERVICE, "org.robolectric.fakes.RoboSensorManager");
    SYSTEM_SERVICE_MAP.put(Context.VIBRATOR_SERVICE, "org.robolectric.fakes.RoboVibrator");

    // the rest are as mapped in docs...
    SYSTEM_SERVICE_MAP.put(Context.LAYOUT_INFLATER_SERVICE, "android.view.LayoutInflater");
    SYSTEM_SERVICE_MAP.put(Context.ACTIVITY_SERVICE, "android.app.ActivityManager");
    SYSTEM_SERVICE_MAP.put(Context.POWER_SERVICE, "android.os.PowerManager");
    SYSTEM_SERVICE_MAP.put(Context.ALARM_SERVICE, "android.app.AlarmManager");
    SYSTEM_SERVICE_MAP.put(Context.NOTIFICATION_SERVICE, "android.app.NotificationManager");
    SYSTEM_SERVICE_MAP.put(Context.KEYGUARD_SERVICE, "android.app.KeyguardManager");
    SYSTEM_SERVICE_MAP.put(Context.LOCATION_SERVICE, "android.location.LocationManager");
    SYSTEM_SERVICE_MAP.put(Context.SEARCH_SERVICE, "android.app.SearchManager");
    SYSTEM_SERVICE_MAP.put(Context.STORAGE_SERVICE, "android.os.storage.StorageManager");
    SYSTEM_SERVICE_MAP.put(Context.CONNECTIVITY_SERVICE, "android.net.ConnectivityManager");
    SYSTEM_SERVICE_MAP.put(Context.WIFI_SERVICE, "android.net.wifi.WifiManager");
    SYSTEM_SERVICE_MAP.put(Context.AUDIO_SERVICE, "android.media.AudioManager");
    SYSTEM_SERVICE_MAP.put(Context.TELEPHONY_SERVICE, "android.telephony.TelephonyManager");
    SYSTEM_SERVICE_MAP.put(Context.INPUT_METHOD_SERVICE, "android.view.inputmethod.InputMethodManager");
    SYSTEM_SERVICE_MAP.put(Context.UI_MODE_SERVICE, "android.app.UiModeManager");
    SYSTEM_SERVICE_MAP.put(Context.DOWNLOAD_SERVICE, "android.app.DownloadManager");
    SYSTEM_SERVICE_MAP.put(Context.TEXT_SERVICES_MANAGER_SERVICE, "android.view.textservice.TextServicesManager");
    SYSTEM_SERVICE_MAP.put(Context.DEVICE_POLICY_SERVICE, "android.app.admin.DevicePolicyManager");
    SYSTEM_SERVICE_MAP.put(Context.DROPBOX_SERVICE, "android.os.DropBoxManager");
    SYSTEM_SERVICE_MAP.put(Context.MEDIA_ROUTER_SERVICE, "android.media.MediaRouter");
    SYSTEM_SERVICE_MAP.put(Context.DISPLAY_SERVICE, "android.hardware.display.DisplayManager");
    SYSTEM_SERVICE_MAP.put(Context.USER_SERVICE, "android.os.UserManager");
    SYSTEM_SERVICE_MAP.put(Context.ACCESSIBILITY_SERVICE, "android.view.accessibility.AccessibilityManager");
    SYSTEM_SERVICE_MAP.put(Context.ACCOUNT_SERVICE, "android.accounts.AccountManager");
    SYSTEM_SERVICE_MAP.put(Context.NFC_SERVICE, "android.nfc.NfcManager");
    SYSTEM_SERVICE_MAP.put(Context.PRINT_SERVICE, "android.print.PrintManager");
  }

  @Implementation
  public File validateFilePath(String name, boolean createDirectory) {
    File dir;
    File f = new File(name);

    if (f.isAbsolute()) {
      dir = f.getParentFile();
    } else {
      dir = directlyOn(realObject, "android.app.ContextImpl", "getDatabasesDir");
      f = directlyOn(realObject, "android.app.ContextImpl", "makeFilename", from(File.class, dir), from(String.class, name));
    }

    if (createDirectory && !dir.isDirectory() && dir.mkdir()) {
      FileUtils.setPermissions(dir.getPath(),
          FileUtils.S_IRWXU|FileUtils.S_IRWXG|FileUtils.S_IXOTH,
          -1, -1);
    }

    return f;
  }

  private Map<String, Object> systemServices = new HashMap<String, Object>();

  @Implements(className = ShadowServiceFetcher.CLASS_NAME, looseSignatures = true)
  public static class ShadowServiceFetcher {
    public static final String CLASS_NAME = "android.app.ContextImpl$ServiceFetcher";

    @Implementation
    public Object createService(Object ctx) {
      return null;
    }
  }

  @Implementation
  public Object getSystemService(String name) {
    if (name.equals(Context.LAYOUT_INFLATER_SERVICE)) {
      return new RoboLayoutInflater(RuntimeEnvironment.application);
    }

    Object service = systemServices.get(name);
    if (service == null) {
      String serviceClassName = SYSTEM_SERVICE_MAP.get(name);
      if (serviceClassName == null) {
        System.err.println("WARNING: unknown service " + name);
        return null;
      }

      try {
        Class<?> clazz = Class.forName(serviceClassName);
        if (serviceClassName.equals("android.app.SearchManager")
            || serviceClassName.equals("android.app.ActivityManager")
            || serviceClassName.equals("android.app.admin.DevicePolicyManager")) {

          service = ReflectionHelpers.callConstructor(clazz,
              ClassParameter.from(Context.class, RuntimeEnvironment.application),
              ClassParameter.from(Handler.class, null));

        } else if (serviceClassName.equals("android.os.storage.StorageManager")) {
          service = ReflectionHelpers.callConstructor(clazz);

        } else if (serviceClassName.equals("android.nfc.NfcManager")) {
          service = ReflectionHelpers.callConstructor(clazz,
              ClassParameter.from(Context.class, RuntimeEnvironment.application));

        } else if (serviceClassName.equals("android.hardware.display.DisplayManager")) {
          service = ReflectionHelpers.callConstructor(clazz, ClassParameter.from(Context.class, RuntimeEnvironment.application));
        } else if (serviceClassName.equals("android.view.accessibility.AccessibilityManager")) {
          service = AccessibilityManager.getInstance(realObject);
        } else if (serviceClassName.equals("android.view.WindowManagerImpl")) {
          Display display = newInstanceOf(Display.class);
          service = ReflectionHelpers.callConstructor(Class.forName("android.view.WindowManagerImpl"), ClassParameter.from(Display.class, display));
        } else if (serviceClassName.equals("android.accounts.AccountManager")) {
          service = AccountManager.get(null);

        } else if (serviceClassName.equals("android.print.PrintManager")) {
          service = ReflectionHelpers.callConstructor(Class.forName("android.print.PrintManager"),
            ClassParameter.from(Context.class, RuntimeEnvironment.application),
            ClassParameter.from(android.print.IPrintManager.class, null),
            ClassParameter.from(int.class, -1),
            ClassParameter.from(int.class, -1));
        } else {
          service = newInstanceOf(clazz);
        }
      } catch (ClassNotFoundException e) {
        throw new RuntimeException(e);
      }

      systemServices.put(name, service);
    }
    return service;
  }

  public void setSystemService(String key, Object service) {
    systemServices.put(key, service);
  }
}
