package org.seasar.doma.jdbc.dialect;

import java.sql.SQLException;
import java.util.Collections;
import org.seasar.doma.DomaNullPointerException;
import org.seasar.doma.expr.ExpressionFunctions;
import org.seasar.doma.internal.jdbc.dialect.H212126ForUpdateTransformer;
import org.seasar.doma.internal.jdbc.dialect.H212126PagingTransformer;
import org.seasar.doma.jdbc.JdbcMappingVisitor;
import org.seasar.doma.jdbc.PreparedSql;
import org.seasar.doma.jdbc.SelectForUpdateType;
import org.seasar.doma.jdbc.SqlKind;
import org.seasar.doma.jdbc.SqlLogFormattingVisitor;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.SqlNode;

/** A dialect for H2 version 1.2.126 and below. */
public class H212126Dialect extends StandardDialect {

  /** the error code that represents unique violation */
  protected static final int UNIQUE_CONSTRAINT_VIOLATION_ERROR_CODE = 23001;

  public H212126Dialect() {
    this(
        new H212126JdbcMappingVisitor(),
        new H212126SqlLogFormattingVisitor(),
        new H212126ExpressionFunctions());
  }

  public H212126Dialect(JdbcMappingVisitor jdbcMappingVisitor) {
    this(
        jdbcMappingVisitor, new H212126SqlLogFormattingVisitor(), new H212126ExpressionFunctions());
  }

  public H212126Dialect(SqlLogFormattingVisitor sqlLogFormattingVisitor) {
    this(
        new H212126JdbcMappingVisitor(), sqlLogFormattingVisitor, new H212126ExpressionFunctions());
  }

  public H212126Dialect(ExpressionFunctions expressionFunctions) {
    this(
        new H212126JdbcMappingVisitor(), new H212126SqlLogFormattingVisitor(), expressionFunctions);
  }

  public H212126Dialect(
      JdbcMappingVisitor jdbcMappingVisitor, SqlLogFormattingVisitor sqlLogFormattingVisitor) {
    this(jdbcMappingVisitor, sqlLogFormattingVisitor, new H212126ExpressionFunctions());
  }

  public H212126Dialect(
      JdbcMappingVisitor jdbcMappingVisitor,
      SqlLogFormattingVisitor sqlLogFormattingVisitor,
      ExpressionFunctions expressionFunctions) {
    super(jdbcMappingVisitor, sqlLogFormattingVisitor, expressionFunctions);
  }

  @Override
  public String getName() {
    return "h2";
  }

  @Override
  public boolean includesIdentityColumn() {
    return true;
  }

  @Override
  public PreparedSql getIdentitySelectSql(
      String catalogName,
      String schemaName,
      String tableName,
      String columnName,
      boolean isQuoteRequired,
      boolean isIdColumnQuoteRequired) {
    if (tableName == null) {
      throw new DomaNullPointerException("tableName");
    }
    if (columnName == null) {
      throw new DomaNullPointerException("columnName");
    }
    String rawSql = "call identity()";
    return new PreparedSql(
        SqlKind.SELECT, rawSql, rawSql, null, Collections.emptyList(), SqlLogType.FORMATTED);
  }

  @Override
  public PreparedSql getSequenceNextValSql(String qualifiedSequenceName, long allocationSize) {
    if (qualifiedSequenceName == null) {
      throw new DomaNullPointerException("qualifiedSequenceName");
    }
    String rawSql = "call next value for " + qualifiedSequenceName;
    return new PreparedSql(
        SqlKind.SELECT, rawSql, rawSql, null, Collections.emptyList(), SqlLogType.FORMATTED);
  }

  @Override
  public boolean isUniqueConstraintViolated(SQLException sqlException) {
    if (sqlException == null) {
      throw new DomaNullPointerException("sqlException");
    }
    int code = getErrorCode(sqlException);
    return UNIQUE_CONSTRAINT_VIOLATION_ERROR_CODE == code;
  }

  @Override
  protected SqlNode toPagingSqlNode(SqlNode sqlNode, long offset, long limit) {
    H212126PagingTransformer transformer = new H212126PagingTransformer(offset, limit);
    return transformer.transform(sqlNode);
  }

  @Override
  protected SqlNode toForUpdateSqlNode(
      SqlNode sqlNode, SelectForUpdateType forUpdateType, int waitSeconds, String... aliases) {
    H212126ForUpdateTransformer transformer =
        new H212126ForUpdateTransformer(forUpdateType, waitSeconds, aliases);
    return transformer.transform(sqlNode);
  }

  @Override
  public boolean supportsIdentity() {
    return true;
  }

  @Override
  public boolean supportsSequence() {
    return true;
  }

  @Override
  public boolean supportsAutoGeneratedKeys() {
    return true;
  }

  @Override
  public boolean supportsSelectForUpdate(SelectForUpdateType type, boolean withTargets) {
    return type == SelectForUpdateType.NORMAL && !withTargets;
  }

  public static class H212126JdbcMappingVisitor extends StandardJdbcMappingVisitor {}

  public static class H212126SqlLogFormattingVisitor extends StandardSqlLogFormattingVisitor {}

  public static class H212126ExpressionFunctions extends StandardExpressionFunctions {

    public H212126ExpressionFunctions() {
      super();
    }

    public H212126ExpressionFunctions(char[] wildcards) {
      super(wildcards);
    }

    protected H212126ExpressionFunctions(char escapeChar, char[] wildcards) {
      super(escapeChar, wildcards);
    }
  }
}
