package org.seasar.doma.jdbc.entity;

import java.sql.Statement;
import java.util.function.Supplier;
import org.seasar.doma.DomaNullPointerException;
import org.seasar.doma.GenerationType;
import org.seasar.doma.internal.jdbc.scalar.Scalar;
import org.seasar.doma.jdbc.JdbcException;
import org.seasar.doma.jdbc.dialect.Dialect;
import org.seasar.doma.jdbc.id.IdGenerationConfig;
import org.seasar.doma.jdbc.id.IdGenerator;
import org.seasar.doma.message.Message;
import org.seasar.doma.wrapper.NumberWrapper;
import org.seasar.doma.wrapper.NumberWrapperVisitor;

/**
 * A description for an identity property whose value is generated.
 *
 * @param <ENTITY> the entity type
 * @param <BASIC> the property basic type
 * @param <CONTAINER> the property container type
 */
public class GeneratedIdPropertyType<ENTITY, BASIC extends Number, CONTAINER>
    extends DefaultPropertyType<ENTITY, BASIC, CONTAINER> {

  protected final IdGenerator idGenerator;

  public GeneratedIdPropertyType(
      Class<ENTITY> entityClass,
      Supplier<Scalar<BASIC, CONTAINER>> scalarSupplier,
      String name,
      String columnName,
      NamingType namingType,
      boolean quoteRequired,
      IdGenerator idGenerator) {
    super(entityClass, scalarSupplier, name, columnName, namingType, true, true, quoteRequired);
    if (idGenerator == null) {
      throw new DomaNullPointerException("idGenerator");
    }
    this.idGenerator = idGenerator;
  }

  @Override
  public boolean isId() {
    return true;
  }

  public void validateGenerationStrategy(IdGenerationConfig config) {
    Dialect dialect = config.getDialect();
    GenerationType generationType = idGenerator.getGenerationType();
    if (!isGenerationTypeSupported(generationType, dialect)) {
      EntityType<?> entityType = config.getEntityType();
      throw new JdbcException(
          Message.DOMA2021, entityType.getName(), name, generationType.name(), dialect.getName());
    }
  }

  protected boolean isGenerationTypeSupported(GenerationType generationType, Dialect dialect) {
    switch (generationType) {
      case IDENTITY:
        return dialect.supportsIdentity();
      case SEQUENCE:
        return dialect.supportsSequence();
      default:
        return true;
    }
  }

  public boolean isIncluded(IdGenerationConfig config) {
    return idGenerator.includesIdentityColumn(config);
  }

  public boolean isBatchSupported(IdGenerationConfig config) {
    return idGenerator.supportsBatch(config);
  }

  public boolean isAutoGeneratedKeysSupported(IdGenerationConfig config) {
    return idGenerator.supportsAutoGeneratedKeys(config);
  }

  public ENTITY preInsert(EntityType<ENTITY> entityType, ENTITY entity, IdGenerationConfig config) {
    return setIfNecessary(entityType, entity, () -> idGenerator.generatePreInsert(config));
  }

  public ENTITY postInsert(
      EntityType<ENTITY> entityType,
      ENTITY entity,
      IdGenerationConfig config,
      Statement statement) {
    return setIfNecessary(
        entityType, entity, () -> idGenerator.generatePostInsert(config, statement));
  }

  protected ENTITY setIfNecessary(
      EntityType<ENTITY> entityType, ENTITY entity, Supplier<Long> supplier) {
    return modifyIfNecessary(entityType, entity, new ValueSetter(), supplier);
  }

  protected static class ValueSetter
      implements NumberWrapperVisitor<Boolean, Supplier<Long>, Void, RuntimeException> {

    @Override
    public <V extends Number> Boolean visitNumberWrapper(
        NumberWrapper<V> wrapper, Supplier<Long> valueSupplier, Void q) throws RuntimeException {
      Number currentValue = wrapper.get();
      if (currentValue == null || currentValue.intValue() < 0) {
        Long value = valueSupplier.get();
        if (value != null) {
          wrapper.set(value);
          return true;
        }
      }
      return false;
    }
  }
}
