package org.seasar.doma.jdbc.id;

import java.sql.Statement;
import org.seasar.doma.GenerationType;
import org.seasar.doma.jdbc.JdbcException;

/**
 * An identity generator.
 *
 * <p>Either {@link #generatePreInsert(IdGenerationConfig)} or {@link
 * #generatePostInsert(IdGenerationConfig, Statement)} must return non-null value.
 *
 * <p>The implementation class must have a public no-args constructor.
 *
 * <p>The implementation class must be thread safe.
 */
public interface IdGenerator {

  /**
   * Whether this generator supports the batch insert.
   *
   * @param config the configuration
   * @return {@code true} if this generator supports
   */
  boolean supportsBatch(IdGenerationConfig config);

  /**
   * Whether this generator supports {@link Statement#getGeneratedKeys()}.
   *
   * @param config the configuration
   * @return {@code true} if this generator supports
   */
  boolean supportsAutoGeneratedKeys(IdGenerationConfig config);

  /**
   * Whether this generator includes the identity column into SQL INSERT statements.
   *
   * @param config the configuration
   * @return {@code true} if supported
   */
  boolean includesIdentityColumn(IdGenerationConfig config);

  /**
   * Generates the identity value before an insert.
   *
   * @param config the configuration
   * @return the generated value or {@code null} if not supported
   * @throws JdbcException if the generation is failed
   */
  Long generatePreInsert(IdGenerationConfig config);

  /**
   * Generates the identity value after an insert.
   *
   * @param config the configuration
   * @param statement the SQL INSERT statement
   * @return the generated value or {@code null} if not supported
   * @throws JdbcException if the generation is failed
   */
  Long generatePostInsert(IdGenerationConfig config, Statement statement);

  /**
   * Returns the generation type.
   *
   * @return the generation type
   */
  GenerationType getGenerationType();
}
