/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.command;

import static org.seasar.doma.internal.util.AssertionUtil.assertNotNull;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import org.seasar.doma.internal.jdbc.command.PreparedSqlParameterBinder;
import org.seasar.doma.internal.jdbc.util.JdbcUtil;
import org.seasar.doma.jdbc.Config;
import org.seasar.doma.jdbc.JdbcLogger;
import org.seasar.doma.jdbc.OptimisticLockException;
import org.seasar.doma.jdbc.PreparedSql;
import org.seasar.doma.jdbc.SqlExecutionException;
import org.seasar.doma.jdbc.UniqueConstraintException;
import org.seasar.doma.jdbc.dialect.Dialect;
import org.seasar.doma.jdbc.query.ModifyQuery;

/**
 * An abstract base class for commands that modify database data (INSERT, UPDATE, DELETE).
 *
 * <p>This class provides common functionality for executing SQL statements that modify data,
 * including connection management, statement preparation, parameter binding, and execution. It also
 * handles error conditions such as SQL exceptions, unique constraint violations, and optimistic
 * lock failures.
 *
 * <p>Subclasses must implement the {@link #executeInternal(PreparedStatement)} method to define the
 * specific execution behavior.
 *
 * @param <QUERY> the type of query this command executes
 */
public abstract class ModifyCommand<QUERY extends ModifyQuery> implements Command<Integer> {

  /**
   * The query object that contains the SQL to be executed and related configuration. This member is
   * initialized in the constructor and remains unchanged throughout the command's lifecycle.
   */
  protected final QUERY query;

  /**
   * The prepared SQL statement that will be executed. This is extracted from the query object and
   * contains both the SQL string and its parameters.
   */
  protected final PreparedSql sql;

  protected ModifyCommand(QUERY query) {
    assertNotNull(query);
    this.query = query;
    this.sql = query.getSql();
  }

  @Override
  public QUERY getQuery() {
    return query;
  }

  /**
   * Executes the SQL command and returns the number of rows affected.
   *
   * <p>This method handles the entire execution process including:
   *
   * <ul>
   *   <li>Checking if the query is executable
   *   <li>Obtaining a database connection
   *   <li>Preparing the statement
   *   <li>Setting up statement options
   *   <li>Binding parameters
   *   <li>Executing the statement
   *   <li>Handling exceptions
   *   <li>Ensuring proper resource cleanup
   * </ul>
   *
   * @return the number of rows affected by the execution, or 0 if the query is not executable
   * @throws SqlExecutionException if a database access error occurs
   */
  @Override
  public Integer execute() {
    if (!query.isExecutable()) {
      JdbcLogger logger = query.getConfig().getJdbcLogger();
      logger.logSqlExecutionSkipping(
          query.getClassName(), query.getMethodName(), query.getSqlExecutionSkipCause());
      return 0;
    }
    Connection connection = JdbcUtil.getConnection(query.getConfig().getDataSource());
    try {
      PreparedStatement preparedStatement = prepareStatement(connection);
      try {
        log();
        setupOptions(preparedStatement);
        bindParameters(preparedStatement);
        return executeInternal(preparedStatement);
      } catch (SQLException e) {
        Dialect dialect = query.getConfig().getDialect();
        throw new SqlExecutionException(
            query.getConfig().getExceptionSqlLogType(), sql, e, dialect.getRootCause(e));
      } finally {
        JdbcUtil.close(preparedStatement, query.getConfig().getJdbcLogger());
      }
    } finally {
      JdbcUtil.close(connection, query.getConfig().getJdbcLogger());
    }
  }

  protected PreparedStatement prepareStatement(Connection connection) {
    if (query.isAutoGeneratedKeysSupported()) {
      Config config = query.getConfig();
      Dialect dialect = config.getDialect();
      switch (dialect.getAutoGeneratedKeysType()) {
        case FIRST_COLUMN:
          return JdbcUtil.prepareStatementForAutoGeneratedKeysOfFirstColumn(connection, sql);
        case DEFAULT:
          return JdbcUtil.prepareStatementForAutoGeneratedKeys(connection, sql);
      }
    }
    return JdbcUtil.prepareStatement(connection, sql);
  }

  /**
   * Executes the internal implementation of the SQL command.
   *
   * <p>This abstract method must be implemented by subclasses to define the specific behavior for
   * executing the SQL statement. It is called by the {@link #execute()} method after the connection
   * is established and the statement is prepared.
   *
   * @param preparedStatement the prepared statement to execute
   * @return the number of rows affected by the execution
   * @throws SQLException if a database access error occurs
   */
  protected abstract int executeInternal(PreparedStatement preparedStatement) throws SQLException;

  protected void log() {
    JdbcLogger logger = query.getConfig().getJdbcLogger();
    logger.logSql(query.getClassName(), query.getMethodName(), sql);
  }

  protected void setupOptions(PreparedStatement preparedStatement) throws SQLException {
    if (query.getQueryTimeout() > 0) {
      preparedStatement.setQueryTimeout(query.getQueryTimeout());
    }
  }

  protected void bindParameters(PreparedStatement preparedStatement) throws SQLException {
    PreparedSqlParameterBinder binder = new PreparedSqlParameterBinder(query);
    binder.bind(preparedStatement, sql.getParameters());
  }

  protected int executeUpdate(PreparedStatement preparedStatement) throws SQLException {
    try {
      int updatedRows = preparedStatement.executeUpdate();
      validateRows(updatedRows);
      return updatedRows;
    } catch (SQLException e) {
      Dialect dialect = query.getConfig().getDialect();
      if (dialect.isUniqueConstraintViolated(e)) {
        throw new UniqueConstraintException(query.getConfig().getExceptionSqlLogType(), sql, e);
      }
      throw e;
    }
  }

  protected void validateRows(int rows) {
    if (query.isOptimisticLockCheckRequired() && rows == 0) {
      throw new OptimisticLockException(query.getConfig().getExceptionSqlLogType(), sql);
    }
  }
}
