/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.dialect;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import org.seasar.doma.DomaNullPointerException;
import org.seasar.doma.expr.ExpressionFunctions;
import org.seasar.doma.jdbc.JdbcException;
import org.seasar.doma.jdbc.JdbcMappingVisitor;
import org.seasar.doma.jdbc.ScriptBlockContext;
import org.seasar.doma.jdbc.SelectForUpdateType;
import org.seasar.doma.jdbc.SelectOptions;
import org.seasar.doma.jdbc.Sql;
import org.seasar.doma.jdbc.SqlLogFormatter;
import org.seasar.doma.jdbc.SqlLogFormattingVisitor;
import org.seasar.doma.jdbc.SqlNode;
import org.seasar.doma.jdbc.criteria.query.CriteriaBuilder;
import org.seasar.doma.jdbc.id.AutoGeneratedKeysType;
import org.seasar.doma.jdbc.query.DeleteAssembler;
import org.seasar.doma.jdbc.query.DeleteAssemblerContext;
import org.seasar.doma.jdbc.query.InsertAssembler;
import org.seasar.doma.jdbc.query.InsertAssemblerContext;
import org.seasar.doma.jdbc.query.MultiInsertAssembler;
import org.seasar.doma.jdbc.query.MultiInsertAssemblerContext;
import org.seasar.doma.jdbc.query.UpdateAssembler;
import org.seasar.doma.jdbc.query.UpdateAssemblerContext;
import org.seasar.doma.jdbc.query.UpsertAssembler;
import org.seasar.doma.jdbc.query.UpsertAssemblerContext;
import org.seasar.doma.jdbc.type.JdbcType;
import org.seasar.doma.wrapper.Wrapper;

/**
 * A database dialect interface that abstracts differences between various RDBMS implementations.
 *
 * <p>This interface is a core component of Doma's database abstraction layer, allowing applications
 * to work with different database systems without changing application code. It provides methods to
 * handle database-specific behaviors such as SQL syntax, identifier quoting, data type mappings,
 * pagination, sequence generation, and other vendor-specific features.
 *
 * <p>Custom dialects can be created by implementing this interface or extending existing
 * implementations to support additional database systems or to customize behavior for specific
 * needs.
 *
 * <p>The implementation instance must be thread safe.
 *
 * @see org.seasar.doma.jdbc.Config#getDialect()
 */
public interface Dialect {

  /**
   * Returns the name of this dialect.
   *
   * <p>The dialect name typically identifies the database system that this dialect supports. This
   * name can be used for logging, debugging, or to make decisions based on the specific database
   * being used.
   *
   * <p>Each dialect implementation should return a unique and descriptive name.
   *
   * @return the name of this dialect
   */
  String getName();

  /**
   * Transforms the SQL node for a SELECT statement according to the specified options.
   *
   * <p>This method is responsible for applying database-specific transformations to SELECT
   * statements, such as adding pagination (LIMIT/OFFSET), FOR UPDATE clauses, or other
   * dialect-specific syntax.
   *
   * @param sqlNode the SQL node representing the SELECT statement to transform
   * @param options the options that specify how to transform the SQL, including pagination and
   *     locking settings
   * @return the transformed SQL node with dialect-specific modifications applied
   * @throws DomaNullPointerException if any argument is {@code null}
   * @throws JdbcException if unsupported options are specified
   */
  SqlNode transformSelectSqlNode(SqlNode sqlNode, SelectOptions options);

  /**
   * Transforms the SQL node to create a query that returns the total count of rows.
   *
   * <p>This method is typically used for pagination scenarios where the total number of results
   * needs to be known. It transforms a regular SELECT statement into a COUNT query that returns the
   * total number of rows that would be returned by the original query without pagination.
   *
   * <p>The implementation should handle removing ORDER BY clauses and other elements that are not
   * necessary for counting rows, while preserving the WHERE conditions and JOIN clauses.
   *
   * @param sqlNode the SQL node representing the original SELECT statement
   * @return the transformed SQL node that will return the total row count
   * @throws DomaNullPointerException if {@code sqlNode} is {@code null}
   */
  SqlNode transformSelectSqlNodeForGettingCount(SqlNode sqlNode);

  /**
   * Determines whether the given SQLException represents a unique constraint violation.
   *
   * <p>Different database systems use different error codes and states to indicate unique
   * constraint violations. This method abstracts these differences.
   *
   * @param sqlException the SQL exception to analyze
   * @return {@code true} if the exception represents a unique constraint violation, {@code false}
   *     otherwise
   * @throws DomaNullPointerException if {@code sqlException} is {@code null}
   */
  boolean isUniqueConstraintViolated(SQLException sqlException);

  /**
   * Determines whether this dialect includes IDENTITY columns in SQL INSERT statements.
   *
   * <p>Some database systems require that IDENTITY columns be excluded from INSERT statements,
   * while others allow or require them to be included.
   *
   * @return {@code true} if this dialect includes IDENTITY columns in INSERT statements, {@code
   *     false} otherwise
   */
  boolean includesIdentityColumn();

  /**
   * Determines if the identity column is included based on the provided identifier value.
   *
   * <p>The type of {@code idValue} must be one of the basic types. It must never be a domain class
   * or an Optional type that wraps a basic type.
   *
   * @param idValue an identity value; may be null
   * @return true if the identity column is included, false otherwise
   */
  default boolean includesIdentityColumn(Object idValue) {
    return includesIdentityColumn();
  }

  /**
   * Determines whether this dialect supports the IDENTITY column.
   *
   * <p>An IDENTITY column is a column that automatically generates a unique value for each row when
   * data is inserted. The exact implementation varies by database system.
   *
   * @return {@code true} if this dialect supports IDENTITY columns, {@code false} otherwise
   */
  boolean supportsIdentity();

  /**
   * Determines whether this dialect supports database sequences.
   *
   * <p>A sequence is a database object that generates a sequence of unique numbers. Sequences are
   * commonly used to generate primary key values.
   *
   * @return {@code true} if this dialect supports sequences, {@code false} otherwise
   */
  boolean supportsSequence();

  /**
   * Determines whether this dialect supports retrieving auto-generated keys via {@link
   * Statement#getGeneratedKeys()}.
   *
   * <p>This feature allows retrieving values that were automatically generated during an insert
   * operation, such as identity column values, without executing a separate query.
   *
   * @return {@code true} if this dialect supports retrieving auto-generated keys, {@code false}
   *     otherwise
   */
  boolean supportsAutoGeneratedKeys();

  /**
   * Determines whether this dialect supports reliable results from {@link
   * Statement#executeBatch()}.
   *
   * <p>Some database systems do not correctly report the number of affected rows for each statement
   * in a batch operation. This method indicates whether the dialect can rely on these results.
   *
   * @return {@code true} if this dialect supports reliable batch update results, {@code false}
   *     otherwise
   */
  boolean supportsBatchUpdateResults();

  /**
   * Determines whether this dialect supports retrieving generated values from batch executions.
   *
   * <p>Some database systems can return auto-generated values (like identity columns) even when
   * executing statements in batch mode, while others cannot.
   *
   * <p>The default implementation returns {@code false}.
   *
   * @return {@code true} if this dialect supports retrieving generated values from batch
   *     executions, {@code false} otherwise
   */
  default boolean supportsBatchExecutionReturningGeneratedValues() {
    return false;
  }

  /**
   * Determines whether this dialect supports pessimistic locking with the specified options.
   *
   * <p>Pessimistic locking (SELECT FOR UPDATE) prevents other transactions from modifying selected
   * rows. Different database systems support different variations of this feature.
   *
   * @param type the type of pessimistic locking (e.g., standard FOR UPDATE, NOWAIT, WAIT)
   * @param withTargets {@code true} if specific columns are targeted for locking, {@code false} for
   *     row-level locking
   * @return {@code true} if this dialect supports the specified locking options, {@code false}
   *     otherwise
   */
  boolean supportsSelectForUpdate(SelectForUpdateType type, boolean withTargets);

  /**
   * Determines whether this dialect supports result sets as OUT parameters in stored procedures.
   *
   * <p>Some database systems allow stored procedures to return result sets through OUT parameters,
   * while others use different mechanisms (like returning result sets directly).
   *
   * @return {@code true} if this dialect supports result sets as OUT parameters, {@code false}
   *     otherwise
   */
  boolean supportsResultSetReturningAsOutParameter();

  /**
   * Determines whether this dialect supports reserving identity values in advance.
   *
   * <p>Some database systems allow pre-allocating a range of identity values, which can improve
   * performance in certain scenarios.
   *
   * @return {@code true} if this dialect supports identity reservation, {@code false} otherwise
   * @deprecated This feature is deprecated and may be removed in a future version
   */
  @Deprecated
  boolean supportsIdentityReservation();

  /**
   * Determines whether this object supports alias reference in DELETE clause as follows:
   *
   * <pre>
   * DELETE t FROM employee t
   * </pre>
   *
   * @return {@code true}, if this object supports it
   */
  default boolean supportsAliasInDeleteClause() {
    return false;
  }

  /**
   * Determines whether the use of table aliases is supported in the context of a DELETE statement.
   *
   * <pre>
   * DELETE FROM employee t
   * </pre>
   *
   * @return true if table aliasing is supported in DELETE statements, false otherwise
   */
  default boolean supportsAliasInDeleteStatement() {
    return true;
  }

  /**
   * Determines whether this object supports alias reference in UPDATE clause as follows:
   *
   * <pre>
   * UPDATE t SET t.age = 30 FROM employee t
   * </pre>
   *
   * @return {@code true}, if this object supports it
   */
  default boolean supportsAliasInUpdateClause() {
    return false;
  }

  /**
   * Determines whether the use of table aliases is supported in the context of an UPDATE statement.
   *
   * <pre>
   * UPDATE employee t SET t.age = 30
   * </pre>
   *
   * @return true if table aliasing is supported in UPDATE statements, false otherwise
   */
  default boolean supportsAliasInUpdateStatement() {
    return true;
  }

  /**
   * Determines whether this object supports mod operator {@code %}.
   *
   * @return {@code true}, if this object supports it
   */
  default boolean supportsModOperator() {
    return true;
  }

  /**
   * Determines whether this dialect supports multi-row insert statements (bulk inserts).
   *
   * <p>Multi-row insert statements allow inserting multiple rows in a single SQL statement, which
   * can significantly improve performance when inserting large amounts of data.
   *
   * <p>The default implementation returns {@code true}.
   *
   * @return {@code true} if this dialect supports multi-row insert statements, {@code false}
   *     otherwise
   */
  default boolean supportsMultiRowInsertStatement() {
    return true;
  }

  /**
   * Determines whether this dialect supports auto-increment columns when inserting multiple rows.
   *
   * <p>Some databases do not properly handle auto-increment columns in multi-row insert statements,
   * or have limitations on how generated values can be retrieved.
   *
   * <p>The default implementation returns {@code true}.
   *
   * @return {@code true} if this dialect supports auto-increment columns in multi-row inserts,
   *     {@code false} otherwise
   */
  default boolean supportsAutoIncrementWhenInsertingMultipleRows() {
    return true;
  }

  /**
   * Determines whether this dialect supports upsert emulation using MERGE statements.
   *
   * <p>An upsert operation (update or insert) can be emulated using MERGE statements in some
   * database systems. This method indicates whether the dialect supports this approach.
   *
   * <p>The default implementation returns {@code false}.
   *
   * @return {@code true} if this dialect supports upsert emulation with MERGE statements, {@code
   *     false} otherwise
   */
  default boolean supportsUpsertEmulationWithMergeStatement() {
    return false;
  }

  /**
   * Determines whether this dialect requires parentheses around set operation operands.
   *
   * <p>Set operations include UNION, INTERSECT, and EXCEPT. Some databases require parentheses
   * around the operands of these operations, especially when they include ORDER BY or LIMIT
   * clauses.
   *
   * <p>The default implementation returns {@code true}.
   *
   * @return {@code true} if this dialect requires parentheses for set operands, {@code false}
   *     otherwise
   */
  default boolean supportsParenthesesForSetOperands() {
    return true;
  }

  /**
   * Returns an SQL object to retrieve IDENTITY values that were generated during an insert
   * operation.
   *
   * <p>This method is used to create the SQL statement that retrieves the last generated identity
   * value for a specific table and column.
   *
   * <p>This method is available only if {@link #supportsIdentity()} returns {@code true}.
   *
   * @param catalogName the catalog name of the table (may be {@code null} if not applicable)
   * @param schemaName the schema name of the table (may be {@code null} if not applicable)
   * @param tableName the name of the table containing the identity column
   * @param columnName the name of the identity column
   * @param isQuoteRequired whether the table name should be quoted in the SQL statement
   * @param isIdColumnQuoteRequired whether the identity column name should be quoted in the SQL
   *     statement
   * @return the SQL object that can be executed to retrieve the generated identity value
   * @throws DomaNullPointerException if either the {@code tableName} or the {@code columnName} is
   *     {@code null}
   */
  Sql<?> getIdentitySelectSql(
      String catalogName,
      String schemaName,
      String tableName,
      String columnName,
      boolean isQuoteRequired,
      boolean isIdColumnQuoteRequired);

  /**
   * Returns an SQL object to reserve identity in the database.
   *
   * <p>This method is available, only if {@link #supportsIdentityReservation()} returns {@code
   * true}.
   *
   * @param catalogName the catalog name
   * @param schemaName the schema name
   * @param tableName the table name
   * @param columnName the IDENTITY column name
   * @param isQuoteRequired whether the quotation marks are required
   * @param isIdColumnQuoteRequired whether the quotation marks are required for the IDENTITY column
   * @param reservationSize the size of the reservation
   * @return the SQL object
   * @throws DomaNullPointerException if either the {@code tableName} or the {@code columnName} is
   *     {@code null}
   */
  @Deprecated
  Sql<?> getIdentityReservationSql(
      String catalogName,
      String schemaName,
      String tableName,
      String columnName,
      boolean isQuoteRequired,
      boolean isIdColumnQuoteRequired,
      int reservationSize);

  /**
   * Returns an SQL object to retrieve the next value from a database sequence.
   *
   * <p>This method creates the SQL statement needed to get the next value from a sequence.
   *
   * <p>The allocation size parameter can be used to optimize sequence access by retrieving multiple
   * values at once in database systems that support this feature.
   *
   * <p>This method is available only if {@link #supportsSequence()} returns {@code true}.
   *
   * @param qualifiedSequenceName the fully qualified name of the sequence, which may include
   *     catalog and schema information depending on the database system
   * @param allocationSize the number of sequence values to allocate at once for optimization; a
   *     value greater than 1 may improve performance by reducing database calls
   * @return the SQL object that can be executed to retrieve the next sequence value(s)
   * @throws DomaNullPointerException if {@code qualifiedSequenceName} is {@code null}
   */
  Sql<?> getSequenceNextValSql(String qualifiedSequenceName, long allocationSize);

  /**
   * Returns the {@link JdbcType} object that corresponds to the {@link ResultSet} class.
   *
   * <p>This method is available, only if {@link #supportsResultSetReturningAsOutParameter()} is
   * {@code true}.
   *
   * @return the {@link JdbcType} object for the {@link ResultSet} class
   */
  JdbcType<ResultSet> getResultSetType();

  /**
   * Encloses the name with quotation marks.
   *
   * @param name the name of a database object such as a table, a column, and so on
   * @return the name that is enclosed with quotation marks
   */
  String applyQuote(String name);

  /**
   * Removes quotation marks from the name
   *
   * @param name the name of a database object such as a table, a column, and so on
   * @return the name that has no enclosing quotation marks
   */
  String removeQuote(String name);

  /**
   * Returns the root cause of the SQL exception.
   *
   * @param sqlException the SQL exception
   * @return the root cause
   * @throws DomaNullPointerException if the {@code sqlException} is {@code null}
   */
  Throwable getRootCause(SQLException sqlException);

  /**
   * Returns the visitor that maps {@link Wrapper} to {@link JdbcType}.
   *
   * @return the visitor
   */
  JdbcMappingVisitor getJdbcMappingVisitor();

  /**
   * Returns the visitor that maps {@link Wrapper} to {@link SqlLogFormatter}.
   *
   * @return the visitor
   */
  SqlLogFormattingVisitor getSqlLogFormattingVisitor();

  /**
   * Returns the aggregation of the expression functions that are available in the SQL templates.
   *
   * @return the aggregation of the expression functions
   */
  ExpressionFunctions getExpressionFunctions();

  /**
   * Creates the context object to process an SQL block in a script.
   *
   * @return the context object
   */
  ScriptBlockContext createScriptBlockContext();

  /**
   * Returns the delimiter that is used as the end of an SQL block in a script.
   *
   * @return the delimiter
   */
  String getScriptBlockDelimiter();

  /**
   * Returns the type of the auto generated keys.
   *
   * @return the type of the auto generated keys
   */
  AutoGeneratedKeysType getAutoGeneratedKeysType();

  /**
   * Returns the criteria builder for this dialect.
   *
   * <p>The criteria builder is used to construct SQL queries using the criteria API, which provides
   * a type-safe way to build queries programmatically. The returned builder will generate SQL that
   * is compatible with the specific database dialect.
   *
   * @return the criteria builder for this dialect
   */
  CriteriaBuilder getCriteriaBuilder();

  /**
   * Returns the UpsertAssembler implementation for the given context.
   *
   * @param context the UpsertAssemblerContext object
   * @return the UpsertAssembler object for the given context
   */
  UpsertAssembler getUpsertAssembler(UpsertAssemblerContext context);

  /**
   * Provides an implementation of {@code MultiInsertAssembler} based on the given context.
   *
   * @param <ENTITY> the type of the entity to be handled by the assembler
   * @param context the context that holds the information required to create the {@code
   *     MultiInsertAssembler}
   * @return an instance of {@code MultiInsertAssembler} specific to the provided entity context
   */
  <ENTITY> MultiInsertAssembler getMultiInsertAssembler(
      MultiInsertAssemblerContext<ENTITY> context);

  /**
   * Provides an implementation of {@code InsertAssembler} based on the given context.
   *
   * @param <ENTITY> the type of the entity to be handled by the assembler
   * @param context the context that holds the information required to create the {@code
   *     InsertAssembler}
   * @return an instance of {@code InsertAssembler} specific to the provided entity context
   */
  default <ENTITY> InsertAssembler getInsertAssembler(InsertAssemblerContext<ENTITY> context) {
    return new DefaultInsertAssembler<>(context);
  }

  /**
   * Provides an implementation of {@code UpdateAssembler} based on the given context.
   *
   * @param <ENTITY> the type of the entity to be handled by the assembler
   * @param context the context that holds the information required to create the {@code
   *     UpdateAssembler}
   * @return an instance of {@code UpdateAssembler} specific to the provided entity context
   */
  default <ENTITY> UpdateAssembler getUpdateAssembler(UpdateAssemblerContext<ENTITY> context) {
    return new DefaultUpdateAssembler<>(context);
  }

  /**
   * Provides an implementation of {@code DeleteAssembler} based on the given context.
   *
   * @param <ENTITY> the type of the entity to be handled by the assembler
   * @param context the context that holds the information required to create the {@code
   *     DeleteAssembler}
   * @return an instance of {@code DeleteAssembler} specific to the provided entity context
   */
  default <ENTITY> DeleteAssembler getDeleteAssembler(DeleteAssemblerContext<ENTITY> context) {
    return new DefaultDeleteAssembler<>(context);
  }

  /**
   * Determines if this dialect supports the SQL RETURNING clause in database operations.
   *
   * <p>The RETURNING clause allows a DML statement (INSERT, UPDATE, DELETE) to return data from
   * rows that were affected by the operation. This feature is particularly useful for retrieving
   * auto-generated values or updated columns in a single database operation, without requiring a
   * separate SELECT statement.
   *
   * <p>The default implementation returns {@code true}, but dialect implementations for databases
   * that don't support this feature should override this method to return {@code false}.
   *
   * @return {@code true} if the RETURNING clause is supported by this dialect, {@code false}
   *     otherwise
   */
  default boolean supportsReturning() {
    return true;
  }
}
