/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.dialect;

import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.function.Consumer;
import org.seasar.doma.DomaNullPointerException;
import org.seasar.doma.expr.ExpressionFunctions;
import org.seasar.doma.internal.jdbc.dialect.Mssql2008ForUpdateTransformer;
import org.seasar.doma.internal.jdbc.dialect.Mssql2008PagingTransformer;
import org.seasar.doma.internal.jdbc.sql.PreparedSqlBuilder;
import org.seasar.doma.jdbc.JdbcMappingVisitor;
import org.seasar.doma.jdbc.ScriptBlockContext;
import org.seasar.doma.jdbc.SelectForUpdateType;
import org.seasar.doma.jdbc.SqlLogFormattingVisitor;
import org.seasar.doma.jdbc.SqlNode;
import org.seasar.doma.jdbc.criteria.metamodel.PropertyMetamodel;
import org.seasar.doma.jdbc.criteria.option.ForUpdateOption;
import org.seasar.doma.jdbc.criteria.query.AliasManager;
import org.seasar.doma.jdbc.criteria.query.CriteriaBuilder;

/** A dialect for Microsoft SQL Server 2008 and below. */
public class Mssql2008Dialect extends StandardDialect {

  /** the error code that represents unique violation */
  protected static final int UNIQUE_CONSTRAINT_VIOLATION_ERROR_CODE = 2627;

  /** the quotation mark of the start */
  protected static final char OPEN_QUOTE = '[';

  /** the quotation mark of the end */
  protected static final char CLOSE_QUOTE = ']';

  public Mssql2008Dialect() {
    this(
        new Mssql2008JdbcMappingVisitor(),
        new Mssql2008SqlLogFormattingVisitor(),
        new Mssql2008ExpressionFunctions());
  }

  public Mssql2008Dialect(JdbcMappingVisitor jdbcMappingVisitor) {
    this(
        jdbcMappingVisitor,
        new Mssql2008SqlLogFormattingVisitor(),
        new Mssql2008ExpressionFunctions());
  }

  public Mssql2008Dialect(SqlLogFormattingVisitor sqlLogFormattingVisitor) {
    this(
        new Mssql2008JdbcMappingVisitor(),
        sqlLogFormattingVisitor,
        new Mssql2008ExpressionFunctions());
  }

  public Mssql2008Dialect(ExpressionFunctions expressionFunctions) {
    this(
        new Mssql2008JdbcMappingVisitor(),
        new Mssql2008SqlLogFormattingVisitor(),
        expressionFunctions);
  }

  public Mssql2008Dialect(
      JdbcMappingVisitor jdbcMappingVisitor, SqlLogFormattingVisitor sqlLogFormattingVisitor) {
    this(jdbcMappingVisitor, sqlLogFormattingVisitor, new Mssql2008ExpressionFunctions());
  }

  public Mssql2008Dialect(
      JdbcMappingVisitor jdbcMappingVisitor,
      SqlLogFormattingVisitor sqlLogFormattingVisitor,
      ExpressionFunctions expressionFunctions) {
    super(jdbcMappingVisitor, sqlLogFormattingVisitor, expressionFunctions);
  }

  @Override
  public String getName() {
    return "mssql";
  }

  @Override
  protected SqlNode toForUpdateSqlNode(
      SqlNode sqlNode, SelectForUpdateType forUpdateType, int waitSeconds, String... aliases) {
    Mssql2008ForUpdateTransformer transformer =
        new Mssql2008ForUpdateTransformer(forUpdateType, waitSeconds, aliases);
    return transformer.transform(sqlNode);
  }

  @Override
  protected SqlNode toPagingSqlNode(SqlNode sqlNode, long offset, long limit) {
    Mssql2008PagingTransformer transformer = new Mssql2008PagingTransformer(offset, limit);
    return transformer.transform(sqlNode);
  }

  @Override
  public boolean isUniqueConstraintViolated(SQLException sqlException) {
    if (sqlException == null) {
      throw new DomaNullPointerException("sqlException");
    }
    int errorCode = getErrorCode(sqlException);
    return errorCode == UNIQUE_CONSTRAINT_VIOLATION_ERROR_CODE;
  }

  @Override
  public boolean supportsIdentity() {
    return true;
  }

  @Override
  public boolean supportsSelectForUpdate(SelectForUpdateType type, boolean withTargets) {
    return (type == SelectForUpdateType.NORMAL || type == SelectForUpdateType.NOWAIT)
        && !withTargets;
  }

  @Override
  public boolean supportsAutoGeneratedKeys() {
    return true;
  }

  @Override
  public boolean supportsAliasInDeleteClause() {
    return true;
  }

  @Override
  public boolean supportsAliasInUpdateClause() {
    return true;
  }

  @Override
  public String getScriptBlockDelimiter() {
    return "GO";
  }

  @Override
  public String applyQuote(String name) {
    return OPEN_QUOTE + name + CLOSE_QUOTE;
  }

  @Override
  public ScriptBlockContext createScriptBlockContext() {
    return new Mssql2008ScriptBlockContext();
  }

  @Override
  public CriteriaBuilder getCriteriaBuilder() {
    return new Mssql2008CriteriaBuilder();
  }

  public static class Mssql2008JdbcMappingVisitor extends StandardJdbcMappingVisitor {}

  public static class Mssql2008SqlLogFormattingVisitor extends StandardSqlLogFormattingVisitor {}

  public static class Mssql2008ExpressionFunctions extends StandardExpressionFunctions {

    private static final char[] DEFAULT_WILDCARDS = {'%', '_', '['};

    public Mssql2008ExpressionFunctions() {
      super(DEFAULT_WILDCARDS);
    }

    public Mssql2008ExpressionFunctions(char[] wildcards) {
      super(wildcards);
    }

    protected Mssql2008ExpressionFunctions(char escapeChar, char[] wildcards) {
      super(escapeChar, wildcards);
    }
  }

  public static class Mssql2008ScriptBlockContext extends StandardScriptBlockContext {

    protected Mssql2008ScriptBlockContext() {
      sqlBlockStartKeywordsList.add(Arrays.asList("create", "procedure"));
      sqlBlockStartKeywordsList.add(Arrays.asList("create", "function"));
      sqlBlockStartKeywordsList.add(Arrays.asList("create", "trigger"));
      sqlBlockStartKeywordsList.add(Arrays.asList("alter", "procedure"));
      sqlBlockStartKeywordsList.add(Arrays.asList("alter", "function"));
      sqlBlockStartKeywordsList.add(Arrays.asList("alter", "trigger"));
      sqlBlockStartKeywordsList.add(Collections.singletonList("declare"));
      sqlBlockStartKeywordsList.add(Collections.singletonList("begin"));
    }
  }

  public static class Mssql2008CriteriaBuilder extends StandardCriteriaBuilder {

    public void concat(PreparedSqlBuilder buf, Runnable leftOperand, Runnable rightOperand) {
      buf.appendSql("(");
      leftOperand.run();
      buf.appendSql(" + ");
      rightOperand.run();
      buf.appendSql(")");
    }

    @Override
    public void lockWithTableHint(
        PreparedSqlBuilder buf, ForUpdateOption option, Consumer<PropertyMetamodel<?>> column) {
      option.accept(
          new ForUpdateOption.Visitor() {

            @Override
            public void visit(ForUpdateOption.Basic basic) {
              buf.appendSql(" with (updlock, rowlock)");
            }

            @Override
            public void visit(ForUpdateOption.NoWait noWait) {
              buf.appendSql(" with (updlock, rowlock, nowait)");
            }

            @Override
            public void visit(ForUpdateOption.Wait wait) {
              buf.appendSql(" with (updlock, rowlock)");
            }
          });
    }

    @Override
    public void forUpdate(
        PreparedSqlBuilder buf,
        ForUpdateOption option,
        Consumer<PropertyMetamodel<?>> column,
        AliasManager aliasManager) {}
  }
}
