/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.dialect;

import java.sql.SQLException;
import java.util.Collections;
import org.seasar.doma.DomaNullPointerException;
import org.seasar.doma.expr.ExpressionFunctions;
import org.seasar.doma.internal.jdbc.dialect.MysqlPagingTransformer;
import org.seasar.doma.internal.jdbc.dialect.SqlitePagingTransformer;
import org.seasar.doma.internal.jdbc.sql.PreparedSqlBuilder;
import org.seasar.doma.jdbc.JdbcMappingVisitor;
import org.seasar.doma.jdbc.PreparedSql;
import org.seasar.doma.jdbc.SelectForUpdateType;
import org.seasar.doma.jdbc.SqlKind;
import org.seasar.doma.jdbc.SqlLogFormattingVisitor;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.SqlNode;
import org.seasar.doma.jdbc.criteria.query.CriteriaBuilder;
import org.seasar.doma.jdbc.query.UpsertAssembler;
import org.seasar.doma.jdbc.query.UpsertAssemblerContext;

/** A dialect for SQLite. */
public class SqliteDialect extends StandardDialect {

  public SqliteDialect() {
    this(
        new SqliteJdbcMappingVisitor(),
        new SqliteSqlLogFormattingVisitor(),
        new SqliteExpressionFunctions());
  }

  public SqliteDialect(JdbcMappingVisitor jdbcMappingVisitor) {
    this(jdbcMappingVisitor, new SqliteSqlLogFormattingVisitor(), new SqliteExpressionFunctions());
  }

  public SqliteDialect(SqlLogFormattingVisitor sqlLogFormattingVisitor) {
    this(new SqliteJdbcMappingVisitor(), sqlLogFormattingVisitor, new SqliteExpressionFunctions());
  }

  public SqliteDialect(ExpressionFunctions expressionFunctions) {
    this(new SqliteJdbcMappingVisitor(), new SqliteSqlLogFormattingVisitor(), expressionFunctions);
  }

  public SqliteDialect(
      JdbcMappingVisitor jdbcMappingVisitor, SqlLogFormattingVisitor sqlLogFormattingVisitor) {
    this(jdbcMappingVisitor, sqlLogFormattingVisitor, new SqliteExpressionFunctions());
  }

  public SqliteDialect(
      JdbcMappingVisitor jdbcMappingVisitor,
      SqlLogFormattingVisitor sqlLogFormattingVisitor,
      ExpressionFunctions expressionFunctions) {
    super(jdbcMappingVisitor, sqlLogFormattingVisitor, expressionFunctions);
  }

  @Override
  public String getName() {
    return "sqlite";
  }

  @Override
  public PreparedSql getIdentitySelectSql(
      String catalogName,
      String schemaName,
      String tableName,
      String columnName,
      boolean isQuoteRequired,
      boolean isIdColumnQuoteRequired) {
    if (tableName == null) {
      throw new DomaNullPointerException("tableName");
    }
    if (columnName == null) {
      throw new DomaNullPointerException("columnName");
    }
    String rawSql = "select last_insert_rowid()";
    return new PreparedSql(
        SqlKind.SELECT, rawSql, rawSql, null, Collections.emptyList(), SqlLogType.FORMATTED);
  }

  @Override
  protected SqlNode toPagingSqlNode(SqlNode sqlNode, long offset, long limit) {
    SqlitePagingTransformer transformer = new SqlitePagingTransformer(offset, limit);
    return transformer.transform(sqlNode);
  }

  @Override
  protected SqlNode toForUpdateSqlNode(
      SqlNode sqlNode, SelectForUpdateType forUpdateType, int waitSeconds, String... aliases) {
    return sqlNode;
  }

  @Override
  public boolean supportsIdentity() {
    return true;
  }

  @Override
  public boolean supportsAutoGeneratedKeys() {
    return true;
  }

  @Override
  public boolean supportsAliasInDeleteStatement() {
    return false;
  }

  @Override
  public boolean supportsAliasInUpdateStatement() {
    return false;
  }

  @Override
  public boolean supportsAutoIncrementWhenInsertingMultipleRows() {
    return false;
  }

  @Override
  public boolean supportsParenthesesForSetOperands() {
    return false;
  }

  @Override
  public boolean isUniqueConstraintViolated(SQLException sqlException) {
    if (sqlException == null) {
      throw new DomaNullPointerException("sqlException");
    }
    SQLException cause = getCauseSQLException(sqlException);
    String message = cause.getMessage();
    return message != null
        && (message.startsWith("[SQLITE_CONSTRAINT_PRIMARYKEY]")
            || message.startsWith("[SQLITE_CONSTRAINT_UNIQUE]"));
  }

  @Override
  public CriteriaBuilder getCriteriaBuilder() {
    return new SqliteCriteriaBuilder();
  }

  @Override
  public UpsertAssembler getUpsertAssembler(UpsertAssemblerContext context) {
    return new SqliteUpsertAssembler(context);
  }

  public static class SqliteJdbcMappingVisitor extends StandardJdbcMappingVisitor {}

  public static class SqliteSqlLogFormattingVisitor extends StandardSqlLogFormattingVisitor {}

  public static class SqliteExpressionFunctions extends StandardExpressionFunctions {

    public SqliteExpressionFunctions() {
      super();
    }

    public SqliteExpressionFunctions(char[] wildcards) {
      super(wildcards);
    }

    protected SqliteExpressionFunctions(char escapeChar, char[] wildcards) {
      super(escapeChar, wildcards);
    }
  }

  public static class SqliteCriteriaBuilder extends StandardCriteriaBuilder {

    @Override
    public void offsetAndFetch(PreparedSqlBuilder buf, int offset, int limit) {
      buf.appendSql(" limit ");
      if (limit > 0) {
        buf.appendSql(Integer.toString(limit));
      } else {
        buf.appendSql(MysqlPagingTransformer.MAXIMUM_LIMIT);
      }
      buf.appendSql(" offset ");
      buf.appendSql(Integer.toString(offset));
    }
  }
}
