/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.id;

import java.sql.Statement;
import java.util.Collections;
import java.util.List;
import org.seasar.doma.GenerationType;
import org.seasar.doma.jdbc.JdbcException;

/**
 * An identity generator.
 *
 * <p>Either {@link #generatePreInsert(IdGenerationConfig)} or {@link
 * #generatePostInsert(IdGenerationConfig, Statement)} must return non-null value.
 *
 * <p>The implementation class must have a public no-args constructor.
 *
 * <p>The implementation class must be thread safe.
 */
public interface IdGenerator {

  /**
   * Whether this generator supports the batch insert.
   *
   * @param config the configuration
   * @return {@code true} if this generator supports
   */
  boolean supportsBatch(IdGenerationConfig config);

  /**
   * Whether this generator supports {@link Statement#getGeneratedKeys()}.
   *
   * @param config the configuration
   * @return {@code true} if this generator supports
   */
  boolean supportsAutoGeneratedKeys(IdGenerationConfig config);

  /**
   * Whether this generator includes the identity column into SQL INSERT statements.
   *
   * @param config the configuration
   * @return {@code true} if supported
   */
  @Deprecated
  boolean includesIdentityColumn(IdGenerationConfig config);

  /**
   * Determines whether the identity column is included in the SQL INSERT statements based on the
   * given configuration and identity value.
   *
   * <p>The type of {@code idValue} must be one of the basic types. It must never be a domain class
   * or an Optional type that wraps a basic type.
   *
   * @param config the configuration for identity generation
   * @param idValue an identity value; may be null
   * @return {@code true} if the identity column is included in the SQL INSERT statements, otherwise
   *     {@code false}
   */
  default boolean includesIdentityColumn(IdGenerationConfig config, Object idValue) {
    return includesIdentityColumn(config);
  }

  /**
   * Generates the identity value before an insert.
   *
   * @param config the configuration
   * @return the generated value or {@code null} if not supported
   * @throws JdbcException if the generation is failed
   */
  Long generatePreInsert(IdGenerationConfig config);

  default List<Long> generateValuesPreInsert(IdGenerationConfig config, int count) {
    return Collections.emptyList();
  }

  /**
   * Generates the identity value after an insert.
   *
   * @param config the configuration
   * @param statement the SQL INSERT statement
   * @return the generated value or {@code null} if not supported
   * @throws JdbcException if the generation is failed
   */
  Long generatePostInsert(IdGenerationConfig config, Statement statement);

  default List<Long> generateValuesPostInsert(IdGenerationConfig config, Statement statement) {
    return Collections.emptyList();
  }

  /**
   * Returns the generation type.
   *
   * @return the generation type
   */
  GenerationType getGenerationType();
}
