/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.query;

import static org.seasar.doma.internal.util.AssertionUtil.assertNotNull;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.seasar.doma.jdbc.JdbcException;
import org.seasar.doma.jdbc.PreparedSql;
import org.seasar.doma.jdbc.SqlExecutionSkipCause;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.entity.EntityPropertyType;
import org.seasar.doma.jdbc.entity.EntityType;
import org.seasar.doma.jdbc.entity.TenantIdPropertyType;
import org.seasar.doma.jdbc.entity.VersionPropertyType;
import org.seasar.doma.message.Message;

/**
 * An abstract base class for auto batch modification queries.
 *
 * <p>This class provides common functionality for batch operations that modify entities in the
 * database, such as batch insert, update, and delete operations. It manages entity properties,
 * optimistic locking, and SQL generation for batch operations.
 *
 * @param <ENTITY> the entity type
 */
public abstract class AutoBatchModifyQuery<ENTITY> extends AbstractQuery
    implements BatchModifyQuery {

  /** Empty string array constant. */
  protected static final String[] EMPTY_STRINGS = new String[] {};

  /** The property types targeted by this query. */
  protected List<EntityPropertyType<ENTITY, ?>> targetPropertyTypes;

  /** The ID property types of the entity. */
  protected List<EntityPropertyType<ENTITY, ?>> idPropertyTypes;

  /** The names of properties to be included in the query. */
  protected String[] includedPropertyNames = EMPTY_STRINGS;

  /** The names of properties to be excluded from the query. */
  protected String[] excludedPropertyNames = EMPTY_STRINGS;

  /** The entity type. */
  protected final EntityType<ENTITY> entityType;

  /** The version property type for optimistic locking. */
  protected VersionPropertyType<ENTITY, ?, ?> versionPropertyType;

  /** The tenant ID property type for multi-tenancy. */
  protected TenantIdPropertyType<ENTITY, ?, ?> tenantIdPropertyType;

  /** Whether optimistic lock checking is required. */
  protected boolean optimisticLockCheckRequired;

  /** Whether auto-generated keys are supported. */
  protected boolean autoGeneratedKeysSupported;

  /** Whether this query is executable. */
  protected boolean executable;

  /** The cause of SQL execution skip if the query is not executable. */
  protected SqlExecutionSkipCause sqlExecutionSkipCause =
      SqlExecutionSkipCause.BATCH_TARGET_NONEXISTENT;

  /** The list of SQL statements to be executed in batch. */
  protected List<PreparedSql> sqls;

  /** The list of entities to be processed. */
  protected List<ENTITY> entities;

  /** The current entity being processed. */
  protected ENTITY currentEntity;

  /** The batch size. */
  protected int batchSize;

  /** The SQL log type. */
  protected SqlLogType sqlLogType;

  public AutoBatchModifyQuery(EntityType<ENTITY> entityType) {
    assertNotNull(entityType);
    this.entityType = entityType;
  }

  /**
   * Prepares the ID, version, and tenant ID property types from the entity type.
   *
   * <p>This method initializes the property types that are used for identifying entities and
   * handling optimistic locking and multi-tenancy.
   */
  protected void prepareIdAndVersionPropertyTypes() {
    idPropertyTypes = entityType.getIdPropertyTypes();
    versionPropertyType = entityType.getVersionPropertyType();
    tenantIdPropertyType = entityType.getTenantIdPropertyType();
  }

  /**
   * Validates that the entity has at least one ID property.
   *
   * <p>This method throws a {@link JdbcException} if the entity doesn't have any ID properties, as
   * batch operations require entities to be identifiable.
   *
   * @throws JdbcException if the entity doesn't have any ID properties
   */
  protected void validateIdExistent() {
    if (idPropertyTypes.isEmpty()) {
      throw new JdbcException(Message.DOMA2022, entityType.getName());
    }
  }

  /**
   * Prepares the query options.
   *
   * <p>This method initializes the query timeout and batch size from the configuration if they
   * haven't been explicitly set or if they have invalid values.
   */
  protected void prepareOptions() {
    if (queryTimeout <= 0) {
      queryTimeout = config.getQueryTimeout();
    }
    if (batchSize <= 0) {
      batchSize = config.getBatchSize();
    }
  }

  /**
   * Determines if a property should be included in the query based on its name.
   *
   * <p>This method checks if the property name is in the included properties list and not in the
   * excluded properties list. If no included properties are specified, all properties except those
   * explicitly excluded are considered targets.
   *
   * @param name the property name to check
   * @return {@code true} if the property should be included, {@code false} otherwise
   */
  protected boolean isTargetPropertyName(String name) {
    if (includedPropertyNames.length > 0) {
      for (String includedName : includedPropertyNames) {
        if (includedName.equals(name)) {
          for (String excludedName : excludedPropertyNames) {
            if (excludedName.equals(name)) {
              return false;
            }
          }
          return true;
        }
      }
      return false;
    }
    if (excludedPropertyNames.length > 0) {
      for (String excludedName : excludedPropertyNames) {
        if (excludedName.equals(name)) {
          return false;
        }
      }
      return true;
    }
    return true;
  }

  /**
   * Sets the entities to be processed by this batch query.
   *
   * <p>This method initializes the internal list of entities and prepares the SQL statements list
   * with the same size as the entities list.
   *
   * @param entities the entities to be processed
   * @throws NullPointerException if {@code entities} is {@code null}
   */
  public void setEntities(Iterable<ENTITY> entities) {
    assertNotNull(entities);
    if (entities instanceof Collection<?>) {
      this.entities = new ArrayList<>((Collection<ENTITY>) entities);
    } else {
      this.entities = new ArrayList<>();
      for (ENTITY entity : entities) {
        this.entities.add(entity);
      }
    }
    this.sqls = new ArrayList<>(this.entities.size());
  }

  /**
   * Returns the list of entities to be processed by this batch query.
   *
   * @return the list of entities
   */
  public List<ENTITY> getEntities() {
    return entities;
  }

  /**
   * Sets the batch size for this query.
   *
   * <p>The batch size determines how many entities are processed in a single batch operation.
   *
   * @param batchSize the batch size
   */
  public void setBatchSize(int batchSize) {
    this.batchSize = batchSize;
  }

  /**
   * Sets the names of properties to be included in the query.
   *
   * <p>If this is set, only the specified properties will be included in the query, unless they are
   * also in the excluded properties list.
   *
   * @param includedPropertyNames the names of properties to include
   */
  public void setIncludedPropertyNames(String... includedPropertyNames) {
    this.includedPropertyNames = includedPropertyNames;
  }

  /**
   * Sets the names of properties to be excluded from the query.
   *
   * <p>If this is set, the specified properties will be excluded from the query, even if they are
   * in the included properties list.
   *
   * @param excludedPropertyNames the names of properties to exclude
   */
  public void setExcludedPropertyNames(String... excludedPropertyNames) {
    this.excludedPropertyNames = excludedPropertyNames;
  }

  /**
   * Sets the SQL log type for this query.
   *
   * <p>The SQL log type determines how SQL statements are logged.
   *
   * @param sqlLogType the SQL log type
   */
  public void setSqlLogType(SqlLogType sqlLogType) {
    this.sqlLogType = sqlLogType;
  }

  /**
   * Returns the first SQL statement in the batch.
   *
   * @return the first SQL statement
   */
  @Override
  public PreparedSql getSql() {
    return sqls.get(0);
  }

  /**
   * Returns all SQL statements in the batch.
   *
   * @return the list of SQL statements
   */
  @Override
  public List<PreparedSql> getSqls() {
    return sqls;
  }

  /**
   * Indicates whether optimistic lock checking is required.
   *
   * @return {@code true} if optimistic lock checking is required, {@code false} otherwise
   */
  @Override
  public boolean isOptimisticLockCheckRequired() {
    return optimisticLockCheckRequired;
  }

  /**
   * Indicates whether auto-generated keys are supported.
   *
   * @return {@code true} if auto-generated keys are supported, {@code false} otherwise
   */
  @Override
  public boolean isAutoGeneratedKeysSupported() {
    return autoGeneratedKeysSupported;
  }

  /**
   * Indicates whether this query is executable.
   *
   * @return {@code true} if this query is executable, {@code false} otherwise
   */
  @Override
  public boolean isExecutable() {
    return executable;
  }

  /**
   * Returns the cause of SQL execution skip if the query is not executable.
   *
   * @return the cause of SQL execution skip, or {@code null} if the query is executable
   */
  @Override
  public SqlExecutionSkipCause getSqlExecutionSkipCause() {
    return sqlExecutionSkipCause;
  }

  /**
   * Returns the batch size for this query.
   *
   * @return the batch size
   */
  @Override
  public int getBatchSize() {
    return batchSize;
  }

  /**
   * Returns the SQL log type for this query.
   *
   * @return the SQL log type
   */
  @Override
  public SqlLogType getSqlLogType() {
    return sqlLogType;
  }

  /**
   * Returns a string representation of this query.
   *
   * @return a string representation of this query
   */
  @Override
  public String toString() {
    return sqls.toString();
  }
}
