/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.query;

import java.util.List;
import org.seasar.doma.internal.util.AssertionUtil;
import org.seasar.doma.jdbc.JdbcException;
import org.seasar.doma.jdbc.PreparedSql;
import org.seasar.doma.jdbc.SqlExecutionSkipCause;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.entity.EntityPropertyType;
import org.seasar.doma.jdbc.entity.EntityType;
import org.seasar.doma.jdbc.entity.TenantIdPropertyType;
import org.seasar.doma.jdbc.entity.VersionPropertyType;
import org.seasar.doma.message.Message;

/**
 * An abstract base class for queries that automatically modify entities in a database.
 *
 * <p>This class provides common functionality for entity-based modification operations such as
 * INSERT, UPDATE, and DELETE. It handles:
 *
 * <ul>
 *   <li>Entity property filtering (inclusion/exclusion)
 *   <li>ID property management
 *   <li>Version property handling for optimistic locking
 *   <li>Tenant ID property handling for multi-tenancy
 *   <li>SQL generation and execution control
 * </ul>
 *
 * <p>Subclasses implement specific modification operations by extending this class and providing
 * operation-specific logic.
 *
 * @param <ENTITY> the entity type
 */
public abstract class AutoModifyQuery<ENTITY> extends AbstractQuery implements ModifyQuery {

  /** An empty string array used as default for property name filters. */
  protected static final String[] EMPTY_STRINGS = new String[] {};

  /** Names of properties to be included in the modification operation. */
  protected String[] includedPropertyNames = EMPTY_STRINGS;

  /** Names of properties to be excluded from the modification operation. */
  protected String[] excludedPropertyNames = EMPTY_STRINGS;

  /** The entity type metadata. */
  protected final EntityType<ENTITY> entityType;

  /** The entity instance to be modified. */
  protected ENTITY entity;

  /** The prepared SQL for this query. */
  protected PreparedSql sql;

  /** The property types targeted by this modification operation. */
  protected List<EntityPropertyType<ENTITY, ?>> targetPropertyTypes;

  /** The ID property types of the entity. */
  protected List<EntityPropertyType<ENTITY, ?>> idPropertyTypes;

  /** The version property type for optimistic locking, if the entity has one. */
  protected VersionPropertyType<ENTITY, ?, ?> versionPropertyType;

  /** The tenant ID property type for multi-tenancy, if the entity has one. */
  protected TenantIdPropertyType<ENTITY, ?, ?> tenantIdPropertyType;

  /** Indicates whether optimistic lock checking is required for this query. */
  protected boolean optimisticLockCheckRequired;

  /** Indicates whether auto-generated keys are supported for this query. */
  protected boolean autoGeneratedKeysSupported;

  /** Indicates whether this query is executable. */
  protected boolean executable;

  /** The cause if SQL execution should be skipped. */
  protected SqlExecutionSkipCause sqlExecutionSkipCause = SqlExecutionSkipCause.STATE_UNCHANGED;

  /** The SQL log type for this query. */
  protected SqlLogType sqlLogType;

  /** The properties to be returned from the modification operation. */
  protected ReturningProperties returning = ReturningProperties.NONE;

  /**
   * Constructs an instance.
   *
   * @param entityType the entity type metadata
   */
  protected AutoModifyQuery(EntityType<ENTITY> entityType) {
    AssertionUtil.assertNotNull(entityType);
    this.entityType = entityType;
  }

  /**
   * Prepares this query for execution.
   *
   * <p>This method performs basic preparation steps and validates that the dialect supports
   * returning properties if they are specified.
   *
   * <p>Subclasses should override this method to perform additional preparation steps, but must
   * call {@code super.prepare()} first.
   *
   * @throws JdbcException if returning properties are specified but not supported by the dialect
   */
  @Override
  public void prepare() {
    super.prepare();
    if (!returning.isNone() && !config.getDialect().supportsReturning()) {
      throw new JdbcException(Message.DOMA2240, config.getDialect().getName());
    }
  }

  /**
   * Prepares special property types for this query.
   *
   * <p>This method initializes the ID, version, and tenant ID property types from the entity type
   * metadata.
   */
  protected void prepareSpecialPropertyTypes() {
    idPropertyTypes = entityType.getIdPropertyTypes();
    versionPropertyType = entityType.getVersionPropertyType();
    tenantIdPropertyType = entityType.getTenantIdPropertyType();
  }

  /**
   * Validates that the entity has at least one ID property.
   *
   * <p>This method is typically called by operations that require an ID property, such as UPDATE
   * and DELETE.
   *
   * @throws JdbcException if the entity has no ID properties
   */
  protected void validateIdExistent() {
    if (idPropertyTypes.isEmpty()) {
      throw new JdbcException(Message.DOMA2022, entityType.getName());
    }
  }

  /**
   * Prepares query options.
   *
   * <p>This method sets the query timeout from the configuration if it's not already set.
   */
  protected void prepareOptions() {
    if (queryTimeout <= 0) {
      queryTimeout = config.getQueryTimeout();
    }
  }

  /**
   * Determines whether a property should be included in the modification operation.
   *
   * <p>This method applies the include and exclude filters to determine if a property should be
   * targeted by the operation. The rules are:
   *
   * <ol>
   *   <li>If includedPropertyNames is not empty, only properties in that list are included, unless
   *       they are also in excludedPropertyNames
   *   <li>If includedPropertyNames is empty but excludedPropertyNames is not, all properties except
   *       those in excludedPropertyNames are included
   *   <li>If both lists are empty, all properties are included
   * </ol>
   *
   * @param name the property name to check
   * @return true if the property should be included, false otherwise
   */
  protected boolean isTargetPropertyName(String name) {
    if (includedPropertyNames.length > 0) {
      for (String includedName : includedPropertyNames) {
        if (includedName.equals(name)) {
          for (String excludedName : excludedPropertyNames) {
            if (excludedName.equals(name)) {
              return false;
            }
          }
          return true;
        }
      }
      return false;
    }
    if (excludedPropertyNames.length > 0) {
      for (String excludedName : excludedPropertyNames) {
        if (excludedName.equals(name)) {
          return false;
        }
      }
      return true;
    }
    return true;
  }

  /**
   * Sets the entity instance to be modified.
   *
   * @param entity the entity instance
   */
  public void setEntity(ENTITY entity) {
    this.entity = entity;
  }

  /**
   * Returns the entity instance to be modified.
   *
   * @return the entity instance
   */
  public ENTITY getEntity() {
    return entity;
  }

  /**
   * Sets the names of properties to be included in the modification operation.
   *
   * <p>If this is set, only the specified properties will be included in the operation, unless they
   * are also in the excluded property names.
   *
   * @param includedPropertyNames the property names to include
   */
  public void setIncludedPropertyNames(String... includedPropertyNames) {
    this.includedPropertyNames = includedPropertyNames;
  }

  /**
   * Sets the names of properties to be excluded from the modification operation.
   *
   * <p>If this is set, the specified properties will be excluded from the operation.
   *
   * @param excludedPropertyNames the property names to exclude
   */
  public void setExcludedPropertyNames(String... excludedPropertyNames) {
    this.excludedPropertyNames = excludedPropertyNames;
  }

  /**
   * Sets the SQL log type for this query.
   *
   * @param sqlLogType the SQL log type
   */
  public void setSqlLogType(SqlLogType sqlLogType) {
    this.sqlLogType = sqlLogType;
  }

  /**
   * Sets the properties to be returned from the modification operation.
   *
   * <p>This is used for database systems that support returning clause in modification statements.
   *
   * @param returning the returning properties
   */
  public void setReturning(ReturningProperties returning) {
    this.returning = returning;
  }

  /** {@inheritDoc} */
  @Override
  public PreparedSql getSql() {
    return sql;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isOptimisticLockCheckRequired() {
    return optimisticLockCheckRequired;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isExecutable() {
    return executable;
  }

  /** {@inheritDoc} */
  @Override
  public SqlExecutionSkipCause getSqlExecutionSkipCause() {
    return sqlExecutionSkipCause;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isAutoGeneratedKeysSupported() {
    return autoGeneratedKeysSupported;
  }

  /** {@inheritDoc} */
  @Override
  public SqlLogType getSqlLogType() {
    return sqlLogType;
  }

  /**
   * Returns a string representation of this query.
   *
   * <p>This method returns the string representation of the SQL statement if it has been prepared,
   * or null otherwise.
   *
   * @return the string representation
   */
  @Override
  public String toString() {
    return sql != null ? sql.toString() : null;
  }
}
