/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.query;

import java.util.List;
import org.seasar.doma.jdbc.PreparedSql;
import org.seasar.doma.jdbc.SqlExecutionSkipCause;
import org.seasar.doma.jdbc.SqlLogType;

/**
 * A query that performs batch operations to modify data in a database.
 *
 * <p>This interface defines operations specific to batch data modification statements such as batch
 * INSERT, UPDATE, and DELETE.
 */
public interface BatchModifyQuery extends Query {

  /**
   * Returns the list of prepared SQL statements for this batch query.
   *
   * @return the list of prepared SQL statements
   */
  List<PreparedSql> getSqls();

  /**
   * Returns a representative prepared SQL statement for this batch query.
   *
   * <p>This method is typically used for logging purposes.
   *
   * @return a representative prepared SQL statement
   */
  @Override
  PreparedSql getSql();

  /**
   * Returns whether optimistic lock checking is required for this batch query.
   *
   * @return {@code true} if optimistic lock checking is required
   */
  @SuppressWarnings("BooleanMethodIsAlwaysInverted")
  boolean isOptimisticLockCheckRequired();

  /**
   * Returns whether auto-generated keys are supported for this batch query.
   *
   * @return {@code true} if auto-generated keys are supported
   */
  boolean isAutoGeneratedKeysSupported();

  /**
   * Returns whether this batch query is executable.
   *
   * @return {@code true} if this batch query is executable
   */
  boolean isExecutable();

  /**
   * Returns the cause if SQL execution should be skipped.
   *
   * @return the cause of SQL execution skip, or {@code null} if execution should not be skipped
   */
  SqlExecutionSkipCause getSqlExecutionSkipCause();

  /**
   * Returns the batch size for this query.
   *
   * @return the batch size
   */
  int getBatchSize();

  /**
   * Returns the SQL log type for this batch query.
   *
   * @return the SQL log type
   */
  SqlLogType getSqlLogType();
}
