/*
 * Copyright Doma Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.seasar.doma.jdbc.query;

import static org.seasar.doma.internal.util.AssertionUtil.assertNotNull;

import java.util.LinkedHashMap;
import java.util.Map;
import org.seasar.doma.internal.expr.ExpressionEvaluator;
import org.seasar.doma.internal.expr.Value;
import org.seasar.doma.internal.jdbc.sql.NodePreparedSqlBuilder;
import org.seasar.doma.jdbc.PreparedSql;
import org.seasar.doma.jdbc.SqlExecutionSkipCause;
import org.seasar.doma.jdbc.SqlKind;
import org.seasar.doma.jdbc.SqlLogType;
import org.seasar.doma.jdbc.SqlNode;

/**
 * An abstract base class for SQL-based data modification queries.
 *
 * <p>This class provides common functionality for SQL-based INSERT, UPDATE, and DELETE operations.
 * It handles SQL node processing, parameter binding, and SQL statement preparation.
 */
public abstract class SqlModifyQuery extends AbstractQuery implements ModifyQuery {

  /** The SQL kind (INSERT, UPDATE, or DELETE). */
  protected final SqlKind kind;

  /** The SQL node representing the query. */
  protected SqlNode sqlNode;

  /** The parameters to be bound to the SQL statement. */
  protected final Map<String, Value> parameters = new LinkedHashMap<>();

  /** The prepared SQL statement. */
  protected PreparedSql sql;

  /** Whether optimistic lock checking is required for this query. */
  protected boolean optimisticLockCheckRequired;

  /** The SQL log type for this query. */
  protected SqlLogType sqlLogType;

  /**
   * Constructs a new instance with the specified SQL kind.
   *
   * @param kind the SQL kind
   */
  protected SqlModifyQuery(SqlKind kind) {
    assertNotNull(kind);
    this.kind = kind;
  }

  /**
   * {@inheritDoc}
   *
   * <p>This implementation prepares the SQL statement by processing the SQL node and binding
   * parameters.
   */
  @Override
  public void prepare() {
    super.prepare();
    assertNotNull(sqlNode);
    prepareOptions();
    prepareSql();
    assertNotNull(sql);
  }

  /**
   * Prepares the query options.
   *
   * <p>This method sets the query timeout from the configuration if it's not already set.
   */
  protected void prepareOptions() {
    if (queryTimeout <= 0) {
      queryTimeout = config.getQueryTimeout();
    }
  }

  /**
   * Prepares the SQL statement.
   *
   * <p>This method evaluates expressions in the SQL node and builds the prepared SQL statement.
   */
  protected void prepareSql() {
    ExpressionEvaluator evaluator =
        new ExpressionEvaluator(
            parameters, config.getDialect().getExpressionFunctions(), config.getClassHelper());
    NodePreparedSqlBuilder sqlBuilder =
        new NodePreparedSqlBuilder(config, kind, null, evaluator, sqlLogType);
    sql = sqlBuilder.build(sqlNode, this::comment);
  }

  /** {@inheritDoc} */
  @Override
  public void complete() {}

  /**
   * Sets the SQL node for this query.
   *
   * @param sqlNode the SQL node
   */
  public void setSqlNode(SqlNode sqlNode) {
    this.sqlNode = sqlNode;
  }

  /**
   * Adds a parameter to this query.
   *
   * @param name the parameter name
   * @param type the parameter type
   * @param value the parameter value
   */
  public void addParameter(String name, Class<?> type, Object value) {
    assertNotNull(name, type);
    parameters.put(name, new Value(type, value));
  }

  /** Clears all parameters from this query. */
  public void clearParameters() {
    parameters.clear();
  }

  /**
   * Sets the SQL log type for this query.
   *
   * @param sqlLogType the SQL log type
   */
  public void setSqlLogType(SqlLogType sqlLogType) {
    this.sqlLogType = sqlLogType;
  }

  /** {@inheritDoc} */
  @Override
  public PreparedSql getSql() {
    return sql;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isOptimisticLockCheckRequired() {
    return optimisticLockCheckRequired;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isExecutable() {
    return true;
  }

  /** {@inheritDoc} */
  @Override
  public SqlExecutionSkipCause getSqlExecutionSkipCause() {
    return null;
  }

  /** {@inheritDoc} */
  @Override
  public boolean isAutoGeneratedKeysSupported() {
    return false;
  }

  /** {@inheritDoc} */
  @Override
  public SqlLogType getSqlLogType() {
    return sqlLogType;
  }

  /** {@inheritDoc} */
  @Override
  public String toString() {
    return sql != null ? sql.toString() : null;
  }
}
