/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.scanner.downloadcache;

import java.io.IOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonarsource.scanner.downloadcache.CachedFile;
import org.sonarsource.scanner.downloadcache.Downloader;
import org.sonarsource.scanner.downloadcache.FileHashes;
import org.sonarsource.scanner.downloadcache.HashMismatchException;

public class DownloadCache {
    private static final Logger LOG = LoggerFactory.getLogger(DownloadCache.class);
    private final Path baseDir;
    private final Path tmpDir;
    private final FileHashes hashes;

    DownloadCache(Path baseDir, FileHashes fileHashes) {
        LOG.debug("Download cache base directory: {}", (Object)baseDir);
        this.hashes = fileHashes;
        this.baseDir = DownloadCache.mkdirs(baseDir);
        this.tmpDir = DownloadCache.mkdirs(baseDir.resolve("_tmp"));
    }

    public DownloadCache(Path baseDir) {
        this(baseDir, new FileHashes());
    }

    public Path getBaseDir() {
        return this.baseDir;
    }

    public Optional<Path> get(String filename, String hash) {
        Path cachedFile = this.hashDir(hash).resolve(filename);
        if (Files.exists(cachedFile, new LinkOption[0])) {
            return Optional.of(cachedFile);
        }
        return Optional.empty();
    }

    public CachedFile getOrDownload(String filename, String expectedFileHash, String hashAlgorithm, Downloader downloader) throws HashMismatchException {
        Path hashDir = this.hashDir(expectedFileHash);
        Path targetFile = hashDir.resolve(filename);
        if (Files.exists(targetFile, new LinkOption[0])) {
            return new CachedFile(targetFile, true);
        }
        Path tempFile = this.newTempFile(filename);
        DownloadCache.download(downloader, filename, tempFile);
        String downloadedFileHash = this.hashes.of(tempFile.toFile(), hashAlgorithm);
        if (!expectedFileHash.equals(downloadedFileHash)) {
            throw new HashMismatchException(expectedFileHash, downloadedFileHash, tempFile.toAbsolutePath());
        }
        DownloadCache.mkdirs(hashDir);
        DownloadCache.renameQuietly(tempFile, targetFile);
        return new CachedFile(targetFile, false);
    }

    private static void download(Downloader downloader, String filename, Path tempFile) {
        try {
            downloader.download(filename, tempFile);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to download " + filename + " to " + String.valueOf(tempFile), e);
        }
    }

    private static void renameQuietly(Path sourceFile, Path targetFile) {
        try {
            Files.move(sourceFile, targetFile, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException ex) {
            LOG.warn("Unable to rename {} to {}", (Object)sourceFile.toAbsolutePath(), (Object)targetFile.toAbsolutePath());
            LOG.warn("A copy/delete will be tempted but with no guarantee of atomicity");
            try {
                Files.move(sourceFile, targetFile, new CopyOption[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Failed to move " + String.valueOf(sourceFile.toAbsolutePath()) + " to " + String.valueOf(targetFile), e);
            }
        }
        catch (FileAlreadyExistsException ex) {
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to move " + String.valueOf(sourceFile.toAbsolutePath()) + " to " + String.valueOf(targetFile), e);
        }
    }

    private Path hashDir(String hash) {
        return this.baseDir.resolve(hash);
    }

    private Path newTempFile(String filename) {
        int dotLocation = filename.lastIndexOf(".");
        String prefix = filename;
        String suffix = null;
        if (dotLocation > 0) {
            prefix = filename.substring(0, dotLocation);
            suffix = filename.substring(dotLocation + 1);
        }
        try {
            return Files.createTempFile(this.tmpDir, prefix, suffix, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create temp file in " + String.valueOf(this.tmpDir), e);
        }
    }

    private static Path mkdirs(Path dir) {
        if (!Files.isDirectory(dir, new LinkOption[0])) {
            LOG.debug("Create: {}", (Object)dir);
            try {
                Files.createDirectories(dir, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Unable to create directory: " + String.valueOf(dir), e);
            }
        }
        return dir;
    }
}

