/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.devtools.filewatch;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import org.springframework.boot.devtools.filewatch.ChangedFiles;
import org.springframework.boot.devtools.filewatch.FileChangeListener;
import org.springframework.boot.devtools.filewatch.FolderSnapshot;
import org.springframework.util.Assert;

public class FileSystemWatcher {
    private static final long DEFAULT_POLL_INTERVAL = 1000L;
    private static final long DEFAULT_QUIET_PERIOD = 400L;
    private List<FileChangeListener> listeners = new ArrayList<FileChangeListener>();
    private final boolean daemon;
    private final long pollInterval;
    private final long quietPeriod;
    private Thread watchThread;
    private AtomicInteger remainingScans = new AtomicInteger(-1);
    private Map<File, FolderSnapshot> folders = new LinkedHashMap<File, FolderSnapshot>();
    private FileFilter triggerFilter;

    public FileSystemWatcher() {
        this(true, 1000L, 400L);
    }

    public FileSystemWatcher(boolean daemon, long pollInterval, long quietPeriod) {
        Assert.isTrue((pollInterval > 0L ? 1 : 0) != 0, (String)"PollInterval must be positive");
        Assert.isTrue((quietPeriod > 0L ? 1 : 0) != 0, (String)"QuietPeriod must be positive");
        Assert.isTrue((pollInterval > quietPeriod ? 1 : 0) != 0, (String)"PollInterval must be greater than QuietPeriod");
        this.daemon = daemon;
        this.pollInterval = pollInterval;
        this.quietPeriod = quietPeriod;
    }

    public synchronized void addListener(FileChangeListener fileChangeListener) {
        Assert.notNull((Object)fileChangeListener, (String)"FileChangeListener must not be null");
        this.checkNotStarted();
        this.listeners.add(fileChangeListener);
    }

    public void addSourceFolders(Iterable<File> folders) {
        Assert.notNull(folders, (String)"Folders must not be null");
        for (File folder : folders) {
            this.addSourceFolder(folder);
        }
    }

    public synchronized void addSourceFolder(File folder) {
        Assert.notNull((Object)folder, (String)"Folder must not be null");
        Assert.isTrue((boolean)folder.isDirectory(), (String)"Folder must not be a file");
        this.checkNotStarted();
        this.folders.put(folder, null);
    }

    public synchronized void setTriggerFilter(FileFilter triggerFilter) {
        this.triggerFilter = triggerFilter;
    }

    private void checkNotStarted() {
        Assert.state((this.watchThread == null ? 1 : 0) != 0, (String)"FileSystemWatcher already started");
    }

    public synchronized void start() {
        this.saveInitialSnapshots();
        if (this.watchThread == null) {
            this.watchThread = new Thread(){

                @Override
                public void run() {
                    int remainingScans = FileSystemWatcher.this.remainingScans.get();
                    while (remainingScans > 0 || remainingScans == -1) {
                        try {
                            if (remainingScans > 0) {
                                FileSystemWatcher.this.remainingScans.decrementAndGet();
                            }
                            FileSystemWatcher.this.scan();
                        }
                        catch (InterruptedException interruptedException) {
                            // empty catch block
                        }
                        remainingScans = FileSystemWatcher.this.remainingScans.get();
                    }
                }
            };
            this.watchThread.setName("File Watcher");
            this.watchThread.setDaemon(this.daemon);
            this.remainingScans = new AtomicInteger(-1);
            this.watchThread.start();
        }
    }

    private void saveInitialSnapshots() {
        for (File folder : this.folders.keySet()) {
            this.folders.put(folder, new FolderSnapshot(folder));
        }
    }

    private void scan() throws InterruptedException {
        Map<File, FolderSnapshot> previous;
        Thread.sleep(this.pollInterval - this.quietPeriod);
        Map<File, FolderSnapshot> current = this.folders;
        do {
            previous = current;
            current = this.getCurrentSnapshots();
            Thread.sleep(this.quietPeriod);
        } while (this.isDifferent(previous, current));
        if (this.isDifferent(this.folders, current)) {
            this.updateSnapshots(current.values());
        }
    }

    private boolean isDifferent(Map<File, FolderSnapshot> previous, Map<File, FolderSnapshot> current) {
        if (!previous.keySet().equals(current.keySet())) {
            return true;
        }
        for (Map.Entry<File, FolderSnapshot> entry : previous.entrySet()) {
            FolderSnapshot currentFolder;
            FolderSnapshot previousFolder = entry.getValue();
            if (previousFolder.equals(currentFolder = current.get(entry.getKey()), this.triggerFilter)) continue;
            return true;
        }
        return false;
    }

    private Map<File, FolderSnapshot> getCurrentSnapshots() {
        LinkedHashMap<File, FolderSnapshot> snapshots = new LinkedHashMap<File, FolderSnapshot>();
        for (File folder : this.folders.keySet()) {
            snapshots.put(folder, new FolderSnapshot(folder));
        }
        return snapshots;
    }

    private void updateSnapshots(Collection<FolderSnapshot> snapshots) {
        LinkedHashMap<File, FolderSnapshot> updated = new LinkedHashMap<File, FolderSnapshot>();
        LinkedHashSet<ChangedFiles> changeSet = new LinkedHashSet<ChangedFiles>();
        for (FolderSnapshot snapshot : snapshots) {
            FolderSnapshot previous = this.folders.get(snapshot.getFolder());
            updated.put(snapshot.getFolder(), snapshot);
            ChangedFiles changedFiles = previous.getChangedFiles(snapshot, this.triggerFilter);
            if (changedFiles.getFiles().isEmpty()) continue;
            changeSet.add(changedFiles);
        }
        if (!changeSet.isEmpty()) {
            this.fireListeners(Collections.unmodifiableSet(changeSet));
        }
        this.folders = updated;
    }

    private void fireListeners(Set<ChangedFiles> changeSet) {
        for (FileChangeListener listener : this.listeners) {
            listener.onChange(changeSet);
        }
    }

    public synchronized void stop() {
        this.stopAfter(0);
    }

    synchronized void stopAfter(int remainingScans) {
        Thread thread = this.watchThread;
        if (thread != null) {
            this.remainingScans.set(remainingScans);
            if (remainingScans <= 0) {
                thread.interrupt();
            }
            if (Thread.currentThread() != thread) {
                try {
                    thread.join();
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
            }
            this.watchThread = null;
        }
    }
}

