/*
 * Copyright 2012-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.boot.http.converter.autoconfigure;

import java.util.Collection;

import org.jspecify.annotations.Nullable;

import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.HttpMessageConverters.ServerBuilder;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.http.converter.json.KotlinSerializationJsonHttpMessageConverter;

@SuppressWarnings("deprecation")
class DefaultServerHttpMessageConvertersCustomizer implements ServerHttpMessageConvertersCustomizer {

	private final @Nullable HttpMessageConverters legacyConverters;

	private final Collection<HttpMessageConverter<?>> converters;

	DefaultServerHttpMessageConvertersCustomizer(@Nullable HttpMessageConverters legacyConverters,
			Collection<HttpMessageConverter<?>> converters) {

		this.legacyConverters = legacyConverters;
		this.converters = converters;
	}

	@Override
	public void customize(ServerBuilder builder) {
		if (this.legacyConverters != null) {
			this.legacyConverters.forEach(builder::addCustomConverter);
		}
		else {
			builder.registerDefaults();
			this.converters.forEach((converter) -> {
				if (converter instanceof StringHttpMessageConverter) {
					builder.withStringConverter(converter);
				}
				else if (converter instanceof KotlinSerializationJsonHttpMessageConverter) {
					builder.withKotlinSerializationJsonConverter(converter);
				}
				else if (supportsMediaType(converter, MediaType.APPLICATION_JSON)) {
					builder.withJsonConverter(converter);
				}
				else if (supportsMediaType(converter, MediaType.APPLICATION_XML)) {
					builder.withXmlConverter(converter);
				}
				else {
					builder.addCustomConverter(converter);
				}
			});
		}
	}

	private static boolean supportsMediaType(HttpMessageConverter<?> converter, MediaType mediaType) {
		for (MediaType supportedMediaType : converter.getSupportedMediaTypes()) {
			if (supportedMediaType.equalsTypeAndSubtype(mediaType)) {
				return true;
			}
		}
		return false;
	}

}
