/*
 * Copyright 2016 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.stream.binding;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * A {@link Bindable} component that wraps a generic output binding target. Useful for
 * binding targets outside the {@link org.springframework.cloud.stream.annotation.Input}
 * and {@link org.springframework.cloud.stream.annotation.Output} annotated interfaces.
 *
 * @author Ilayaperumal Gopinathan
 * @author Marius Bogoevici
 */
public class SingleBindingTargetBindable<T> implements Bindable {

	private final String name;

	private final T bindingTarget;

	public SingleBindingTargetBindable(String name, T bindingTarget) {
		this.name = name;
		this.bindingTarget = bindingTarget;
	}

	@Override
	public void bindOutputs(BindingService bindingService) {
		bindingService.bindProducer(bindingTarget, name);
	}

	@Override
	public void unbindOutputs(BindingService bindingService) {
		bindingService.unbindProducers(name);
	}

	@Override
	public Set<String> getOutputs() {
		return Collections.unmodifiableSet(new HashSet<>(Arrays.asList(name)));
	}
}
