/*
 * Copyright 2020-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.cloud.stream.function;

import org.springframework.cloud.stream.binding.SupportedBindableFeatures;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.util.StringUtils;

/**
 *
 * @author Oleg Zhurakousky
 * @since 4.0.1
 *
 */
public class PollableSourceInitializer implements ApplicationContextInitializer<GenericApplicationContext> {

	@Override
	public void initialize(GenericApplicationContext applicationContext) {
		Environment environment = applicationContext.getEnvironment();
		if (StringUtils.hasText(environment.getProperty("spring.cloud.stream.pollable-source"))) {
			String[] sourceNames = environment.getProperty("spring.cloud.stream.pollable-source").split(";");

			for (String sourceName : sourceNames) {
				final SupportedBindableFeatures supportedBindableFeatures = new SupportedBindableFeatures();
				supportedBindableFeatures.setPollable(true);
				supportedBindableFeatures.setReactive(false);

				BindableFunctionProxyFactory proxyFactory =
					new BindableFunctionProxyFactory(sourceName, 1, 0, new StreamFunctionProperties(), supportedBindableFeatures);
				applicationContext.registerBean(sourceName + "_binding", BindableFunctionProxyFactory.class, () -> proxyFactory);
			}
		}
	}

}
