/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.mongodb.core.query;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.bson.Document;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.InvalidMongoDbApiUsageException;
import org.springframework.data.mongodb.core.query.Collation;
import org.springframework.data.mongodb.core.query.CriteriaDefinition;
import org.springframework.data.mongodb.core.query.Field;
import org.springframework.data.mongodb.core.query.Meta;
import org.springframework.data.mongodb.core.query.SerializationUtils;
import org.springframework.data.mongodb.util.BsonUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

public class Query {
    private static final String RESTRICTED_TYPES_KEY = "_$RESTRICTED_TYPES";
    private final Set<Class<?>> restrictedTypes = new HashSet();
    private final Map<String, CriteriaDefinition> criteria = new LinkedHashMap<String, CriteriaDefinition>();
    @Nullable
    private Field fieldSpec = null;
    private Sort sort = Sort.unsorted();
    private long skip;
    private int limit;
    @Nullable
    private String hint;
    private Meta meta = new Meta();
    private Optional<Collation> collation = Optional.empty();

    public static Query query(CriteriaDefinition criteriaDefinition) {
        return new Query(criteriaDefinition);
    }

    public Query() {
    }

    public Query(CriteriaDefinition criteriaDefinition) {
        this.addCriteria(criteriaDefinition);
    }

    public Query addCriteria(CriteriaDefinition criteriaDefinition) {
        CriteriaDefinition existing = this.criteria.get(criteriaDefinition.getKey());
        String key = criteriaDefinition.getKey();
        if (existing != null) {
            throw new InvalidMongoDbApiUsageException(String.format("Due to limitations of the com.mongodb.BasicDocument, you can't add a second '%s' criteria. Query already contains '%s'", key, SerializationUtils.serializeToJsonSafely(existing.getCriteriaObject())));
        }
        this.criteria.put(key, criteriaDefinition);
        return this;
    }

    public Field fields() {
        if (this.fieldSpec == null) {
            this.fieldSpec = new Field();
        }
        return this.fieldSpec;
    }

    public Query skip(long skip) {
        this.skip = skip;
        return this;
    }

    public Query limit(int limit) {
        this.limit = limit;
        return this;
    }

    @Deprecated
    public Query withHint(String hint) {
        Assert.hasText((String)hint, (String)"Hint must not be empty or null!");
        this.hint = hint;
        return this;
    }

    public Query withHint(Document hint) {
        Assert.notNull((Object)hint, (String)"Hint must not be null!");
        this.hint = hint.toJson();
        return this;
    }

    public Query with(Pageable pageable) {
        if (pageable.isUnpaged()) {
            return this;
        }
        this.limit = pageable.getPageSize();
        this.skip = pageable.getOffset();
        return this.with(pageable.getSort());
    }

    public Query with(Sort sort) {
        Assert.notNull((Object)sort, (String)"Sort must not be null!");
        if (sort.isUnsorted()) {
            return this;
        }
        sort.stream().filter(Sort.Order::isIgnoreCase).findFirst().ifPresent(it -> {
            throw new IllegalArgumentException(String.format("Given sort contained an Order for %s with ignore case! MongoDB does not support sorting ignoring case currently!", it.getProperty()));
        });
        this.sort = this.sort.and(sort);
        return this;
    }

    public Set<Class<?>> getRestrictedTypes() {
        return this.restrictedTypes;
    }

    public Query restrict(Class<?> type, Class<?> ... additionalTypes) {
        Assert.notNull(type, (String)"Type must not be null!");
        Assert.notNull(additionalTypes, (String)"AdditionalTypes must not be null");
        this.restrictedTypes.add(type);
        this.restrictedTypes.addAll(Arrays.asList(additionalTypes));
        return this;
    }

    public Document getQueryObject() {
        Document document = new Document();
        for (CriteriaDefinition definition : this.criteria.values()) {
            document.putAll((Map)definition.getCriteriaObject());
        }
        if (!this.restrictedTypes.isEmpty()) {
            document.put(RESTRICTED_TYPES_KEY, this.getRestrictedTypes());
        }
        return document;
    }

    public Document getFieldsObject() {
        return this.fieldSpec == null ? new Document() : this.fieldSpec.getFieldsObject();
    }

    public Document getSortObject() {
        if (this.sort.isUnsorted()) {
            return new Document();
        }
        Document document = new Document();
        this.sort.stream().forEach(order -> document.put(order.getProperty(), (Object)(order.isAscending() ? 1 : -1)));
        return document;
    }

    public long getSkip() {
        return this.skip;
    }

    public int getLimit() {
        return this.limit;
    }

    @Nullable
    @Deprecated
    public String getHint() {
        return this.hint;
    }

    public Query maxTimeMsec(long maxTimeMsec) {
        this.meta.setMaxTimeMsec(maxTimeMsec);
        return this;
    }

    @Deprecated
    public Query maxTime(long timeout, TimeUnit timeUnit) {
        this.meta.setMaxTime(timeout, timeUnit);
        return this;
    }

    public Query maxTime(Duration timeout) {
        this.meta.setMaxTime(timeout);
        return this;
    }

    @Deprecated
    public Query maxScan(long maxScan) {
        this.meta.setMaxScan(maxScan);
        return this;
    }

    public Query comment(String comment) {
        this.meta.setComment(comment);
        return this;
    }

    @Deprecated
    public Query useSnapshot() {
        this.meta.setSnapshot(true);
        return this;
    }

    public Query cursorBatchSize(int batchSize) {
        this.meta.setCursorBatchSize(batchSize);
        return this;
    }

    public Query noCursorTimeout() {
        this.meta.addFlag(Meta.CursorOption.NO_TIMEOUT);
        return this;
    }

    public Query exhaust() {
        this.meta.addFlag(Meta.CursorOption.EXHAUST);
        return this;
    }

    public Query slaveOk() {
        this.meta.addFlag(Meta.CursorOption.SLAVE_OK);
        return this;
    }

    public Query partialResults() {
        this.meta.addFlag(Meta.CursorOption.PARTIAL);
        return this;
    }

    public Meta getMeta() {
        return this.meta;
    }

    public void setMeta(Meta meta) {
        Assert.notNull((Object)meta, (String)"Query meta might be empty but must not be null.");
        this.meta = meta;
    }

    public Query collation(@Nullable Collation collation) {
        this.collation = Optional.ofNullable(collation);
        return this;
    }

    public Optional<Collation> getCollation() {
        return this.collation;
    }

    protected List<CriteriaDefinition> getCriteria() {
        return new ArrayList<CriteriaDefinition>(this.criteria.values());
    }

    public static Query of(Query source) {
        Assert.notNull((Object)source, (String)"Source must not be null!");
        final Document sourceFields = source.getFieldsObject();
        final Document sourceSort = source.getSortObject();
        final Document sourceQuery = source.getQueryObject();
        Query target = new Query(){

            @Override
            public Document getFieldsObject() {
                return BsonUtils.merge(sourceFields, super.getFieldsObject());
            }

            @Override
            public Document getSortObject() {
                return BsonUtils.merge(sourceSort, super.getSortObject());
            }

            @Override
            public Document getQueryObject() {
                return BsonUtils.merge(sourceQuery, super.getQueryObject());
            }
        };
        target.criteria.putAll(source.criteria);
        target.skip = source.skip;
        target.limit = source.limit;
        target.sort = Sort.unsorted().and(source.sort);
        target.hint = source.hint;
        target.collation = source.collation;
        target.restrictedTypes.addAll(source.restrictedTypes);
        if (source.getMeta().hasValues()) {
            target.setMeta(new Meta(source.getMeta()));
        }
        return target;
    }

    public String toString() {
        return String.format("Query: %s, Fields: %s, Sort: %s", SerializationUtils.serializeToJsonSafely(this.getQueryObject()), SerializationUtils.serializeToJsonSafely(this.getFieldsObject()), SerializationUtils.serializeToJsonSafely(this.getSortObject()));
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || !this.getClass().equals(obj.getClass())) {
            return false;
        }
        return this.querySettingsEquals((Query)obj);
    }

    protected boolean querySettingsEquals(Query that) {
        boolean criteriaEqual = this.criteria.equals(that.criteria);
        boolean fieldsEqual = ObjectUtils.nullSafeEquals((Object)this.fieldSpec, (Object)that.fieldSpec);
        boolean sortEqual = this.sort.equals((Object)that.sort);
        boolean hintEqual = ObjectUtils.nullSafeEquals((Object)this.hint, (Object)that.hint);
        boolean skipEqual = this.skip == that.skip;
        boolean limitEqual = this.limit == that.limit;
        boolean metaEqual = ObjectUtils.nullSafeEquals((Object)this.meta, (Object)that.meta);
        boolean collationEqual = ObjectUtils.nullSafeEquals(this.collation.orElse(null), that.collation.orElse(null));
        return criteriaEqual && fieldsEqual && sortEqual && hintEqual && skipEqual && limitEqual && metaEqual && collationEqual;
    }

    public int hashCode() {
        int result = 17;
        result += 31 * this.criteria.hashCode();
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.fieldSpec);
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.sort);
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.hint);
        result = (int)((long)result + 31L * this.skip);
        result += 31 * this.limit;
        result += 31 * ObjectUtils.nullSafeHashCode((Object)this.meta);
        return result += 31 * ObjectUtils.nullSafeHashCode(this.collation.orElse(null));
    }

    @Deprecated
    public static boolean isRestrictedTypeKey(String key) {
        return RESTRICTED_TYPES_KEY.equals(key);
    }
}

