/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.neo4j.examples.movies;

import java.util.Arrays;
import java.util.Calendar;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.TimeZone;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Transaction;
import org.neo4j.ogm.cypher.Filter;
import org.neo4j.ogm.session.Session;
import org.neo4j.ogm.testutil.GraphTestUtils;
import org.neo4j.ogm.testutil.MultiDriverTestClass;
import org.neo4j.tooling.GlobalGraphOperations;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataAccessException;
import org.springframework.data.neo4j.examples.movies.context.MoviesContext;
import org.springframework.data.neo4j.examples.movies.domain.Actor;
import org.springframework.data.neo4j.examples.movies.domain.Cinema;
import org.springframework.data.neo4j.examples.movies.domain.Genre;
import org.springframework.data.neo4j.examples.movies.domain.Movie;
import org.springframework.data.neo4j.examples.movies.domain.Rating;
import org.springframework.data.neo4j.examples.movies.domain.ReleasedMovie;
import org.springframework.data.neo4j.examples.movies.domain.TempMovie;
import org.springframework.data.neo4j.examples.movies.domain.User;
import org.springframework.data.neo4j.examples.movies.repo.AbstractAnnotatedEntityRepository;
import org.springframework.data.neo4j.examples.movies.repo.AbstractEntityRepository;
import org.springframework.data.neo4j.examples.movies.repo.ActorRepository;
import org.springframework.data.neo4j.examples.movies.repo.CinemaRepository;
import org.springframework.data.neo4j.examples.movies.repo.RatingRepository;
import org.springframework.data.neo4j.examples.movies.repo.TempMovieRepository;
import org.springframework.data.neo4j.examples.movies.repo.UserRepository;
import org.springframework.data.neo4j.examples.movies.service.UserService;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.SpringJUnit4ClassRunner;

@ContextConfiguration(classes={MoviesContext.class})
@RunWith(value=SpringJUnit4ClassRunner.class)
@DirtiesContext
public class MoviesIntegrationTest
extends MultiDriverTestClass {
    private final Logger logger = LoggerFactory.getLogger(MoviesIntegrationTest.class);
    @Autowired
    private Session session;
    @Autowired
    private UserRepository userRepository;
    @Autowired
    private UserService userService;
    @Autowired
    private CinemaRepository cinemaRepository;
    @Autowired
    private AbstractAnnotatedEntityRepository abstractAnnotatedEntityRepository;
    @Autowired
    private AbstractEntityRepository abstractEntityRepository;
    @Autowired
    private TempMovieRepository tempMovieRepository;
    @Autowired
    private ActorRepository actorRepository;
    @Autowired
    private RatingRepository ratingRepository;

    @Before
    public void clear() {
        this.session.clear();
        this.session.purgeDatabase();
    }

    @Test
    public void shouldSaveUser() {
        User user = new User("Michal");
        this.userRepository.save(user);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person {name:'Michal'})");
    }

    @Test
    public void shouldSaveUserWithoutName() {
        User user = new User();
        this.userRepository.save(user);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person)");
    }

    @Test
    public void shouldSaveReleasedMovie() {
        Calendar cinemaReleaseDate = this.createDate(1994, 8, 10, "GMT");
        Calendar cannesReleaseDate = this.createDate(1994, 4, 12, "GMT");
        ReleasedMovie releasedMovie = new ReleasedMovie("Pulp Fiction", cinemaReleaseDate.getTime(), cannesReleaseDate.getTime());
        this.abstractAnnotatedEntityRepository.save(releasedMovie);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:ReleasedMovie:AbstractAnnotatedEntity {cinemaRelease:'1994-09-10T00:00:00.000Z',cannesRelease:768700800000,title:'Pulp Fiction'})");
    }

    @Test
    public void shouldSaveReleasedMovie2() {
        Calendar cannesReleaseDate = this.createDate(1994, 4, 12, "GMT");
        ReleasedMovie releasedMovie = new ReleasedMovie("Pulp Fiction", null, cannesReleaseDate.getTime());
        this.abstractAnnotatedEntityRepository.save(releasedMovie);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:ReleasedMovie:AbstractAnnotatedEntity {cannesRelease:768700800000,title:'Pulp Fiction'})");
    }

    @Test
    public void shouldSaveMovie() {
        Movie movie = new Movie("Pulp Fiction");
        movie.setTags(new String[]{"cool", "classic"});
        movie.setImage(new byte[]{1, 2, 3});
        this.abstractEntityRepository.save(movie);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:Movie {name:'Pulp Fiction', tags:['cool','classic'], image:'AQID'})");
    }

    @Test
    public void shouldSaveUsers() {
        HashSet<User> set = new HashSet<User>();
        set.add(new User("Michal"));
        set.add(new User("Adam"));
        set.add(new User("Vince"));
        this.userRepository.save(set);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (:User:Person {name:'Michal'}),(:User:Person {name:'Vince'}),(:User:Person {name:'Adam'})");
        Assert.assertEquals((long)3L, (long)this.userRepository.count());
    }

    @Test
    public void shouldSaveUsers2() {
        LinkedList<User> list = new LinkedList<User>();
        list.add(new User("Michal"));
        list.add(new User("Adam"));
        list.add(new User("Vince"));
        this.userRepository.save(list);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (:User:Person {name:'Michal'}),(:User:Person {name:'Vince'}),(:User:Person {name:'Adam'})");
        Assert.assertEquals((long)3L, (long)this.userRepository.count());
    }

    @Test
    public void shouldUpdateUserUsingRepository() {
        User user = (User)this.userRepository.save(new User("Michal"));
        user.setName("Adam");
        this.userRepository.save(user);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person {name:'Adam'})");
    }

    @Test
    @Ignore
    public void shouldUpdateUserUsingTransactionalService() {
        User user = new User("Michal");
        this.userRepository.save(user);
        this.userService.updateUser(user, "Adam");
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User {name:'Adam'})");
    }

    @Test
    public void shouldFindUser() {
        User user = new User("Michal");
        this.userRepository.save(user);
        User loaded = (User)this.userRepository.findOne(user.getId());
        Assert.assertEquals((Object)"Michal", (Object)loaded.getName());
        Assert.assertTrue((boolean)loaded.equals(user));
        Assert.assertTrue((loaded == user ? 1 : 0) != 0);
    }

    @Test
    public void shouldFindActorByNumericValueOfStringProperty() {
        Actor actor = new Actor("1", "Tom Hanks");
        this.actorRepository.save(actor);
        Assert.assertNotNull(this.findByProperty(Actor.class, "id", "1").iterator().next());
    }

    @Test
    @Ignore
    public void shouldFindUserWithoutName() {
        User user = new User();
        this.userRepository.save(user);
        User loaded = (User)this.userRepository.findOne(user.getId());
        Assert.assertNull((Object)loaded.getName());
        Assert.assertTrue((boolean)loaded.equals(user));
        Assert.assertTrue((loaded == user ? 1 : 0) != 0);
    }

    @Test
    public void shouldDeleteUser() {
        User user = new User("Michal");
        this.userRepository.save(user);
        this.userRepository.delete(user);
        Assert.assertFalse((boolean)this.userRepository.findAll().iterator().hasNext());
        Assert.assertFalse((boolean)this.userRepository.findAll(1).iterator().hasNext());
        Assert.assertFalse((boolean)this.userRepository.exists(user.getId()));
        Assert.assertEquals((long)0L, (long)this.userRepository.count());
        Assert.assertNull((Object)this.userRepository.findOne(user.getId()));
        Assert.assertNull((Object)this.userRepository.findOne(user.getId(), 10));
        try (Transaction tx = MoviesIntegrationTest.getGraphDatabaseService().beginTx();){
            Assert.assertFalse((boolean)GlobalGraphOperations.at((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService()).getAllNodes().iterator().hasNext());
            tx.success();
        }
    }

    @Test
    public void shouldHandleMultipleConcurrentRequests() throws InterruptedException, Neo4jFailedToStartException {
        ExecutorService executor = Executors.newFixedThreadPool(10);
        final CountDownLatch latch = new CountDownLatch(100);
        for (int i = 0; i < 100; ++i) {
            executor.submit(new Runnable(){

                @Override
                public void run() {
                    MoviesIntegrationTest.this.userRepository.save(new User());
                    latch.countDown();
                }
            });
        }
        latch.await();
        System.out.println("all threads joined");
        executor.shutdown();
        Assert.assertEquals((long)100L, (long)this.userRepository.count());
    }

    @Test(expected=DataAccessException.class)
    public void shouldInterceptOGMExceptions() {
        this.ratingRepository.findAll(0);
    }

    @Test
    public void shouldSaveUserAndNewGenre() {
        User user = new User("Michal");
        user.interestedIn(new Genre("Drama"));
        this.userRepository.save(user);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person {name:'Michal'})-[:INTERESTED]->(g:Genre {name:'Drama'})");
    }

    @Test
    public void shouldSaveUserAndNewGenres() {
        User user = new User("Michal");
        user.interestedIn(new Genre("Drama"));
        user.interestedIn(new Genre("Historical"));
        user.interestedIn(new Genre("Thriller"));
        this.userRepository.save(user);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person {name:'Michal'}),(g1:Genre {name:'Drama'}),(g2:Genre {name:'Historical'}),(g3:Genre {name:'Thriller'}),(u)-[:INTERESTED]->(g1),(u)-[:INTERESTED]->(g2),(u)-[:INTERESTED]->(g3)");
    }

    @Test
    public void shouldSaveUserAndNewGenre2() {
        User user = new User("Michal");
        user.interestedIn(new Genre("Drama"));
        this.userRepository.save(user, 1);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person {name:'Michal'})-[:INTERESTED]->(g:Genre {name:'Drama'})");
    }

    @Test
    public void shouldSaveUserAndExistingGenre() {
        User michal = new User("Michal");
        Genre drama = new Genre("Drama");
        michal.interestedIn(drama);
        this.userRepository.save(michal);
        User vince = new User("Vince");
        vince.interestedIn(drama);
        this.userRepository.save(vince);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User:Person {name:'Michal'}),(v:User:Person {name:'Vince'}),(g:Genre {name:'Drama'}),(m)-[:INTERESTED]->(g),(v)-[:INTERESTED]->(g)");
    }

    @Test
    public void shouldSaveUserButNotGenre() {
        User user = new User("Michal");
        user.interestedIn(new Genre("Drama"));
        this.userRepository.save(user, 0);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person {name:'Michal'})");
    }

    @Test
    public void shouldUpdateGenreWhenSavedThroughUser() {
        User michal = new User("Michal");
        Genre drama = new Genre("Drama");
        michal.interestedIn(drama);
        this.userRepository.save(michal);
        drama.setName("New Drama");
        this.userRepository.save(michal);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User:Person {name:'Michal'}),(g:Genre {name:'New Drama'}),(m)-[:INTERESTED]->(g)");
    }

    @Test
    public void shouldRemoveGenreFromUser() {
        User michal = new User("Michal");
        Genre drama = new Genre("Drama");
        michal.interestedIn(drama);
        this.userRepository.save(michal);
        michal.notInterestedIn(drama);
        this.userRepository.save(michal);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User:Person {name:'Michal'}),(g:Genre {name:'Drama'})");
    }

    @Test
    public void shouldRemoveGenreFromUserUsingService() {
        User michal = new User("Michal");
        Genre drama = new Genre("Drama");
        michal.interestedIn(drama);
        this.userRepository.save(michal);
        this.userService.notInterestedIn(michal.getId(), drama.getId());
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User:Person {name:'Michal'}),(g:Genre {name:'Drama'})");
    }

    @Test
    public void shouldAddNewVisitorToCinema() {
        Cinema cinema = new Cinema("Odeon");
        cinema.addVisitor(new User("Michal"));
        this.cinemaRepository.save(cinema);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User:Person {name:'Michal'}),(c:Theatre {name:'Odeon', capacity:0}),(m)-[:VISITED]->(c)");
    }

    @Test
    public void shouldAddExistingVisitorToCinema() {
        User michal = new User("Michal");
        this.userRepository.save(michal);
        Cinema cinema = new Cinema("Odeon");
        cinema.addVisitor(michal);
        this.cinemaRepository.save(cinema);
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User:Person {name:'Michal'}),(c:Theatre {name:'Odeon', capacity:0}),(m)-[:VISITED]->(c)");
    }

    @Test
    public void shouldBefriendPeople() {
        User michal = new User("Michal");
        michal.befriend(new User("Adam"));
        this.userRepository.save(michal);
        try {
            GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User {name:'Michal'})-[:FRIEND_OF]->(a:User:Person {name:'Adam'})");
        }
        catch (AssertionError error) {
            GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (m:User:Person {name:'Michal'})<-[:FRIEND_OF]-(a:User:Person {name:'Adam'})");
        }
    }

    @Test
    public void shouldLoadOutgoingFriendsWhenUndirected() {
        MoviesIntegrationTest.getGraphDatabaseService().execute("CREATE (m:User {name:'Michal'})-[:FRIEND_OF]->(a:User {name:'Adam'})");
        User michal = (User)this.findByProperty(User.class, "name", "Michal").iterator().next();
        Assert.assertEquals((long)1L, (long)michal.getFriends().size());
        User adam = michal.getFriends().iterator().next();
        Assert.assertEquals((Object)"Adam", (Object)adam.getName());
        Assert.assertEquals((long)1L, (long)adam.getFriends().size());
        Assert.assertTrue((michal == adam.getFriends().iterator().next() ? 1 : 0) != 0);
        Assert.assertTrue((boolean)michal.equals(adam.getFriends().iterator().next()));
    }

    @Test
    public void shouldLoadIncomingFriendsWhenUndirected() {
        MoviesIntegrationTest.getGraphDatabaseService().execute("CREATE (m:User {name:'Michal'})<-[:FRIEND_OF]-(a:User {name:'Adam'})");
        User michal = (User)this.findByProperty(User.class, "name", "Michal").iterator().next();
        Assert.assertEquals((long)1L, (long)michal.getFriends().size());
        User adam = michal.getFriends().iterator().next();
        Assert.assertEquals((Object)"Adam", (Object)adam.getName());
        Assert.assertEquals((long)1L, (long)adam.getFriends().size());
        Assert.assertTrue((michal == adam.getFriends().iterator().next() ? 1 : 0) != 0);
        Assert.assertTrue((boolean)michal.equals(adam.getFriends().iterator().next()));
    }

    @Test
    public void shouldSaveNewUserAndNewMovieWithRatings() {
        User user = new User("Michal");
        TempMovie movie = new TempMovie("Pulp Fiction");
        user.rate(movie, 5, "Best movie ever");
        this.userRepository.save(user);
        User michal = (User)this.findByProperty(User.class, "name", "Michal").iterator().next();
        GraphTestUtils.assertSameGraph((GraphDatabaseService)MoviesIntegrationTest.getGraphDatabaseService(), (String)"CREATE (u:User:Person {name:'Michal'})-[:RATED {stars:5, comment:'Best movie ever', ratingTimestamp:0}]->(m:Movie {name:'Pulp Fiction'})");
    }

    @Test
    public void shouldSaveNewUserRatingsForAnExistingMovie() {
        TempMovie movie = new TempMovie("Pulp Fiction");
        movie = (TempMovie)this.tempMovieRepository.save(movie);
        User user = new User("Michal");
        user.rate(movie, 5, "Best movie ever");
        this.userRepository.save(user);
        TempMovie tempMovie = (TempMovie)this.findByProperty(TempMovie.class, "name", "Pulp Fiction").iterator().next();
        Assert.assertEquals((long)1L, (long)tempMovie.getRatings().size());
    }

    @Test
    public void findOneShouldConsiderTheEntityType() {
        TempMovie movie = new TempMovie("Pulp Fiction");
        movie = (TempMovie)this.tempMovieRepository.save(movie);
        User user = new User("Michal");
        user.rate(movie, 5, "Best movie ever");
        this.userRepository.save(user);
        Assert.assertEquals((Object)movie.getName(), (Object)((TempMovie)this.tempMovieRepository.findOne(movie.getId())).getName());
        Assert.assertEquals((Object)user.getName(), (Object)((User)this.userRepository.findOne(user.getId())).getName());
        Assert.assertEquals((long)5L, (long)((Rating)this.ratingRepository.findOne(user.getRatings().iterator().next().getId())).getStars());
        Assert.assertNull((Object)this.tempMovieRepository.findOne(user.getId()));
        Assert.assertNull((Object)this.userRepository.findOne(movie.getId(), 0));
        Assert.assertNull((Object)this.ratingRepository.findOne(user.getId()));
    }

    @Test
    public void shouldSaveAndReturnManyEntities() {
        User michal = new User("Michal");
        User adam = new User("Adam");
        User daniela = new User("Daniela");
        List<User> users = Arrays.asList(michal, adam, daniela);
        Iterable savedUsers = this.userRepository.save(users);
        for (User user : savedUsers) {
            Assert.assertNotNull((Object)user.getId());
        }
    }

    private Calendar createDate(int y, int m, int d, String tz) {
        Calendar calendar = Calendar.getInstance();
        calendar.set(y, m, d);
        calendar.setTimeZone(TimeZone.getTimeZone(tz));
        calendar.set(11, 0);
        calendar.set(12, 0);
        calendar.set(13, 0);
        calendar.set(14, 0);
        return calendar;
    }

    private void waitForNeo4jToStart(long maxTimeToWait) throws Neo4jFailedToStartException {
        Transaction transaction;
        long startTime = System.currentTimeMillis();
        while ((transaction = MoviesIntegrationTest.getGraphDatabaseService().beginTx()) == null && System.currentTimeMillis() - startTime <= maxTimeToWait) {
        }
        if (transaction == null) {
            throw new Neo4jFailedToStartException(maxTimeToWait);
        }
        transaction.close();
    }

    protected Iterable<?> findByProperty(Class clazz, String propertyName, Object propertyValue) {
        return this.session.loadAll(clazz, new Filter(propertyName, propertyValue));
    }

    protected Iterable<?> findByProperty(Class clazz, String propertyName, Object propertyValue, int depth) {
        return this.session.loadAll(clazz, new Filter(propertyName, propertyValue), depth);
    }

    private static class Neo4jFailedToStartException
    extends Exception {
        private Neo4jFailedToStartException(long timeoutValue) {
            super(String.format("Could not start neo4j instance in [%d] ms", timeoutValue));
        }
    }
}

