/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.io.buffer;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.CompletionHandler;
import java.nio.channels.ReadableByteChannel;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import org.reactivestreams.Publisher;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.core.io.buffer.PooledDataBuffer;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxSink;
import reactor.core.publisher.SynchronousSink;

public abstract class DataBufferUtils {
    public static Flux<DataBuffer> read(InputStream inputStream, DataBufferFactory dataBufferFactory, int bufferSize) {
        Assert.notNull((Object)inputStream, "InputStream must not be null");
        Assert.notNull((Object)dataBufferFactory, "DataBufferFactory must not be null");
        ReadableByteChannel channel = Channels.newChannel(inputStream);
        return DataBufferUtils.read(channel, dataBufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(ReadableByteChannel channel, DataBufferFactory dataBufferFactory, int bufferSize) {
        Assert.notNull((Object)channel, "ReadableByteChannel must not be null");
        Assert.notNull((Object)dataBufferFactory, "DataBufferFactory must not be null");
        return Flux.generate(() -> channel, (BiFunction)new ReadableByteChannelGenerator(dataBufferFactory, bufferSize), DataBufferUtils::closeChannel);
    }

    public static Flux<DataBuffer> read(AsynchronousFileChannel channel, DataBufferFactory dataBufferFactory, int bufferSize) {
        return DataBufferUtils.read(channel, 0L, dataBufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(AsynchronousFileChannel channel, long position, DataBufferFactory dataBufferFactory, int bufferSize) {
        Assert.notNull((Object)channel, "'channel' must not be null");
        Assert.notNull((Object)dataBufferFactory, "'dataBufferFactory' must not be null");
        Assert.isTrue(position >= 0L, "'position' must be >= 0");
        ByteBuffer byteBuffer = ByteBuffer.allocate(bufferSize);
        return Flux.create(emitter -> {
            emitter.onDispose(() -> DataBufferUtils.closeChannel(channel));
            AsynchronousFileChannelCompletionHandler completionHandler = new AsynchronousFileChannelCompletionHandler((FluxSink)emitter, position, dataBufferFactory, byteBuffer);
            channel.read(byteBuffer, position, channel, completionHandler);
        });
    }

    private static void closeChannel(@Nullable Channel channel) {
        try {
            if (channel != null) {
                channel.close();
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static Flux<DataBuffer> takeUntilByteCount(Publisher<DataBuffer> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be a positive number");
        AtomicLong byteCountDown = new AtomicLong(maxByteCount);
        return Flux.from(publisher).takeWhile(dataBuffer -> {
            int delta = -dataBuffer.readableByteCount();
            long currentCount = byteCountDown.getAndAdd(delta);
            return currentCount >= 0L;
        }).map(dataBuffer -> {
            long currentCount = byteCountDown.get();
            if (currentCount >= 0L) {
                return dataBuffer;
            }
            int size = (int)(currentCount + (long)dataBuffer.readableByteCount());
            return dataBuffer.slice(0, size);
        });
    }

    public static Flux<DataBuffer> skipUntilByteCount(Publisher<DataBuffer> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be a positive number");
        AtomicLong byteCountDown = new AtomicLong(maxByteCount);
        return Flux.from(publisher).skipUntil(dataBuffer -> {
            int delta = -dataBuffer.readableByteCount();
            long currentCount = byteCountDown.addAndGet(delta);
            if (currentCount < 0L) {
                return true;
            }
            DataBufferUtils.release(dataBuffer);
            return false;
        }).map(dataBuffer -> {
            long currentCount = byteCountDown.get();
            if (currentCount < 0L) {
                int skip = (int)(currentCount + (long)dataBuffer.readableByteCount());
                byteCountDown.set(0L);
                return dataBuffer.slice(skip, dataBuffer.readableByteCount() - skip);
            }
            return dataBuffer;
        });
    }

    public static <T extends DataBuffer> T retain(T dataBuffer) {
        if (dataBuffer instanceof PooledDataBuffer) {
            return (T)((PooledDataBuffer)dataBuffer).retain();
        }
        return dataBuffer;
    }

    public static boolean release(DataBuffer dataBuffer) {
        if (dataBuffer instanceof PooledDataBuffer) {
            return ((PooledDataBuffer)dataBuffer).release();
        }
        return false;
    }

    private static class AsynchronousFileChannelCompletionHandler
    implements CompletionHandler<Integer, AsynchronousFileChannel> {
        private final FluxSink<DataBuffer> emitter;
        private final ByteBuffer byteBuffer;
        private final DataBufferFactory dataBufferFactory;
        private long position;

        private AsynchronousFileChannelCompletionHandler(FluxSink<DataBuffer> emitter, long position, DataBufferFactory dataBufferFactory, ByteBuffer byteBuffer) {
            this.emitter = emitter;
            this.position = position;
            this.dataBufferFactory = dataBufferFactory;
            this.byteBuffer = byteBuffer;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void completed(Integer read, AsynchronousFileChannel channel) {
            if (read != -1) {
                this.position += (long)read.intValue();
                this.byteBuffer.flip();
                boolean release = true;
                DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(read);
                try {
                    dataBuffer.write(this.byteBuffer);
                    release = false;
                    this.emitter.next((Object)dataBuffer);
                }
                finally {
                    if (release) {
                        DataBufferUtils.release(dataBuffer);
                    }
                }
                this.byteBuffer.clear();
                if (!this.emitter.isCancelled()) {
                    channel.read(this.byteBuffer, this.position, channel, this);
                }
            } else {
                this.emitter.complete();
                DataBufferUtils.closeChannel(channel);
            }
        }

        @Override
        public void failed(Throwable exc, AsynchronousFileChannel channel) {
            this.emitter.error(exc);
            DataBufferUtils.closeChannel(channel);
        }
    }

    private static class ReadableByteChannelGenerator
    implements BiFunction<ReadableByteChannel, SynchronousSink<DataBuffer>, ReadableByteChannel> {
        private final DataBufferFactory dataBufferFactory;
        private final ByteBuffer byteBuffer;

        public ReadableByteChannelGenerator(DataBufferFactory dataBufferFactory, int chunkSize) {
            this.dataBufferFactory = dataBufferFactory;
            this.byteBuffer = ByteBuffer.allocate(chunkSize);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public ReadableByteChannel apply(ReadableByteChannel channel, SynchronousSink<DataBuffer> sub) {
            block8: {
                try {
                    int read = channel.read(this.byteBuffer);
                    if (read >= 0) {
                        this.byteBuffer.flip();
                        boolean release = true;
                        DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(read);
                        try {
                            dataBuffer.write(this.byteBuffer);
                            release = false;
                            sub.next((Object)dataBuffer);
                        }
                        finally {
                            if (release) {
                                DataBufferUtils.release(dataBuffer);
                            }
                        }
                        this.byteBuffer.clear();
                        break block8;
                    }
                    sub.complete();
                }
                catch (IOException ex) {
                    sub.error((Throwable)ex);
                }
            }
            return channel;
        }
    }
}

