/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.core.io.buffer;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.channels.Channel;
import java.nio.channels.Channels;
import java.nio.channels.CompletionHandler;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.file.StandardOpenOption;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscription;
import org.springframework.core.io.Resource;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.DataBufferFactory;
import org.springframework.core.io.buffer.PooledDataBuffer;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import reactor.core.CoreSubscriber;
import reactor.core.publisher.BaseSubscriber;
import reactor.core.publisher.Flux;
import reactor.core.publisher.FluxSink;
import reactor.core.publisher.SynchronousSink;

public abstract class DataBufferUtils {
    private static final Consumer<DataBuffer> RELEASE_CONSUMER = DataBufferUtils::release;

    public static Flux<DataBuffer> read(InputStream inputStream, DataBufferFactory dataBufferFactory, int bufferSize) {
        Assert.notNull((Object)inputStream, "InputStream must not be null");
        Assert.notNull((Object)dataBufferFactory, "DataBufferFactory must not be null");
        ReadableByteChannel channel = Channels.newChannel(inputStream);
        return DataBufferUtils.read(channel, dataBufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(ReadableByteChannel channel, DataBufferFactory dataBufferFactory, int bufferSize) {
        Assert.notNull((Object)channel, "ReadableByteChannel must not be null");
        Assert.notNull((Object)dataBufferFactory, "DataBufferFactory must not be null");
        return Flux.generate(() -> channel, (BiFunction)new ReadableByteChannelGenerator(dataBufferFactory, bufferSize), DataBufferUtils::closeChannel);
    }

    public static Flux<DataBuffer> read(AsynchronousFileChannel channel, DataBufferFactory dataBufferFactory, int bufferSize) {
        return DataBufferUtils.read(channel, 0L, dataBufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(AsynchronousFileChannel channel, long position, DataBufferFactory dataBufferFactory, int bufferSize) {
        Assert.notNull((Object)channel, "'channel' must not be null");
        Assert.notNull((Object)dataBufferFactory, "'dataBufferFactory' must not be null");
        Assert.isTrue(position >= 0L, "'position' must be >= 0");
        ByteBuffer byteBuffer = ByteBuffer.allocate(bufferSize);
        return Flux.create(sink -> {
            sink.onDispose(() -> DataBufferUtils.closeChannel(channel));
            AsynchronousFileChannelReadCompletionHandler completionHandler = new AsynchronousFileChannelReadCompletionHandler((FluxSink)sink, position, dataBufferFactory, byteBuffer);
            channel.read(byteBuffer, position, channel, completionHandler);
        });
    }

    public static Flux<DataBuffer> read(Resource resource, DataBufferFactory dataBufferFactory, int bufferSize) {
        return DataBufferUtils.read(resource, 0L, dataBufferFactory, bufferSize);
    }

    public static Flux<DataBuffer> read(Resource resource, long position, DataBufferFactory dataBufferFactory, int bufferSize) {
        try {
            if (resource.isFile()) {
                File file = resource.getFile();
                AsynchronousFileChannel channel = AsynchronousFileChannel.open(file.toPath(), StandardOpenOption.READ);
                return DataBufferUtils.read(channel, position, dataBufferFactory, bufferSize);
            }
        }
        catch (IOException file) {
            // empty catch block
        }
        try {
            ReadableByteChannel channel = resource.readableChannel();
            Flux<DataBuffer> in = DataBufferUtils.read(channel, dataBufferFactory, bufferSize);
            return DataBufferUtils.skipUntilByteCount(in, position);
        }
        catch (IOException ex) {
            return Flux.error((Throwable)ex);
        }
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source, OutputStream outputStream) {
        Assert.notNull(source, "'source' must not be null");
        Assert.notNull((Object)outputStream, "'outputStream' must not be null");
        WritableByteChannel channel = Channels.newChannel(outputStream);
        return DataBufferUtils.write(source, channel);
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source, WritableByteChannel channel) {
        Assert.notNull(source, "'source' must not be null");
        Assert.notNull((Object)channel, "'channel' must not be null");
        Flux flux = Flux.from(source);
        return Flux.create(sink -> flux.subscribe(dataBuffer -> {
            try {
                ByteBuffer byteBuffer = dataBuffer.asByteBuffer();
                while (byteBuffer.hasRemaining()) {
                    channel.write(byteBuffer);
                }
                sink.next(dataBuffer);
            }
            catch (IOException ex) {
                sink.error((Throwable)ex);
            }
        }, arg_0 -> ((FluxSink)sink).error(arg_0), () -> ((FluxSink)sink).complete()));
    }

    public static Flux<DataBuffer> write(Publisher<DataBuffer> source, AsynchronousFileChannel channel, long position) {
        Assert.notNull(source, "'source' must not be null");
        Assert.notNull((Object)channel, "'channel' must not be null");
        Assert.isTrue(position >= 0L, "'position' must be >= 0");
        Flux flux = Flux.from(source);
        return Flux.create(sink -> {
            AsynchronousFileChannelWriteCompletionHandler subscriber = new AsynchronousFileChannelWriteCompletionHandler((FluxSink<DataBuffer>)sink, channel, position);
            flux.subscribe((CoreSubscriber)subscriber);
        });
    }

    private static void closeChannel(@Nullable Channel channel) {
        try {
            if (channel != null) {
                channel.close();
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static Flux<DataBuffer> takeUntilByteCount(Publisher<DataBuffer> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be a positive number");
        AtomicLong byteCountDown = new AtomicLong(maxByteCount);
        return Flux.from(publisher).takeWhile(dataBuffer -> {
            int delta = -dataBuffer.readableByteCount();
            long currentCount = byteCountDown.getAndAdd(delta);
            return currentCount >= 0L;
        }).map(dataBuffer -> {
            long currentCount = byteCountDown.get();
            if (currentCount >= 0L) {
                return dataBuffer;
            }
            int size = (int)(currentCount + (long)dataBuffer.readableByteCount());
            return dataBuffer.slice(0, size);
        });
    }

    public static Flux<DataBuffer> skipUntilByteCount(Publisher<DataBuffer> publisher, long maxByteCount) {
        Assert.notNull(publisher, "Publisher must not be null");
        Assert.isTrue(maxByteCount >= 0L, "'maxByteCount' must be a positive number");
        AtomicLong byteCountDown = new AtomicLong(maxByteCount);
        return Flux.from(publisher).skipUntil(dataBuffer -> {
            int delta = -dataBuffer.readableByteCount();
            long currentCount = byteCountDown.addAndGet(delta);
            if (currentCount < 0L) {
                return true;
            }
            DataBufferUtils.release(dataBuffer);
            return false;
        }).map(dataBuffer -> {
            long currentCount = byteCountDown.get();
            if (currentCount < 0L) {
                int skip = (int)(currentCount + (long)dataBuffer.readableByteCount());
                byteCountDown.set(0L);
                return dataBuffer.slice(skip, dataBuffer.readableByteCount() - skip);
            }
            return dataBuffer;
        });
    }

    public static <T extends DataBuffer> T retain(T dataBuffer) {
        if (dataBuffer instanceof PooledDataBuffer) {
            return (T)((PooledDataBuffer)dataBuffer).retain();
        }
        return dataBuffer;
    }

    public static boolean release(@Nullable DataBuffer dataBuffer) {
        if (dataBuffer instanceof PooledDataBuffer) {
            return ((PooledDataBuffer)dataBuffer).release();
        }
        return false;
    }

    public static Consumer<DataBuffer> releaseConsumer() {
        return RELEASE_CONSUMER;
    }

    private static class AsynchronousFileChannelWriteCompletionHandler
    extends BaseSubscriber<DataBuffer>
    implements CompletionHandler<Integer, ByteBuffer> {
        private final FluxSink<DataBuffer> sink;
        private final AsynchronousFileChannel channel;
        private final AtomicBoolean completed = new AtomicBoolean();
        private long position;
        @Nullable
        private DataBuffer dataBuffer;

        public AsynchronousFileChannelWriteCompletionHandler(FluxSink<DataBuffer> sink, AsynchronousFileChannel channel, long position) {
            this.sink = sink;
            this.channel = channel;
            this.position = position;
        }

        protected void hookOnSubscribe(Subscription subscription) {
            this.request(1L);
        }

        protected void hookOnNext(DataBuffer value) {
            this.dataBuffer = value;
            ByteBuffer byteBuffer = value.asByteBuffer();
            this.channel.write(byteBuffer, this.position, byteBuffer, this);
        }

        protected void hookOnError(Throwable throwable) {
            this.sink.error(throwable);
        }

        protected void hookOnComplete() {
            this.completed.set(true);
        }

        @Override
        public void completed(Integer written, ByteBuffer byteBuffer) {
            this.position += (long)written.intValue();
            if (byteBuffer.hasRemaining()) {
                this.channel.write(byteBuffer, this.position, byteBuffer, this);
                return;
            }
            if (this.dataBuffer != null) {
                this.sink.next((Object)this.dataBuffer);
            }
            if (this.completed.get()) {
                this.sink.complete();
            } else {
                this.request(1L);
            }
        }

        @Override
        public void failed(Throwable exc, ByteBuffer byteBuffer) {
            this.sink.error(exc);
        }
    }

    private static class AsynchronousFileChannelReadCompletionHandler
    implements CompletionHandler<Integer, AsynchronousFileChannel> {
        private final FluxSink<DataBuffer> sink;
        private final ByteBuffer byteBuffer;
        private final DataBufferFactory dataBufferFactory;
        private long position;

        private AsynchronousFileChannelReadCompletionHandler(FluxSink<DataBuffer> sink, long position, DataBufferFactory dataBufferFactory, ByteBuffer byteBuffer) {
            this.sink = sink;
            this.position = position;
            this.dataBufferFactory = dataBufferFactory;
            this.byteBuffer = byteBuffer;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void completed(Integer read, AsynchronousFileChannel channel) {
            if (read != -1) {
                this.position += (long)read.intValue();
                this.byteBuffer.flip();
                boolean release = true;
                DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(read);
                try {
                    dataBuffer.write(this.byteBuffer);
                    release = false;
                    this.sink.next((Object)dataBuffer);
                }
                finally {
                    if (release) {
                        DataBufferUtils.release(dataBuffer);
                    }
                }
                this.byteBuffer.clear();
                if (!this.sink.isCancelled()) {
                    channel.read(this.byteBuffer, this.position, channel, this);
                }
            } else {
                this.sink.complete();
                DataBufferUtils.closeChannel(channel);
            }
        }

        @Override
        public void failed(Throwable exc, AsynchronousFileChannel channel) {
            this.sink.error(exc);
            DataBufferUtils.closeChannel(channel);
        }
    }

    private static class ReadableByteChannelGenerator
    implements BiFunction<ReadableByteChannel, SynchronousSink<DataBuffer>, ReadableByteChannel> {
        private final DataBufferFactory dataBufferFactory;
        private final ByteBuffer byteBuffer;

        public ReadableByteChannelGenerator(DataBufferFactory dataBufferFactory, int chunkSize) {
            this.dataBufferFactory = dataBufferFactory;
            this.byteBuffer = ByteBuffer.allocate(chunkSize);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public ReadableByteChannel apply(ReadableByteChannel channel, SynchronousSink<DataBuffer> sub) {
            block8: {
                try {
                    int read = channel.read(this.byteBuffer);
                    if (read >= 0) {
                        this.byteBuffer.flip();
                        boolean release = true;
                        DataBuffer dataBuffer = this.dataBufferFactory.allocateBuffer(read);
                        try {
                            dataBuffer.write(this.byteBuffer);
                            release = false;
                            sub.next((Object)dataBuffer);
                        }
                        finally {
                            if (release) {
                                DataBufferUtils.release(dataBuffer);
                            }
                        }
                        this.byteBuffer.clear();
                        break block8;
                    }
                    sub.complete();
                }
                catch (IOException ex) {
                    sub.error((Throwable)ex);
                }
            }
            return channel;
        }
    }
}

